package betterwithmods.util.player;

import betterwithmods.common.BWMRecipes;
import betterwithmods.common.BWOreDictionary;
import betterwithmods.common.registry.BrokenToolRegistry;
import betterwithmods.module.hardcore.needs.HCArmor;
import com.google.common.collect.Sets;
import com.mojang.authlib.GameProfile;
import net.minecraft.block.Block;
import net.minecraft.block.material.Material;
import net.minecraft.block.state.IBlockState;
import net.minecraft.entity.EntityLivingBase;
import net.minecraft.entity.SharedMonsterAttributes;
import net.minecraft.entity.ai.attributes.AttributeModifier;
import net.minecraft.entity.ai.attributes.IAttributeInstance;
import net.minecraft.entity.passive.EntitySquid;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.init.Items;
import net.minecraft.inventory.EntityEquipmentSlot;
import net.minecraft.item.ItemArmor;
import net.minecraft.item.ItemStack;
import net.minecraft.item.crafting.Ingredient;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.util.EnumHand;
import net.minecraft.util.math.BlockPos;
import net.minecraft.world.IBlockAccess;
import net.minecraft.world.World;
import net.minecraftforge.common.ForgeHooks;

import java.util.Objects;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

/**
 * Set of methods dealing with EntityPlayer
 *
 * @author Koward
 */
public final class PlayerHelper {
    public final static UUID PENALTY_SPEED_UUID = UUID.fromString("c5595a67-9410-4fb2-826a-bcaf432c6a6f");

    private PlayerHelper() {

    }

    public static ItemStack getHolding(EntityPlayer player, EnumHand hand) {
        if (hand != null) {
            return player.func_184586_b(hand);
        }
        return player.func_184586_b(EnumHand.MAIN_HAND);
    }

    public static Set<ItemStack> getHolding(EntityPlayer player) {
        return Sets.newHashSet(player.func_184614_ca(),player.func_184592_cb()).stream().filter(s -> !s.func_190926_b()).collect(Collectors.toSet());
    }

    public static boolean isHolding(EntityPlayer player, Ingredient ingredient) {
        Set<ItemStack> held = getHolding(player);
        if (held.isEmpty())
            return false;
        for (ItemStack h : held)
            if (ingredient.apply(h))
                return true;
        return false;
    }

    public static boolean isSurvival(EntityPlayer player) {
        return player != null && !player.func_184812_l_() && !player.func_175149_v() && !player.func_175149_v();
    }

    /**
     * This will at least keep players from sticking to the bottom of a pool.
     *
     * @param player The swimming player.
     * @return Whether the player is near the bottom of the pool or not.
     */
    public static boolean isNearBottom(EntityPlayer player) {
        World world = player.func_130014_f_();
        BlockPos a = player.func_180425_c().func_177979_c(1);
        BlockPos b = player.func_180425_c().func_177979_c(2);
        return !world.func_180495_p(a).func_185904_a().func_76222_j() && !world.func_180495_p(b).func_185904_a().func_76222_j();
    }

    public static void changeAttack(EntityLivingBase entity,
                                    UUID attackModifierUUID, String name, double multiplier) {
        AttributeModifier attackModifier = (new AttributeModifier(
                attackModifierUUID, name, multiplier - 1, 2));
        IAttributeInstance iattributeinstance = entity
                .func_110148_a(SharedMonsterAttributes.field_111264_e);

        if (iattributeinstance.func_111127_a(attackModifierUUID) != null) {
            iattributeinstance.func_111124_b(attackModifier);
        }
        iattributeinstance.func_111121_a(attackModifier);
    }

    /**
     * Edit the speed of an entity.
     *
     * @param entity           The entity whose speed will be changed.
     * @param name             Unique name for easier debugging
     * @param modifier         The speed will be multiplied by this number
     * @param penaltySpeedUuid
     */
    public static void changeSpeed(EntityLivingBase entity,
                                   String name, double modifier, UUID penaltySpeedUuid) {
        //2 operator multiples the current value by 1+x, thus modifier-1 neutralizes the extra 1
        AttributeModifier speedModifier = (new AttributeModifier(penaltySpeedUuid, name, modifier-1, 2));
        IAttributeInstance iattributeinstance = entity.func_110148_a(SharedMonsterAttributes.field_111263_d);

        if (iattributeinstance.func_111127_a(penaltySpeedUuid) != null) {
            iattributeinstance.func_111124_b(speedModifier);
        }
        iattributeinstance.func_111121_a(speedModifier);
    }

    public static int getWornArmorWeight(EntityPlayer player) {
        int weight = 0;
        for (ItemStack stack : player.field_71071_by.field_70460_b) {
            if (stack != null)
                weight += HCArmor.getWeight(stack);
        }
        return weight;
    }


    /**
     * This pos-sensitive version should be used when it's available, as it uses {@link IBlockState#getActualState(IBlockAccess, BlockPos)}.
     *
     * @param player
     * @param pos
     * @param state
     * @return
     */
    public static boolean isCurrentToolEffectiveOnBlock(EntityPlayer player, BlockPos pos, IBlockState state) {
        ItemStack stack = BrokenToolRegistry.findItem(player, state);
        if (player == null || state == null)
            return false;
        return isCurrentToolEffectiveOnBlock(stack, state, Sets.newHashSet()) || ForgeHooks.isToolEffective(player.func_130014_f_(), pos, stack);
    }

    /**
     * Partial copy of {@link ForgeHooks#isToolEffective(IBlockAccess, BlockPos, ItemStack)} build 2185
     *
     * @param stack               The tool.
     * @param state               The block.
     * @param effective_materials
     * @return Whether the tool can harvest well the block.
     */
    public static boolean isCurrentToolEffectiveOnBlock(ItemStack stack, IBlockState state, Set<Material> effective_materials) {

        if (stack == null) return false;
        if (stack.func_77942_o()) {
            NBTTagCompound stats = stack.func_179543_a("Stats");
            if (stats != null) {
                return stats.func_74771_c("Broken") != 1;
            }
        }

        ItemStack block = BWMRecipes.getStackFromState(state);

        Material material = state.func_185904_a();
        if (effective_materials.contains(material))
            return true;
        for (String type : stack.func_77973_b().getToolClasses(stack)) {
            if (Objects.equals(type, "mattock"))
                return state.func_177230_c().isToolEffective("shovel", state) || state.func_177230_c().isToolEffective("axe", state);
            if (Objects.equals(type, "bwmmattock")) {
                return state.func_177230_c().isToolEffective("shovel", state) || state.func_177230_c().isToolEffective("pickaxe", state);
            }
            if (state.func_177230_c().isToolEffective(type, state) || BWOreDictionary.isToolForOre(type, block))
                return true;
        }
        return false;
    }

    public static ItemStack getPlayerHead(EntityPlayer player) {
        ItemStack head = new ItemStack(Items.field_151144_bL, 1, 3);
        NBTTagCompound name = new NBTTagCompound();
        name.func_74778_a("SkullOwner", player.getDisplayNameString());
        head.func_77982_d(name);
        return head;
    }

    public static boolean hasFullSet(EntityPlayer player, Class<? extends ItemArmor> armor) {
        for (ItemStack stack : player.func_184193_aE()) {
            if (!armor.isAssignableFrom(stack.func_77973_b().getClass()))
                return false;
        }
        return true;
    }

    public static boolean hasPart(EntityLivingBase living, EntityEquipmentSlot type, Class<? extends ItemArmor> armor) {
        return armor.isAssignableFrom(living.func_184582_a(type).func_77973_b().getClass());
    }


    public static UUID getUUID(EntityPlayer player) {
        GameProfile profile = player.func_146103_bH();
        if (profile != null)
            return profile.getId();
        return player.func_110124_au();
    }

    public static boolean isMoving(EntityPlayer player) {
        return player.field_70159_w != 0 && player.field_70179_y != 0;
    }
}
