package betterwithmods.util;

import com.google.common.collect.Sets;
import net.minecraft.block.BlockLiquid;
import net.minecraft.block.material.Material;
import net.minecraft.block.state.BlockFaceShape;
import net.minecraft.block.state.IBlockState;
import net.minecraft.entity.item.EntityItem;
import net.minecraft.entity.monster.EntityGhast;
import net.minecraft.init.Blocks;
import net.minecraft.item.ItemStack;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.math.AxisAlignedBB;
import net.minecraft.util.math.BlockPos;
import net.minecraft.util.math.Vec3d;
import net.minecraft.world.EnumSkyBlock;
import net.minecraft.world.World;
import net.minecraft.world.chunk.Chunk;
import net.minecraft.world.chunk.storage.ExtendedBlockStorage;
import net.minecraftforge.event.entity.living.LivingDropsEvent;

import java.util.Calendar;
import java.util.HashSet;
import java.util.Random;
import java.util.Set;
import java.util.stream.StreamSupport;

import static net.minecraft.world.chunk.Chunk.NULL_BLOCK_STORAGE;

/**
 * @author Koward
 */
public final class WorldUtils {

    private static final HashSet<Material> SOLID_MATERIALS = Sets.newHashSet(
            Material.field_151576_e,
            Material.field_151574_g,
            Material.field_151592_s,
            Material.field_151573_f,
            Material.field_151588_w,
            Material.field_151598_x,
            Material.field_151591_t,
            Material.field_76233_E
    );

    private WorldUtils() {
    }

    public static boolean isSolid(World world, BlockPos pos, EnumFacing facing, IBlockState state) {
        return SOLID_MATERIALS.contains(state.func_185904_a()) && state.func_193401_d(world, pos, facing.func_176734_d()) == BlockFaceShape.SOLID;
    }

    /**
     * Based on {@link World#getLightFromNeighbors(BlockPos)} build 2185
     */
    public static int getNaturalLightFromNeighbors(World worldIn, BlockPos pos) {
        return getNaturalLight(worldIn, pos, true, 0);
    }

    /**
     * Based on {@link World#getLight(BlockPos, boolean)} build 2185
     */
    private static int getNaturalLight(World worldIn, BlockPos pos, boolean checkNeighbors, int amount) {
        if (pos.func_177958_n() >= -30000000 && pos.func_177952_p() >= -30000000 && pos.func_177958_n() < 30000000 && pos.func_177952_p() < 30000000) {
            if (checkNeighbors && worldIn.func_180495_p(pos).func_185916_f()) {
                int i1 = getNaturalLight(worldIn, pos.func_177984_a(), false, 0);
                int i = getNaturalLight(worldIn, pos.func_177974_f(), false, 0);
                int j = getNaturalLight(worldIn, pos.func_177976_e(), false, 0);
                int k = getNaturalLight(worldIn, pos.func_177968_d(), false, 0);
                int l = getNaturalLight(worldIn, pos.func_177978_c(), false, 0);

                if (i > i1) {
                    i1 = i;
                }

                if (j > i1) {
                    i1 = j;
                }

                if (k > i1) {
                    i1 = k;
                }

                if (l > i1) {
                    i1 = l;
                }

                return i1;
            } else if (pos.func_177956_o() < 0) {
                return 0;
            } else {
                if (pos.func_177956_o() >= 256) {
                    pos = new BlockPos(pos.func_177958_n(), 255, pos.func_177952_p());
                }

                Chunk chunk = worldIn.func_175726_f(pos);
                return getNaturalLightSubtracted(chunk, pos, amount);
            }
        } else {
            return 15;
        }
    }

    /**
     * Based on {@link Chunk#getLightSubtracted(BlockPos, int)} build 2185
     */
    private static int getNaturalLightSubtracted(Chunk chunkIn, BlockPos pos, int amount) {
        int i = pos.func_177958_n() & 15;
        int j = pos.func_177956_o();
        int k = pos.func_177952_p() & 15;
        ExtendedBlockStorage extendedblockstorage = chunkIn.func_76587_i()[j >> 4];

        if (extendedblockstorage == field_186036_a) {
            return chunkIn.func_177412_p().field_73011_w.func_191066_m() && amount < EnumSkyBlock.SKY.field_77198_c ? EnumSkyBlock.SKY.field_77198_c - amount : 0;
        } else {
            int l = !chunkIn.func_177412_p().field_73011_w.func_191066_m() ? 0 : extendedblockstorage.func_76670_c(i, j & 15, k);
            l = l - amount;

            if (l < 0) {
                l = 0;
            }

            return l;
        }
    }

    public static double getDistance(BlockPos pos1, BlockPos pos2) {
        assert (pos1 != null);
        assert (pos2 != null);
        return new Vec3d(pos1).func_72438_d(new Vec3d(pos2));
    }

    public static boolean spawnGhast(World world, BlockPos pos) {
        EntityGhast ghast = new EntityGhast(world);
        double failures = 1;
        for (int i = 0; i < 200; i++) {
            double xPos = pos.func_177958_n() + (world.field_73012_v.nextDouble() - world.field_73012_v.nextDouble()) * Math.max(20, failures);
            double yPos = pos.func_177956_o() + failures;
            double zPos = pos.func_177952_p() + (world.field_73012_v.nextDouble() - world.field_73012_v.nextDouble()) * Math.max(20, failures);

            ghast.func_70012_b(xPos, yPos, zPos, world.field_73012_v.nextFloat() * 360.0F, 0.0F);
            AxisAlignedBB box = ghast.func_174813_aQ().func_186670_a(ghast.func_180425_c().func_177981_b(5));
            boolean blocked = StreamSupport.stream(BlockPos.MutableBlockPos.func_177980_a(getMin(box), getMax(box)).spliterator(), false).anyMatch(p -> !world.func_175623_d(p));
            if (!blocked) {
                return world.func_72838_d(ghast);
            } else {
                failures++;
            }
        }
        return false;
    }

    public static boolean isWater(World world, BlockPos pos) {
        IBlockState state = world.func_180495_p(pos);
        return state.func_177230_c() == Blocks.field_150355_j || state.func_177230_c() == Blocks.field_150358_i;
    }

    public static boolean isWaterSource(World world, BlockPos pos) {
        IBlockState state = world.func_180495_p(pos);
        return state.func_177230_c() == Blocks.field_150355_j && state.func_177229_b(BlockLiquid.field_176367_b) == 0;
    }

    public static AxisAlignedBB toInts(AxisAlignedBB box) {
        return new AxisAlignedBB((int) box.field_72340_a, (int) box.field_72338_b, (int) box.field_72339_c, (int) box.field_72336_d, (int) box.field_72337_e, (int) box.field_72334_f);
    }

    public static Set<BlockPos> getPosAround(BlockPos pos, EnumFacing.Axis axis) {
        Set<BlockPos> posSet = Sets.newHashSet();
        for (int i = -1; i <= 1; i++) {
            for (int j = -1; j <= 1; j++) {
                switch (axis) {
                    case X:
                        posSet.add(pos.func_177982_a(0, i, j));
                        break;
                    case Y:
                        posSet.add(pos.func_177982_a(i, 0, j));
                        break;
                    case Z:
                        posSet.add(pos.func_177982_a(i, j, 0));
                        break;
                }
            }
        }
        return posSet;
    }

    public static BlockPos getMin(AxisAlignedBB box) {
        return new BlockPos(box.field_72340_a, box.field_72338_b, box.field_72339_c);
    }

    public static BlockPos getMax(AxisAlignedBB box) {
        return new BlockPos(box.field_72336_d, box.field_72337_e, box.field_72334_f);
    }

    public static boolean matches(IBlockState a, IBlockState b) {
        return b == null || a.equals(b);
    }

    public static void addDrop(LivingDropsEvent evt, ItemStack drop) {
        EntityItem item = new EntityItem(evt.getEntityLiving().func_130014_f_(), evt.getEntityLiving().field_70165_t, evt.getEntityLiving().field_70163_u, evt.getEntityLiving().field_70161_v, drop);
        item.func_174869_p();
        evt.getDrops().add(item);
    }

    public static boolean isPast(World world, TimeFrame frame) {
        return frame.start < getDayTicks(world);
    }

    public static boolean isTimeFrame(World world, TimeFrame frame) {
        return frame.isBetween((int) getDayTicks(world));
    }

    public static boolean isMoonPhase(World world, MoonPhase phase) {
        return phase.ordinal() == world.field_73011_w.func_76559_b(world.func_72820_D());
    }

    public static int getDayTicks(World world) {
        return (int) (world.func_72820_D() % Time.DAY.getTicks());
    }



    public enum MoonPhase {
        Full,
        WaningGibbous,
        LastQuarter,
        WaningCrescent,
        New,
        WaxingCrescent,
        FirstQuarter,
        WaxingGibbous
    }

    public enum Time {
        SECOND(0.27),
        MINUTE(16.6),
        HOUR(1000),
        DAY(24000);
        private double ticks;

        Time(double ticks) {
            this.ticks = ticks;
        }

        public double getTicks() {
            return ticks;
        }
    }

    public enum TimeFrame {
        DAWN(0, 3600),
        NOON(5000, 7000),
        DUSK(10200, 12700),
        MIDNIGHT(17000, 19000),
        NIGHT(13001, 24000),
        DAY(0, 13000);
        private static final Random rand = new Random();
        private int start, end;

        TimeFrame(int start, int end) {
            this.start = start;
            this.end = end;
        }

        public boolean isBetween(int time) {
            return time >= start && time <= end;
        }

        public int randomBetween() {
            return rand.nextInt((end - start) + 1) + start;
        }
    }

}
