/*
 * Copyright (c) 2015, 2016, 2017 Adrian Siekierka
 *
 * This file is part of Charset.
 *
 * Charset is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Charset is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with Charset.  If not, see <http://www.gnu.org/licenses/>.
 */

package betterwithmods.util;

import net.minecraft.entity.Entity;
import net.minecraft.entity.EntityLivingBase;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.math.*;
import net.minecraft.world.chunk.Chunk;
import org.apache.commons.lang3.tuple.Pair;

import javax.annotation.Nullable;
import javax.vecmath.Vector3d;
import java.util.Collection;

/**
 * Operations on AxisAlignedBB (aka 'Box'), Vec3d, EnumFacing, Entities, and conversions between them.
 */
public final class SpaceUtils {

    public static final byte GET_POINT_MIN = 0x0;
    public static final byte GET_POINT_MAX = 0x7;
    private static final int[][] ROTATION_MATRIX = {
            {0, 1, 4, 5, 3, 2},
            {0, 1, 5, 4, 2, 3},
            {5, 4, 2, 3, 0, 1},
            {4, 5, 2, 3, 1, 0},
            {2, 3, 1, 0, 4, 5},
            {3, 2, 0, 1, 4, 5},
            {0, 1, 2, 3, 4, 5}
    };
    private static final int[][] ROTATION_MATRIX_INV = new int[6][6];

    static {
        for (int axis = 0; axis < 6; axis++)
            for (int dir = 0; dir < 6; dir++) {
                int out = dir;
                out = ROTATION_MATRIX[axis][out];
                out = ROTATION_MATRIX[axis][out];
                out = ROTATION_MATRIX[axis][out];
                ROTATION_MATRIX_INV[axis][dir] = out;
            }
    }

    public static EnumFacing determineOrientation(EntityLivingBase player) {
        if (player.field_70125_A > 75) {
            return EnumFacing.DOWN;
        } else if (player.field_70125_A <= -75) {
            return EnumFacing.UP;
        } else {
            return determineFlatOrientation(player);
        }
    }

    public static EnumFacing determineFlatOrientation(EntityLivingBase player) {
        int var7 = MathHelper.func_76128_c((double) ((180 + player.field_70177_z) * 4.0F / 360.0F) + 0.5D) & 3;
        int r = var7 == 0 ? 2 : (var7 == 1 ? 5 : (var7 == 2 ? 3 : (var7 == 3 ? 4 : 0)));
        return EnumFacing.field_82609_l[r];
    }

    public static Vec3d copy(Vec3d a) {
        return new Vec3d(a.field_72450_a, a.field_72448_b, a.field_72449_c);
    }

    public static AxisAlignedBB copy(AxisAlignedBB box) {
        return new AxisAlignedBB(box.field_72340_a, box.field_72338_b, box.field_72339_c, box.field_72336_d, box.field_72337_e, box.field_72334_f);
    }

    public static Vec3d getEntityVelocity(Entity ent) {
        return new Vec3d(ent.field_70159_w, ent.field_70181_x, ent.field_70179_y);
    }

    public static void setEntityVelocity(Entity ent, Vec3d vec) {
        ent.field_70159_w = vec.field_72450_a;
        ent.field_70181_x = vec.field_72448_b;
        ent.field_70179_y = vec.field_72449_c;
    }

    public static int ordinal(@Nullable EnumFacing side) {
        return side == null ? 6 : side.ordinal();
    }

    @Nullable
    public static EnumFacing getFacing(int ordinal) {
        return ordinal == 6 ? null : EnumFacing.func_82600_a(ordinal);
    }

    public static Vec3d fromPlayerEyePos(EntityPlayer ent) {
        // This is all iChun's fault. :/
        // Uh...
        if (ent.field_70170_p.field_72995_K) {
            return new Vec3d(ent.field_70165_t, ent.field_70163_u + (ent.func_70047_e() - ent.getDefaultEyeHeight()), ent.field_70161_v);
        } else {
            return new Vec3d(ent.field_70165_t, ent.field_70163_u + ent.func_70047_e(), ent.field_70161_v);
        }
    }

    /**
     * Sets the entity's position directly. Does *NOT* update the bounding box!
     */
    public static void setEntityPosition(Entity ent, Vec3d pos) {
        ent.field_70165_t = pos.field_72450_a;
        ent.field_70163_u = pos.field_72448_b;
        ent.field_70161_v = pos.field_72449_c;
    }

    /**
     * Sets the entity's position using its setter. Will (presumably) update the bounding box.
     */
    public static void setEntPos(Entity ent, Vec3d pos) {
        ent.func_70107_b(pos.field_72450_a, pos.field_72448_b, pos.field_72449_c);
    }

    public static AxisAlignedBB setMin(AxisAlignedBB aabb, Vec3d v) {
        return new AxisAlignedBB(
                v.field_72450_a, v.field_72448_b, v.field_72449_c,
                aabb.field_72336_d, aabb.field_72337_e, aabb.field_72334_f);
    }

    public static Vec3d getMax(AxisAlignedBB aabb) {
        return new Vec3d(aabb.field_72336_d, aabb.field_72337_e, aabb.field_72334_f);
    }

    public static Vec3d getMin(AxisAlignedBB aabb) {
        return new Vec3d(aabb.field_72340_a, aabb.field_72338_b, aabb.field_72339_c);
    }

    public static AxisAlignedBB setMax(AxisAlignedBB aabb, Vec3d v) {
        return new AxisAlignedBB(
                aabb.field_72340_a, aabb.field_72338_b, aabb.field_72339_c,
                v.field_72450_a, v.field_72448_b, v.field_72449_c);
    }

    public static Vec3d getMiddle(AxisAlignedBB ab) {
        return new Vec3d(
                (ab.field_72340_a + ab.field_72336_d) / 2,
                (ab.field_72338_b + ab.field_72337_e) / 2,
                (ab.field_72339_c + ab.field_72334_f) / 2);
    }

    public static Vec3d fromDirection(EnumFacing dir) {
        return new Vec3d(dir.func_176730_m());
    }

    public static Pair<Vec3d, Vec3d> sort(Vec3d left, Vec3d right) {
        double minX = Math.min(left.field_72450_a, right.field_72450_a);
        double maxX = Math.max(left.field_72450_a, right.field_72450_a);
        double minY = Math.min(left.field_72448_b, right.field_72448_b);
        double maxY = Math.max(left.field_72448_b, right.field_72448_b);
        double minZ = Math.min(left.field_72449_c, right.field_72449_c);
        double maxZ = Math.max(left.field_72449_c, right.field_72449_c);
        return Pair.of(new Vec3d(minX, minY, minZ), new Vec3d(maxX, maxY, maxZ));
    }

    /**
     * Copies a point on box into target.
     * pointFlags is a bit-flag, like <Z, Y, X>.
     * So if the value is 0b000, then target is the minimum point,
     * and 0b111 the target is the maximum.
     */
    public static Vec3d getVertex(AxisAlignedBB box, byte pointFlags) {
        boolean xSide = (pointFlags & 1) == 1;
        boolean ySide = (pointFlags & 2) == 2;
        boolean zSide = (pointFlags & 4) == 4;
        return new Vec3d(
                xSide ? box.field_72340_a : box.field_72336_d,
                ySide ? box.field_72338_b : box.field_72337_e,
                zSide ? box.field_72339_c : box.field_72334_f
        );
    }

    /**
     * @param box  The box to be flattened
     * @param face The side of the box that will remain untouched; the opposite face will be brought to it
     * @return A new box, with a volume of 0. Returns null if face is invalid.
     */
    public static AxisAlignedBB flatten(AxisAlignedBB box, EnumFacing face) {
        byte[] lows = new byte[]{0x2, 0x0, 0x4, 0x0, 0x1, 0x0};
        byte[] hghs = new byte[]{0x7, 0x5, 0x7, 0x3, 0x7, 0x6};
        byte low = lows[face.ordinal()];
        byte high = hghs[face.ordinal()];
        assert low != high;
        assert (~low & 0x7) != high;
        return new AxisAlignedBB(getVertex(box, low), getVertex(box, high));
    }

    public static double getDiagonalLength(AxisAlignedBB ab) {
        double x = ab.field_72336_d - ab.field_72340_a;
        double y = ab.field_72337_e - ab.field_72338_b;
        double z = ab.field_72334_f - ab.field_72339_c;
        return Math.sqrt(x * x + y * y + z * z);
    }

    public static Vec3d average(Vec3d a, Vec3d b) {
        return new Vec3d((a.field_72450_a + b.field_72450_a) / 2, (a.field_72448_b + b.field_72448_b) / 2, (a.field_72449_c + b.field_72449_c) / 2);
    }

    public static double getAngle(Vec3d a, Vec3d b) {
        double dot = a.func_72430_b(b);
        double mags = a.func_72433_c() * b.func_72433_c();
        double div = dot / mags;
        if (div > 1) div = 1;
        if (div < -1) div = -1;
        return Math.acos(div);
    }

    public static AxisAlignedBB withPoints(Vec3d[] parts) {
        return new AxisAlignedBB(getLowest(parts), getHighest(parts));
    }

    public static Vec3d scale(Vec3d base, double s) {
        return new Vec3d(base.field_72450_a * s, base.field_72448_b * s, base.field_72449_c * s);
    }

    public static Vec3d componentMultiply(Vec3d a, Vec3d b) {
        return new Vec3d(a.field_72450_a + b.field_72450_a, a.field_72448_b + b.field_72448_b, a.field_72449_c + b.field_72449_c);
    }

    public static Vec3d componentMultiply(Vec3d a, double x, double y, double z) {
        return new Vec3d(a.field_72450_a + x, a.field_72448_b + y, a.field_72449_c + z);
    }

    public static AxisAlignedBB sortedBox(Vec3d min, Vec3d max) {
        double minX = Math.min(min.field_72450_a, max.field_72450_a);
        double minY = Math.min(min.field_72448_b, max.field_72448_b);
        double minZ = Math.min(min.field_72449_c, max.field_72449_c);
        double maxX = Math.max(min.field_72450_a, max.field_72450_a);
        double maxY = Math.max(min.field_72448_b, max.field_72448_b);
        double maxZ = Math.max(min.field_72449_c, max.field_72449_c);
        return new AxisAlignedBB(minX, minY, minZ, maxX, maxY, maxZ);
    }

    public static AxisAlignedBB withPoint(AxisAlignedBB box, Vec3d vec) {
        return new AxisAlignedBB(
                vec.field_72450_a < box.field_72340_a ? vec.field_72450_a : box.field_72340_a,
                vec.field_72448_b < box.field_72338_b ? vec.field_72448_b : box.field_72338_b,
                vec.field_72449_c < box.field_72339_c ? vec.field_72449_c : box.field_72339_c,
                box.field_72336_d < vec.field_72450_a ? vec.field_72450_a : box.field_72336_d,
                box.field_72337_e < vec.field_72448_b ? vec.field_72448_b : box.field_72337_e,
                box.field_72334_f < vec.field_72449_c ? vec.field_72449_c : box.field_72334_f
        );
    }

    public static Vec3d[] getCorners(AxisAlignedBB box) {
        return new Vec3d[]{
                new Vec3d(box.field_72340_a, box.field_72338_b, box.field_72339_c),
                new Vec3d(box.field_72340_a, box.field_72337_e, box.field_72339_c),
                new Vec3d(box.field_72336_d, box.field_72337_e, box.field_72339_c),
                new Vec3d(box.field_72336_d, box.field_72338_b, box.field_72339_c),

                new Vec3d(box.field_72340_a, box.field_72338_b, box.field_72334_f),
                new Vec3d(box.field_72340_a, box.field_72337_e, box.field_72334_f),
                new Vec3d(box.field_72336_d, box.field_72337_e, box.field_72334_f),
                new Vec3d(box.field_72336_d, box.field_72338_b, box.field_72334_f)
        };
    }

    public static Vec3d getLowest(Vec3d[] vs) {
        double x, y, z;
        x = y = z = 0;
        boolean first = true;
        for (int i = 0; i < vs.length; i++) {
            Vec3d v = vs[i];
            if (v == null) continue;
            if (first) {
                first = false;
                x = v.field_72450_a;
                y = v.field_72448_b;
                z = v.field_72449_c;
                continue;
            }
            if (v.field_72450_a < x) x = v.field_72450_a;
            if (v.field_72448_b < y) y = v.field_72448_b;
            if (v.field_72449_c < z) z = v.field_72449_c;
        }
        return new Vec3d(x, y, z);
    }

    public static Vec3d getHighest(Vec3d[] vs) {
        double x, y, z;
        x = y = z = 0;
        boolean first = true;
        for (int i = 0; i < vs.length; i++) {
            Vec3d v = vs[i];
            if (v == null) continue;
            if (first) {
                first = false;
                x = v.field_72450_a;
                y = v.field_72448_b;
                z = v.field_72449_c;
                continue;
            }
            if (v.field_72450_a > x) x = v.field_72450_a;
            if (v.field_72448_b > y) y = v.field_72448_b;
            if (v.field_72449_c > z) z = v.field_72449_c;
        }
        return new Vec3d(x, y, z);
    }

    public static boolean isZero(Vec3d vec) {
        return vec.field_72450_a == 0 && vec.field_72448_b == 0 && vec.field_72449_c == 0;
    }

    /**
     * Return the distance between point and the line defined as passing through the origin and lineVec
     *
     * @param lineVec The vector defining the line, relative to the origin.
     * @param point   The point being measured, relative to the origin
     * @return the distance between line defined by lineVec and point
     */
    public static double lineDistance(Vec3d lineVec, Vec3d point) {
        // http://mathworld.wolfram.com/Point-LineDistance3-Dimensional.html equation 9
        double mag = lineVec.func_72433_c();
        Vec3d nPoint = scale(point, -1);
        return lineVec.func_72431_c(nPoint).func_72433_c() / mag;
    }

    public static double lineDistance(Vec3d origin, Vec3d lineVec, Vec3d point) {
        return lineDistance(lineVec.func_178788_d(origin), point.func_178788_d(origin));
    }

    public static EnumFacing getOrientation(int ordinal) {
        if (ordinal < 0) return null;
        if (ordinal >= 6) return null;
        return EnumFacing.field_82609_l[ordinal];
    }

//    public static Orientation getOrientation(World world, BlockPos pos, EntityLivingBase placer, EnumFacing face, float hitX, float hitY, float hitZ) {
//        Vec3d hitVec = null;
//        if (face == null) {
//            RayTraceResult hit = RayTraceUtils.getCollision(world, pos, placer, Block.FULL_BLOCK_AABB, 0);
//            if (hit != null) {
//                face = hit.sideHit;
//                hitVec = hit.hitVec != null ? hit.hitVec.subtract(new Vec3d(pos)) : null;
//            }
//        } else {
//            hitVec = new Vec3d(hitX, hitY, hitZ);
//        }
//
//        if (hitVec != null) {
//            return SpaceUtils.getOrientation(placer, face.getOpposite(), hitVec);
//        } else if (face != null) {
//            return Orientation.fromDirection(face);
//        } else {
//            return Orientation.FACE_UP_POINT_NORTH;
//        }
//    }
//
//    public static Orientation getOrientation(EntityLivingBase player, EnumFacing facing, Vec3d hit) {
//        if (facing == null) {
//            facing = EnumFacing.DOWN;
//        }
//        if (facing != null)
//            return Orientation.fromDirection(facing);
////        return Orientation.FACE_NORTH_POINT_DOWN;
//        double u, v;
//        if (facing == null) {
//            facing = EnumFacing.DOWN;
//        }
//        switch (facing) {
//            default:
//            case DOWN:
//                u = 1 - hit.x;
//                v = hit.z;
//                break;
//            case UP:
//                u = hit.x;
//                v = hit.z;
//                break;
//            case NORTH:
//                u = hit.x;
//                v = hit.y;
//                break;
//            case SOUTH:
//                u = 1 - hit.x;
//                v = hit.y;
//                break;
//            case WEST:
//                u = 1 - hit.z;
//                v = hit.y;
//                break;
//            case EAST:
//                u = hit.z;
//                v = hit.y;
//                break;
//        }
//        u -= 0.5;
//        v -= 0.5;
//        double angle = Math.toDegrees(Math.atan2(v, u)) + 180;
//        angle = (angle + 45) % 360;
//        int pointy = (int) (angle / 90);
//        pointy = (pointy + 1) % 4;
//
//        Orientation fo = Orientation.fromDirection(facing);
//        for (int X = 0; X < pointy; X++) {
//            fo = fo.getNextRotationOnFace();
//        }
//        EnumFacing orient = SpaceUtils.determineOrientation(player);
//        if (orient.getAxis() != EnumFacing.Axis.Y
//                && facing.getAxis() == EnumFacing.Axis.Y) {
//            facing = orient;
//            fo = Orientation.fromDirection(orient.getOpposite());
//            if (fo != null) {
//                Orientation perfect = fo.pointTopTo(EnumFacing.UP);
//                if (perfect != null) {
//                    fo = perfect;
//                }
//            }
//        }
//        double dist = Math.max(Math.abs(u), Math.abs(v));
//        if (dist < 0.33) {
//            Orientation perfect = fo.pointTopTo(EnumFacing.UP);
//            if (perfect != null) {
//                fo = perfect;
//            }
//        }
//        return fo;
//    }

    public static int sign(EnumFacing dir) {
        return dir != null ? dir.func_176743_c().func_179524_a() : 0;
    }

    public static double componentSum(Vec3d vec) {
        return vec.field_72450_a + vec.field_72448_b + vec.field_72449_c;
    }

    public static EnumFacing getClosestDirection(Vec3d vec) {
        return getClosestDirection(vec, null);
    }

    public static EnumFacing getClosestDirection(Vec3d vec, EnumFacing not) {
        if (isZero(vec)) return null;
        Vec3i work;
        double bestAngle = Double.POSITIVE_INFINITY;
        EnumFacing closest = null;
        for (EnumFacing dir : EnumFacing.field_82609_l) {
            if (dir == not) continue;
            work = dir.func_176730_m();
            double dot = getAngle(vec, new Vec3d(work));
            if (dot < bestAngle) {
                bestAngle = dot;
                closest = dir;
            }
        }
        return closest;
    }

    public static Vec3d floor(Vec3d vec) {
        return new Vec3d(
                Math.floor(vec.field_72450_a),
                Math.floor(vec.field_72448_b),
                Math.floor(vec.field_72449_c));
    }

    public static Vec3d normalize(Vec3d v) {
        // Vanilla's threshold is too low for my purposes.
        double length = v.func_72433_c();
        if (length == 0) return Vec3d.field_186680_a;
        double inv = 1.0 / length;
        if (Double.isNaN(inv) || Double.isInfinite(inv)) return Vec3d.field_186680_a;
        return scale(v, inv);
    }

    public static AxisAlignedBB include(AxisAlignedBB box, BlockPos at) {
        double minX = box.field_72340_a;
        double maxX = box.field_72336_d;
        double minY = box.field_72338_b;
        double maxY = box.field_72337_e;
        double minZ = box.field_72339_c;
        double maxZ = box.field_72334_f;

        if (at.func_177958_n() < minX) minX = at.func_177958_n();
        if (at.func_177958_n() + 1 > maxX) maxX = at.func_177958_n() + 1;
        if (at.func_177956_o() < minY) minY = at.func_177956_o();
        if (at.func_177956_o() + 1 > maxY) maxY = at.func_177956_o() + 1;
        if (at.func_177952_p() < minZ) minZ = at.func_177952_p();
        if (at.func_177952_p() + 1 > maxZ) maxZ = at.func_177952_p() + 1;

        return new AxisAlignedBB(
                minX, minY, minZ,
                maxX, maxY, maxZ);
    }

    public static AxisAlignedBB include(AxisAlignedBB box, Vec3d at) {
        double minX = box.field_72340_a;
        double maxX = box.field_72336_d;
        double minY = box.field_72338_b;
        double maxY = box.field_72337_e;
        double minZ = box.field_72339_c;
        double maxZ = box.field_72334_f;

        if (at.field_72450_a < minX) minX = at.field_72450_a;
        if (at.field_72450_a > maxX) maxX = at.field_72450_a;
        if (at.field_72448_b < minY) minY = at.field_72448_b;
        if (at.field_72448_b > maxY) maxY = at.field_72448_b;
        if (at.field_72449_c < minZ) minZ = at.field_72449_c;
        if (at.field_72449_c > maxZ) maxZ = at.field_72449_c;

        return new AxisAlignedBB(
                minX, minY, minZ,
                maxX, maxY, maxZ);
    }

    public static double getVolume(AxisAlignedBB box) {
        if (box == null) return 0;
        double x = box.field_72336_d - box.field_72340_a;
        double y = box.field_72337_e - box.field_72338_b;
        double z = box.field_72334_f - box.field_72339_c;
        double volume = x * y * z;

        if (volume < 0) return 0;
        return volume;
    }

    public static AxisAlignedBB createBox(BlockPos at, int radius) {
        return new AxisAlignedBB(at.func_177982_a(-radius, -radius, -radius), at.func_177982_a(+radius + 1, +radius + 1, +radius + 1));
    }

    /**
     * Rotate the allowed direction that is nearest to the rotated dir.
     *
     * @param dir   The original direction
     * @param rot   The rotation to apply
     * @param allow The directions that may be used.
     * @return A novel direction
     */
    public static EnumFacing rotateDirection(EnumFacing dir, Quaternion rot, Iterable<EnumFacing> allow) {
        Vec3d v = fromDirection(dir);
        rot.applyRotation(v);
        EnumFacing best = null;
        double bestDot = Double.POSITIVE_INFINITY;
        for (EnumFacing fd : allow) {
            Vec3d f = fromDirection(fd);
            rot.applyRotation(f);
            double dot = v.func_72430_b(f);
            if (dot < bestDot) {
                bestDot = dot;
                best = fd;
            }
        }
        return best;
    }

    public static EnumFacing rotateDirectionAndExclude(EnumFacing dir, Quaternion rot, Collection<EnumFacing> allow) {
        EnumFacing ret = rotateDirection(dir, rot, allow);
        allow.remove(ret);
        allow.remove(ret.func_176734_d());
        return ret;
    }

    public static EnumFacing rotateCounterclockwise(EnumFacing dir, EnumFacing axis) {
        return EnumFacing.field_82609_l[ROTATION_MATRIX[axis.ordinal()][dir.ordinal()]];
    }

    public static EnumFacing rotateClockwise(EnumFacing dir, EnumFacing axis) {
        return EnumFacing.field_82609_l[ROTATION_MATRIX_INV[axis.ordinal()][dir.ordinal()]];
    }

    public static Iterable<BlockPos.MutableBlockPos> iterateAround(BlockPos src, int radius) {
        return BlockPos.func_177975_b(src.func_177982_a(-radius, -radius, -radius), src.func_177982_a(+radius, +radius, +radius));
    }

    public static Vector3d toJavaVector(Vec3d val) {
        return new Vector3d(val.field_72450_a, val.field_72448_b, val.field_72449_c);
    }

    public static AxisAlignedBB getChunkBoundingBox(Chunk chunk) {
        int minX = chunk.field_76635_g << 4;
        int minZ = chunk.field_76647_h << 4;
        return new AxisAlignedBB(minX, 0, minZ, minX + 16, 0xFF, minZ + 16);
    }
}
