package betterwithmods.util;

import net.minecraft.entity.Entity;
import net.minecraft.entity.item.EntityItem;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.inventory.ItemStackHelper;
import net.minecraft.item.Item;
import net.minecraft.item.ItemStack;
import net.minecraft.item.crafting.Ingredient;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.NonNullList;
import net.minecraft.util.ResourceLocation;
import net.minecraft.util.math.AxisAlignedBB;
import net.minecraft.util.math.BlockPos;
import net.minecraft.util.math.MathHelper;
import net.minecraft.world.World;
import net.minecraft.world.WorldServer;
import net.minecraft.world.storage.loot.LootContext;
import net.minecraftforge.common.ForgeHooks;
import net.minecraftforge.fluids.capability.CapabilityFluidHandler;
import net.minecraftforge.items.CapabilityItemHandler;
import net.minecraftforge.items.IItemHandler;
import net.minecraftforge.items.IItemHandlerModifiable;
import net.minecraftforge.items.ItemStackHandler;
import net.minecraftforge.oredict.OreDictionary;
import org.apache.commons.lang3.ArrayUtils;

import javax.annotation.Nonnull;
import java.util.*;
import java.util.function.BiPredicate;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

public class InvUtils {

    public static boolean containsIngredient(List<Ingredient> collection, List<Ingredient> ingredient) {
        return matchesPredicate(collection, ingredient, (a,b) -> {
            if(a.func_193365_a().length > 0)
                return Arrays.stream(a.func_193365_a()).allMatch(b::apply);
            return false;
        });
    }

    public static boolean isIngredientValid(Ingredient ingredient) {
        return ArrayUtils.isEmpty(ingredient.func_193365_a());
    }

    public static boolean containsIngredient(ItemStackHandler handler, Ingredient ingredient) {
        return InventoryIterator.stream(handler).anyMatch(ingredient::apply);
    }

    public static boolean isFluidContainer(ItemStack stack) {
        return !stack.func_190926_b() && (stack.hasCapability(CapabilityFluidHandler.FLUID_HANDLER_CAPABILITY, null) || stack.hasCapability(CapabilityFluidHandler.FLUID_HANDLER_ITEM_CAPABILITY, null));
    }

    public static <T> NonNullList<T> asNonnullList(T... array) {
        NonNullList<T> nonNullList = NonNullList.func_191196_a();
        if (array != null)
            nonNullList.addAll(Arrays.stream(array).filter(Objects::nonNull).collect(Collectors.toList()));
        return nonNullList;
    }

    public static <T> NonNullList<T> asNonnullList(List<T> list) {
        NonNullList<T> nonNullList = NonNullList.func_191196_a();
        if (list != null)
            nonNullList.addAll(list.stream().filter(Objects::nonNull).collect(Collectors.toList()));
        return nonNullList;
    }


    public static void givePlayer(EntityPlayer player, EnumFacing inv, NonNullList<ItemStack> stacks) {
        IItemHandlerModifiable inventory = (IItemHandlerModifiable) player.getCapability(CapabilityItemHandler.ITEM_HANDLER_CAPABILITY, inv);
        if (inventory != null)
            insert(inventory, stacks, false);
    }

    public static boolean usePlayerItemStrict(EntityPlayer player, EnumFacing inv, Ingredient ingredient, int amount) {
        IItemHandlerModifiable inventory = (IItemHandlerModifiable) player.getCapability(CapabilityItemHandler.ITEM_HANDLER_CAPABILITY, inv);
        return inventory != null && consumeItemsInInventoryStrict(inventory, ingredient, amount, false);
    }

    public static boolean usePlayerItem(EntityPlayer player, EnumFacing inv, ItemStack stack, int amount) {
        IItemHandlerModifiable inventory = (IItemHandlerModifiable) player.getCapability(CapabilityItemHandler.ITEM_HANDLER_CAPABILITY, inv);
        return inventory != null && consumeItemsInInventory(inventory, stack, amount, false);
    }

    public static boolean usePlayerItem(EntityPlayer player, EnumFacing inv, Ingredient ingredient) {
        IItemHandlerModifiable inventory = (IItemHandlerModifiable) player.getCapability(CapabilityItemHandler.ITEM_HANDLER_CAPABILITY, inv);
        boolean result = false;

        if (inventory != null) {
            NonNullList<ItemStack> containers = NonNullList.func_191196_a();
            result = consumeItemsInInventory(inventory, ingredient, false, containers);
            givePlayer(player, inv, containers);
        }
        return result;
    }

    public static Optional<IItemHandler> getItemHandler(World world, BlockPos pos, EnumFacing facing) {
        if (!world.field_72995_K) {
            TileEntity tile = world.func_175625_s(pos);
            if (tile != null) {
                return Optional.ofNullable(tile.getCapability(CapabilityItemHandler.ITEM_HANDLER_CAPABILITY, facing));
            } else {
                List<Entity> entities = world.func_175647_a(Entity.class, new AxisAlignedBB(pos.func_177958_n(), pos.func_177956_o(), pos.func_177952_p(), pos.func_177958_n() + 1, pos.func_177956_o() + 1, pos.func_177952_p() + 1), entity -> entity.hasCapability(CapabilityItemHandler.ITEM_HANDLER_CAPABILITY, facing));
                Optional<Entity> entity = entities.stream().findFirst();
                if (entity.isPresent()) {
                    return Optional.ofNullable(entity.get().getCapability(CapabilityItemHandler.ITEM_HANDLER_CAPABILITY, facing));
                }
            }
        }
        return Optional.ofNullable(null);
    }

    public static void ejectInventoryContents(World world, BlockPos pos, IItemHandler inv) {
        for (int i = 0; i < inv.getSlots(); ++i) {
            ejectStackWithOffset(world, pos, inv.getStackInSlot(i));
        }
    }

    public static void clearInventory(IItemHandlerModifiable inv) {
        for (int i = 0; i < inv.getSlots(); ++i) {
            ItemStack stack = inv.getStackInSlot(i);
            if (!stack.func_190926_b()) {
                inv.setStackInSlot(i, ItemStack.field_190927_a);
            }
        }
    }

    public static void copyTags(ItemStack destStack, ItemStack sourceStack) {
        if (sourceStack.func_77942_o()) {
            destStack.func_77982_d(sourceStack.func_77978_p().func_74737_b());
        }

    }

    public static ItemStack decrStackSize(IItemHandlerModifiable inv, int slot, int amount) {
        if (!inv.getStackInSlot(slot).func_190926_b()) {
            ItemStack splitStack;
            if (inv.getStackInSlot(slot).func_190916_E() <= amount) {
                splitStack = inv.getStackInSlot(slot);
                inv.setStackInSlot(slot, ItemStack.field_190927_a);
                return splitStack;
            } else {
                splitStack = inv.getStackInSlot(slot).func_77979_a(amount);
                if (inv.getStackInSlot(slot).func_190916_E() < 1) {
                    inv.setStackInSlot(slot, ItemStack.field_190927_a);
                }
                return splitStack;
            }
        } else {
            return null;
        }
    }

    public static ItemStack insertSingle(IItemHandler inv, ItemStack stack, boolean simulate) {
        return insert(inv, stack, 1, simulate);
    }

    public static ItemStack insert(IItemHandler inv, ItemStack stack, int count, boolean simulate) {
        ItemStack copy = stack.func_77946_l();
        if (copy.func_190916_E() > count) {
            copy.func_190920_e(count);
        }
        return insert(inv, copy, simulate);
    }

    public static boolean canInsert(IItemHandler inv, ItemStack stack, int count) {
        ItemStack copy = stack.func_77946_l();
        if (copy.func_190916_E() > count) {
            copy.func_190920_e(count);
        }
        ItemStack inserted = insert(inv, copy, true);
        if (inserted.equals(copy)) {
            return false;
        }
        return true;
    }


    public static void insert(IItemHandler inv, NonNullList<ItemStack> stacks, boolean simulate) {
        stacks.forEach(stack -> insert(inv, stack, 0, inv.getSlots(), simulate));
    }

    public static ItemStack insert(IItemHandler inv, ItemStack stack, boolean simulate) {
        return insert(inv, stack, 0, inv.getSlots(), simulate);
    }

    public static ItemStack insert(IItemHandler inv, ItemStack stack, int minSlot, int maxSlot, boolean simulate) {
        return attemptInsert(inv, stack, minSlot, maxSlot, simulate);
    }

    public static ItemStack attemptInsert(IItemHandler inv, ItemStack stack, int minSlot, int maxSlot, boolean simulate) {
        if (isFull(inv))
            return stack;
        for (int slot = minSlot; slot < maxSlot; slot++) {
            stack = inv.insertItem(slot, stack, simulate);
            if (stack.func_190926_b())
                break;
        }
        return stack;
    }

    public static boolean insertFromWorld(IItemHandler inv, EntityItem entity, int minSlot, int maxSlot, boolean simulate) {
        ItemStack stack = entity.func_92059_d().func_77946_l();
        ItemStack leftover = attemptInsert(inv, stack, minSlot, maxSlot, simulate);
        if (leftover.func_190926_b()) {
            entity.func_70106_y();
            return true;
        } else {
            entity.func_92058_a(leftover);
            return false;
        }
    }

    public static boolean isFull(IItemHandler inv) {
        boolean full = true;
        for (int slot = 0; slot < inv.getSlots(); slot++) {
            ItemStack stack = inv.getStackInSlot(slot);
            if (stack.func_190916_E() != inv.getSlotLimit(slot)) {
                full = false;
                break;
            }
        }
        return full;
    }


    public static int getFirstOccupiedStackInRange(IItemHandler inv, int minSlot, int maxSlot) {
        for (int slot = minSlot; slot <= maxSlot; ++slot) {
            if (!inv.getStackInSlot(slot).func_190926_b()) {
                return slot;
            }
        }
        return -1;
    }

    public static int getFirstEmptyStackInRange(IItemHandler inv, int minSlot, int maxSlot) {
        for (int slot = minSlot; slot <= maxSlot; ++slot) {
            if (inv.getStackInSlot(slot).func_190926_b()) {
                return slot;
            }
        }

        return -1;
    }

    public static int getOccupiedStacks(IItemHandler inv) {
        return getOccupiedStacks(inv, 0, inv.getSlots() - 1);
    }

    public static int getOccupiedStacks(IItemHandler inv, int min, int max) {
        int count = 0;

        for (int i = min; i <= max; ++i) {
            if (inv.getStackInSlot(i).func_190926_b()) {
                ++count;
            }
        }

        return count;
    }

    public static int countItemStacksInInventory(IItemHandler inv, ItemStack toCheck) {
        int itemCount = 0;
        for (int i = 0; i < inv.getSlots(); i++) {
            ItemStack stack = inv.getStackInSlot(i);
            if (!stack.func_190926_b()) {
                if (ItemStack.func_179545_c(toCheck, stack) ||
                        (toCheck.func_77973_b() == stack.func_77973_b() && toCheck.func_77952_i() == OreDictionary.WILDCARD_VALUE)) {
                    if (toCheck.func_77942_o()) {
                        if (ItemStack.func_77970_a(toCheck, stack))
                            itemCount += stack.func_190916_E();
                    } else
                        itemCount += stack.func_190916_E();
                }
            }
        }
        return itemCount;
    }

    public static int countItemsInInventory(IItemHandler inv, Item item) {
        return countItemsInInventory(inv, item, OreDictionary.WILDCARD_VALUE);
    }

    public static int countItemsInInventory(IItemHandler inv, Item item, int meta) {
        int itemCount = 0;
        for (int i = 0; i < inv.getSlots(); i++) {
            ItemStack stack = inv.getStackInSlot(i);
            if (!stack.func_190926_b()) {
                if (stack.func_77973_b() == item) {
                    if ((meta == OreDictionary.WILDCARD_VALUE) || (stack.func_77952_i() == meta)) {
                        itemCount += inv.getStackInSlot(i).func_190916_E();
                    }
                }
            }
        }
        return itemCount;
    }

    public static int countOresInInventory(IItemHandler inv, List<ItemStack> list) {
        int ret = 0;
        if (list != null && !list.isEmpty()) {
            for (ItemStack oreStack : list) {
                ret += countItemStacksInInventory(inv, oreStack);
            }
        }
        return ret;
    }

    public static boolean consumeItemsInInventoryStrict(IItemHandlerModifiable inv, Ingredient ingredient, int sizeOfStack, boolean simulate) {
        for (int i = 0; i < inv.getSlots(); i++) {
            ItemStack stack = inv.getStackInSlot(i);
            if (!stack.func_190926_b()) {
                if (ingredient.apply(stack)) {
                    if (stack.func_190916_E() >= sizeOfStack) {
                        decrStackSize(inv, i, sizeOfStack);
                        return true;
                    }
                }
            }
        }
        return false;
    }

    public static boolean consumeItemsInInventory(IItemHandler inv, ItemStack toCheck, int sizeOfStack, boolean simulate) {
        boolean extracted = false;
        for (int i = 0; i < inv.getSlots(); i++) {
            ItemStack inSlot = inv.getStackInSlot(i);
            if (InvUtils.matches(toCheck, inSlot)) {
                ItemStack container = ForgeHooks.getContainerItem(inSlot);
                if (inv.extractItem(i, sizeOfStack, simulate).func_190916_E() >= sizeOfStack) {
                    extracted = true;
                    InvUtils.insert(inv, container, false);
                    break;
                }
            }
        }
        return extracted;
    }

    public static boolean consumeItemsInInventory(IItemHandler inv, Ingredient ingredient, boolean simulate, NonNullList<ItemStack> containers) {
        int count;
        for (int i = 0; i < inv.getSlots(); i++) {
            ItemStack inSlot = inv.getStackInSlot(i);
            if (ingredient.apply(inSlot)) {
                ItemStack container = ForgeHooks.getContainerItem(inSlot);
                if (!container.func_190926_b())
                    containers.add(container);
                if (ingredient instanceof StackIngredient)
                    count = ((StackIngredient) ingredient).getCount(inSlot);
                else
                    count = 1;
                return inv.extractItem(i, count, simulate).func_190916_E() >= count;
            }
        }
        return false;
    }

    public static boolean consumeItemsInInventory(IItemHandler inv, StackIngredient ingredient, boolean simulate, NonNullList<ItemStack> containers) {
        for (int i = 0; i < inv.getSlots(); i++) {
            ItemStack inSlot = inv.getStackInSlot(i);
            if (ingredient.apply(inSlot)) {
                int sizeOfStack = ingredient.getCount(inSlot);
                ItemStack container = ForgeHooks.getContainerItem(inSlot);
                if (!container.func_190926_b())
                    containers.add(container);
                return inv.extractItem(i, sizeOfStack, simulate).func_190916_E() >= sizeOfStack;
            }
        }
        return false;
    }

    public static boolean consumeItemsInInventory(IItemHandlerModifiable inv, Item item, int meta, int stackSize) {
        for (int i = 0; i < inv.getSlots(); ++i) {
            ItemStack stack = inv.getStackInSlot(i);
            if (!stack.func_190926_b() && stack.func_77973_b() == item && (meta == OreDictionary.WILDCARD_VALUE || stack.func_77952_i() == meta)) {
                if (stack.func_190916_E() >= stackSize) {
                    decrStackSize(inv, i, stackSize);
                    return false;
                }

                stackSize -= stack.func_190916_E();
                inv.setStackInSlot(i, ItemStack.field_190927_a);
            }
        }
        return false;
    }

    public static boolean consumeOresInInventory(IItemHandlerModifiable inv, List<?> list, int stackSize) {
        if (list.size() > 0) {
            for (Object aList : list) {
                ItemStack tempStack = (ItemStack) aList;
                Item item = tempStack.func_77973_b();
                int meta = tempStack.func_77952_i();

                for (int j = 0; j < inv.getSlots(); ++j) {
                    ItemStack stack = inv.getStackInSlot(j);
                    if (!stack.func_190926_b() && stack.func_77973_b() == item && (stack.func_77952_i() == meta || meta == OreDictionary.WILDCARD_VALUE)) {
                        if (tempStack.func_77942_o()) {
                            if (ItemStack.func_77970_a(tempStack, stack)) {
                                if (stack.func_190916_E() >= stackSize) {
                                    decrStackSize(inv, j, stackSize);
                                    return false;
                                }

                                stackSize -= stack.func_190916_E();
                                inv.setStackInSlot(j, ItemStack.field_190927_a);
                            }
                        } else {
                            if (stack.func_190916_E() >= stackSize) {
                                decrStackSize(inv, j, stackSize);
                                return false;
                            }

                            stackSize -= stack.func_190916_E();
                            inv.setStackInSlot(j, ItemStack.field_190927_a);
                        }
                    }
                }
            }
        }

        return false;
    }

    public static int getFirstOccupiedStackNotOfItem(IItemHandler inv, Item item) {
        return getFirstOccupiedStackNotOfItem(inv, item, OreDictionary.WILDCARD_VALUE);
    }

    public static int getFirstOccupiedStackNotOfItem(IItemHandler inv, Item item, int meta) {
        for (int i = 0; i < inv.getSlots(); ++i) {
            if (!inv.getStackInSlot(i).func_190926_b()) {
                int tempMeta = inv.getStackInSlot(i).func_77952_i();
                if (inv.getStackInSlot(i).func_77973_b() != item && (meta == OreDictionary.WILDCARD_VALUE || tempMeta != meta)) {
                    return i;
                }
            }
        }

        return -1;
    }

    public static int getFirstOccupiedStackOfItem(IItemHandler inv, Item item) {
        return getFirstOccupiedStackOfItem(inv, item);
    }

    public static int getFirstOccupiedStackOfItem(IItemHandler inv, ItemStack stack) {
        return getFirstOccupiedStackOfItem(inv, StackIngredient.fromStacks(stack));
    }

    public static int getFirstOccupiedStackOfItem(IItemHandler inv, Ingredient ingred) {
        for (int i = 0; i < inv.getSlots(); ++i) {
            if (!inv.getStackInSlot(i).func_190926_b()) {
                if (ingred.apply(inv.getStackInSlot(i))) {
                    return i;
                }
            }
        }
        return -1;
    }

    public static boolean spawnStack(World world, BlockPos pos, List<ItemStack> stacks) {
        boolean spawned = true;
        for (ItemStack stack : stacks) {
            if (!stack.func_190926_b())
                if (!spawnStack(world, pos, stack, 10))
                    spawned = false;
        }
        return spawned;
    }

    public static boolean spawnStack(World world, BlockPos pos, ItemStack stack, int pickupDelay) {
        return spawnStack(world, pos.func_177958_n() + 0.5f, pos.func_177956_o() + 0.5f, pos.func_177952_p() + 0.5f, stack, pickupDelay);
    }


    public static boolean spawnStack(World world, double x, double y, double z, ItemStack stack, int pickupDelay) {
        EntityItem item = new EntityItem(world, x, y, z, stack);
        item.field_70159_w = 0;
        item.field_70181_x = 0;
        item.field_70179_y = 0;
        item.func_174867_a(pickupDelay);
        return world.func_72838_d(item);
    }

    public static void spawnStack(World world, double x, double y, double z, int count, ItemStack stack) {
        ItemStack copy = stack.func_77946_l();
        if (copy.func_190916_E() > count)
            copy.func_190920_e(count);
        spawnStack(world, x, y, z, copy, 10);
    }

    public static void ejectStackWithOffset(World world, BlockPos pos, List<ItemStack> stacks) {
        for (ItemStack stack : stacks) {
            if (!stack.func_190926_b())
                ejectStackWithOffset(world, pos, stack.func_77946_l());
        }
    }

    public static void ejectStackWithOffset(World world, BlockPos pos, ItemStack... stacks) {
        for (ItemStack stack : stacks) {
            if (!stack.func_190926_b())
                ejectStackWithOffset(world, pos, stack.func_77946_l());
        }
    }

    public static void ejectStackWithOffset(World world, BlockPos pos, ItemStack stack) {
        if (stack.func_190926_b())
            return;
        VectorBuilder builder = new VectorBuilder();
        new StackEjector(world, stack,builder.set(pos).rand(0.5f).offset(0.25f).build(), builder.setGaussian(0.05f).build()).ejectStack();
    }


    public static void ejectStack(@Nonnull World world, double x, double y, double z, ItemStack stack, int pickupDelay) {
        if (world.field_72995_K)
            return;

        VectorBuilder builder = new VectorBuilder();
        StackEjector ejector = new StackEjector(world, stack,builder.set(x,y,z).build());
        ejector.setPickupDelay(pickupDelay);
        ejector.ejectStack();
    }

    public static void ejectStack(@Nonnull World world, double x, double y, double z, ItemStack stack) {
        ejectStack(world, x, y, z, stack, 10);
    }


    public static void ejectBrokenItems(World world, BlockPos pos, ResourceLocation lootLocation) {
        if (!world.field_72995_K) {
            LootContext.Builder build = new LootContext.Builder((WorldServer) world);
            List<ItemStack> stacks = world.func_184146_ak().func_186521_a(lootLocation).func_186462_a(world.field_73012_v, build.func_186471_a());
            if (!stacks.isEmpty()) {
                ejectStackWithOffset(world, pos, stacks);
            }
        }
    }

    public static void writeToStack(IItemHandler inv, ItemStack stack) {
        NonNullList<ItemStack> list = NonNullList.func_191197_a(inv.getSlots(), ItemStack.field_190927_a);
        for (int i = 0; i < inv.getSlots(); i++) {
            if (!inv.getStackInSlot(i).func_190926_b()) {
                list.set(i, inv.getStackInSlot(i).func_77946_l());
            }
        }
        NBTTagCompound tag = ItemStackHelper.func_191282_a(new NBTTagCompound(), list);

        if (!tag.func_82582_d())
            stack.func_77982_d(tag);
    }

    public static void readFromStack(IItemHandler inv, ItemStack stack) {
        if (!stack.func_190926_b() && stack.func_77942_o()) {
            NonNullList<ItemStack> list = NonNullList.func_191197_a(inv.getSlots(), ItemStack.field_190927_a);
            NBTTagCompound tag = stack.func_77978_p();
            if (tag != null) {
                ItemStackHelper.func_191283_b(tag, list);
                for (int i = 0; i < inv.getSlots(); i++) {
                    inv.insertItem(i, list.get(i), false);
                }
            }
        }
    }

    public static int calculateComparatorLevel(@Nonnull IItemHandler inventory) {
        int i = 0;
        float f = 0.0F;
        for (int j = 0; j < inventory.getSlots(); ++j) {
            ItemStack itemstack = inventory.getStackInSlot(j);
            if (!itemstack.func_190926_b()) {
                f += (float) itemstack.func_190916_E() / (float) itemstack.func_77976_d();
                ++i;
            }
        }
        f = f / (float) inventory.getSlots();
        return MathHelper.func_76141_d(f * 14.0F) + (i > 0 ? 1 : 0);
    }

    public static ItemStack setCount(ItemStack input, int count) {
        if(input.func_190926_b())
            return input;
        ItemStack stack = input.func_77946_l();
        stack.func_190920_e(count);
        return stack;
    }


    public static boolean matches(ItemStack one, ItemStack two) {
        if (one.func_77969_a(two))
            return true;
        if (one.func_77973_b() == two.func_77973_b() && (one.func_77960_j() == OreDictionary.WILDCARD_VALUE || two.func_77960_j() == OreDictionary.WILDCARD_VALUE))
            return true;
        return false;
    }

    public static boolean matchesSize(ItemStack one, ItemStack two) {
        return one.func_190916_E() == two.func_190916_E() && matches(one, two);
    }


    public static <T> boolean matchesPredicate(List<T> oneList, List<T> twoList, BiPredicate<T,T> matches) {
        if (oneList.size() != twoList.size())
            return false; //trivial case
        HashSet<T> alreadyMatched = new HashSet<>();
        for (T one : oneList) {
            Optional<T> found = twoList.stream().filter(two -> !alreadyMatched.contains(two) && matches.test(one,two)).findFirst();
            if (found.isPresent())
                alreadyMatched.add(found.get()); //Don't match twice
            else
                return false; //This T doesn't match, thus the two lists don't match
        }
        return true;
    }


    public static boolean matchesExact(List<ItemStack> oneList, List<ItemStack> twoList) {
        return matchesPredicate(oneList, twoList, InvUtils::matchesSize);
    }

    public static boolean matches(List<ItemStack> oneList, List<ItemStack> twoList) {
        return matchesPredicate(oneList,twoList, InvUtils::matches);
    }

    public static <T> List<List<T>> splitIntoBoxes(List<T> stacks, int boxes) {
        ArrayList<List<T>> splitStacks = new ArrayList<>();
        for (int i = 0; i < boxes; i++) {
            final int finalI = i;
            splitStacks.add(IntStream.range(0, stacks.size()).filter(index -> index % boxes == finalI).mapToObj(stacks::get).collect(Collectors.toList()));
        }
        return splitStacks;
    }

    public static boolean isEmpty(IItemHandler inventory) {
        int inventorySize = inventory.getSlots();
        for (int i = 0; i < inventorySize; i++){
            if (!inventory.getStackInSlot(i).func_190926_b()){
                return false;
            }
        }
        return true;
    }
}
