package betterwithmods.common.registry;

import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import net.minecraft.block.Block;
import net.minecraft.block.state.IBlockState;
import net.minecraft.init.Blocks;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.Rotation;
import net.minecraft.util.math.BlockPos;
import net.minecraft.world.World;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.function.Predicate;

public class TurntableRotationManager {


    public static final HashSet<Predicate<Block>> BLOCK_PREDICATE_ATTACHMENTS = Sets.newHashSet();
    public static final HashSet<Block> BLOCK_ATTACHMENTS = Sets.newHashSet();
    public static final HashMap<Predicate<Block>, IRotation> PREDICATE_ROTATIONS = Maps.newHashMap();
    public static final HashMap<Block, IRotation> BLOCK_ROTATIONS = Maps.newHashMap();

    private static IRotation NO_ROTATION = (world, pos) -> false, BASE_ROTATION = (world, pos) -> true;

    public interface IRotation {
        boolean isValid(World world, BlockPos pos);

        default boolean rotate(World world, BlockPos pos, Rotation rotation) {
            IBlockState state = world.func_180495_p(pos);
            return world.func_175656_a(pos, state.func_185907_a(rotation));
        }

        default boolean canTransmitVertically(World world, BlockPos pos) {
            Block block = world.func_180495_p(pos).func_177230_c();
            if (block == Blocks.field_150359_w || block == Blocks.field_150399_cn)
                return true;
            return world.func_175677_d(pos, false);
        }

        default boolean canTransmitHorizontally(World world, BlockPos pos) {
            return true;
        }
    }

    public static boolean isAttachment(Block block) {
        return BLOCK_ATTACHMENTS.contains(block) || BLOCK_PREDICATE_ATTACHMENTS.stream().anyMatch(p -> p.test(block));
    }

    public static void addAttachment(Block block) {
        BLOCK_ATTACHMENTS.add(block);
    }

    public static void addAttachment(Predicate<Block> block) {
        BLOCK_PREDICATE_ATTACHMENTS.add(block);
    }

    public static void addRotationBlacklist(Predicate<Block> predicate) {
        addRotationHandler(predicate, NO_ROTATION);
    }

    public static void addRotationHandler(Predicate<Block> predicate, IRotation rotation) {
        PREDICATE_ROTATIONS.put(predicate, rotation);
    }

    public static void addRotationHandler(Block block, IRotation rotation) {
        BLOCK_ROTATIONS.put(block, rotation);
    }

    public static IRotation rotate(World world, BlockPos pos, Rotation rotation) {
        Block block = world.func_180495_p(pos).func_177230_c();

        IRotation handler = BLOCK_ROTATIONS.getOrDefault(block, null);
        if (handler == null) {
            for (Map.Entry<Predicate<Block>, IRotation> entry : PREDICATE_ROTATIONS.entrySet()) {
                if (entry.getKey().test(block)) {
                    handler = entry.getValue();
                    break;
                }
            }
        }
        if (handler == null)
            handler = BASE_ROTATION;

        if (handler.isValid(world, pos)) {
            if (handler.rotate(world, pos, rotation)) {
                world.func_180497_b(pos, block, block.func_149738_a(world), 1);
                world.func_175685_c(pos, block, true);
            }
            return handler;
        }
        return null;
    }

    private static BlockPos rotateAround(BlockPos centerPos, EnumFacing facing, Rotation rotation) {
        return centerPos.func_177982_a(facing.func_82601_c(), 0, facing.func_82599_e());
    }

    public static void rotateAttachments(World world, BlockPos pos, Rotation rotation) {
        HashMap<EnumFacing, IBlockState> blocks = Maps.newHashMap();
        for (EnumFacing facing : EnumFacing.field_176754_o) {
            BlockPos newPos = pos.func_177972_a(facing);
            IBlockState state = world.func_180495_p(newPos);
            if (isAttachment(state.func_177230_c())) {
                blocks.put(facing, state);
                world.func_175698_g(newPos);
            }
        }
        if (blocks.isEmpty())
            return;
        for (EnumFacing facing : blocks.keySet()) {
            IBlockState state = blocks.get(facing);
            EnumFacing newFacing = rotation == Rotation.CLOCKWISE_90 ? facing.func_176746_e() : facing.func_176735_f();
            BlockPos newPos = rotateAround(pos, newFacing, rotation);
            if(!world.func_180495_p(newPos).func_185904_a().func_76222_j()) {
                state.func_177230_c().func_176226_b(world, pos.func_177972_a(facing), state, 0);
                world.func_175698_g(pos.func_177972_a(facing));
            } else {
                world.func_175656_a(newPos, state.func_185907_a(rotation));
            }
        }
    }

}
