package betterwithmods.common.blocks.mechanical.tile;

import betterwithmods.api.BWMAPI;
import betterwithmods.api.capabilities.CapabilityMechanicalPower;
import betterwithmods.api.tile.ICrankable;
import betterwithmods.api.tile.IHeated;
import betterwithmods.api.tile.IMechanicalPower;
import betterwithmods.api.util.IProgressSource;
import betterwithmods.common.blocks.mechanical.BlockCookingPot;
import betterwithmods.common.blocks.tile.TileEntityVisibleInventory;
import betterwithmods.common.registry.bulk.manager.CraftingManagerBulk;
import betterwithmods.common.registry.bulk.recipes.CookingPotRecipe;
import betterwithmods.common.registry.heat.BWMHeatRegistry;
import betterwithmods.util.DirUtils;
import betterwithmods.util.InvUtils;
import net.minecraft.block.Block;
import net.minecraft.block.state.IBlockState;
import net.minecraft.entity.item.EntityItem;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.init.Items;
import net.minecraft.init.SoundEvents;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.util.EntitySelectors;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.EnumParticleTypes;
import net.minecraft.util.SoundCategory;
import net.minecraft.util.math.AxisAlignedBB;
import net.minecraft.util.math.BlockPos;
import net.minecraft.util.math.Vec3i;
import net.minecraft.world.World;
import net.minecraftforge.common.capabilities.Capability;

import java.util.HashMap;
import java.util.List;
import java.util.Random;


public abstract class TileEntityCookingPot extends TileEntityVisibleInventory implements IMechanicalPower, IHeated, ICrankable, IProgressSource {
    private static final int MAX_TIME = 1000;
    public int cookProgress, cookTime;
    public EnumFacing facing;
    public int heat;
    protected CraftingManagerBulk<CookingPotRecipe> manager;

    public TileEntityCookingPot(CraftingManagerBulk<CookingPotRecipe> manager) {
        this.manager = manager;
        this.cookProgress = 0;
        this.cookTime = 0;
        this.occupiedSlots = 0;
        this.facing = EnumFacing.UP;
        this.hasCapability = f -> f == facing;
    }

    @Override
    public boolean hasFastRenderer() {
        return false;
    }

    @Override
    public boolean hasCapability(Capability<?> capability, EnumFacing facing) {
        return capability == CapabilityMechanicalPower.MECHANICAL_POWER || super.hasCapability(capability, facing);
    }

    @Override
    public <T> T getCapability(Capability<T> capability, EnumFacing facing) {
        if (capability == CapabilityMechanicalPower.MECHANICAL_POWER) {
            return CapabilityMechanicalPower.MECHANICAL_POWER.cast(this);
        }
        return super.getCapability(capability, facing);
    }

    private boolean isInputtingPower(EnumFacing facing) {
        return this.getMechanicalInput(facing) > 0;
    }

    private EnumFacing getPoweredSide() {
        for (EnumFacing facing : EnumFacing.field_176754_o) {
            if (isInputtingPower(facing))
                return facing;
        }
        return EnumFacing.UP;
    }

    private boolean isPowered() {
        for (EnumFacing facing : EnumFacing.field_176754_o) {
            if (isInputtingPower(facing))
                return true;
        }
        return false;
    }

    @Override
    public int getInventorySize() {
        return 27;
    }

    @Override
    public void func_145839_a(NBTTagCompound tag) {
        super.func_145839_a(tag);
        this.facing = EnumFacing.func_82600_a(value(tag, "facing", EnumFacing.UP.func_176745_a()));
        this.cookProgress = value(tag, "progress", 0);
        this.cookTime = value(tag, "time", 4000);
        this.heat = value(tag, "heat", 0);
    }

    @Override
    public NBTTagCompound func_189515_b(NBTTagCompound tag) {
        NBTTagCompound t = super.func_189515_b(tag);
        t.func_74768_a("facing", facing.func_176745_a());
        t.func_74768_a("progress", this.cookProgress);
        t.func_74768_a("heat", this.heat);
        t.func_74768_a("time", this.cookTime);
        return t;
    }

    @Override
    public void func_73660_a() {

        if (getBlock() instanceof BlockCookingPot) {
            IBlockState state = this.getBlockWorld().func_180495_p(this.field_174879_c);
            if (isPowered()) {
                this.cookProgress = 0;
                this.facing = getPoweredSide();

                ejectInventory(DirUtils.rotateFacingAroundY(this.facing, false));
            } else {
                if (this.facing != EnumFacing.UP)
                    this.facing = EnumFacing.UP;

                spawnParticles();

                entityCollision();

                //Only do crafting on the server
                if(!field_145850_b.field_72995_K && !InvUtils.isEmpty(inventory)) {
                    int heat = findHeat(func_174877_v());
                    if (this.heat != heat) {
                        this.heat = heat;
                        this.cookProgress = 0;
                    }
                    int time = findCookTime();
                    if (this.cookTime != time) {
                        this.cookTime = time;
                    }
                    manager.craftRecipe(field_145850_b, this, inventory);
                }
            }
            if (facing != state.func_177229_b(DirUtils.TILTING)) {
                field_145850_b.func_175656_a(field_174879_c, state.func_177226_a(DirUtils.TILTING, facing));
            }
        }
    }

    private int findCookTime() {
        int divisor = -heat;
        for (int x = -1; x <= 1; x++) {
            for (int z = -1; z <= 1; z++) {
                divisor += findHeat(field_174879_c.func_177982_a(x, 0, z));
            }
        }
        if (divisor != 0)
            return MAX_TIME / divisor;
        return MAX_TIME;
    }


    @Override
    public int getHeat(World world, BlockPos pos) {
        return heat;
    }

    private int findHeat(BlockPos pos) {
        return getHeatCached(pos.func_177977_b());
    }

    private HashMap<BlockPos, BWMHeatRegistry.HeatSource> heatCache = new HashMap<>();
    private int getHeatCached(BlockPos pos){
        BWMHeatRegistry.HeatSource src = heatCache.get(pos);
        if (src != null && src.matches(field_145850_b, pos)){
            return src.getHeat();
        } else if (src!=null){
            heatCache.remove(pos);
        }
        src = BWMHeatRegistry.get(field_145850_b, pos);
        if (src != null){
            heatCache.put(pos, src);
            return src.getHeat();
        }
        return 0;
    }

    private void spawnParticles() {
        Random random = this.getBlockWorld().field_73012_v;
        if (heat >= BWMHeatRegistry.STOKED_HEAT && random.nextDouble() < 0.2) {
            double xOffset = 0.25 + random.nextDouble() * 0.5;
            double zOffset = 0.25 + random.nextDouble() * 0.5;
            this.getBlockWorld().func_175688_a(EnumParticleTypes.CLOUD, field_174879_c.func_177958_n() + xOffset, field_174879_c.func_177956_o() + 0.75F, field_174879_c.func_177952_p() + zOffset, 0, 0.05 + random.nextDouble() * 0.05, 0);
        }
    }

    //TODO move this to Block#onEntityCollision by lowering the bounding box a bit, like the filtered hopper
    private void entityCollision() {
        if (captureDroppedItems()) {
            getBlockWorld().func_180497_b(field_174879_c, this.func_145838_q(), this.func_145838_q().func_149738_a(getBlockWorld()), 5);
            this.func_70296_d();
        }
    }


    public List<EntityItem> getCaptureItems(World worldIn, BlockPos pos) {
        return worldIn.func_175647_a(EntityItem.class, new AxisAlignedBB(pos.func_177958_n(), pos.func_177956_o(), pos.func_177952_p(), pos.func_177958_n() + 1D, pos.func_177956_o() + 1.5D, pos.func_177952_p() + 1D), EntitySelectors.field_94557_a);
    }

    private boolean captureDroppedItems() {
        boolean insert = false;
        if (!InvUtils.isFull(inventory)) {
            List<EntityItem> items = this.getCaptureItems(getBlockWorld(), func_174877_v());
            for (EntityItem item : items)
                insert |= InvUtils.insertFromWorld(inventory, item, 0, 27, false);
        }
        if (insert) {
            this.getBlockWorld().func_184148_a(null, field_174879_c.func_177958_n(), field_174879_c.func_177956_o(), field_174879_c.func_177952_p(), SoundEvents.field_187638_cR, SoundCategory.PLAYERS, 0.2F, ((getBlockWorld().field_73012_v.nextFloat() - getBlockWorld().field_73012_v.nextFloat()) * 0.7F + 1.0F) * 2.0F);
            return true;
        }
        return false;
    }

    public void ejectInventory(EnumFacing facing) {
        int index = InvUtils.getFirstOccupiedStackNotOfItem(inventory, Items.field_151118_aC);
        if (index >= 0 && index < inventory.getSlots()) {
            ItemStack stack = inventory.getStackInSlot(index);
            int ejectStackSize = 8;
            if (8 > stack.func_190916_E()) {
                ejectStackSize = stack.func_190916_E();
            }
            BlockPos target = field_174879_c.func_177972_a(facing);
            ItemStack eject = new ItemStack(stack.func_77973_b(), ejectStackSize, stack.func_77952_i());
            InvUtils.copyTags(eject, stack);
            IBlockState targetState = getBlockWorld().func_180495_p(target);
            boolean ejectIntoWorld = getBlockWorld().func_175623_d(target) ||
                    targetState.func_177230_c().func_176200_f(getBlockWorld(), target) ||
                    !targetState.func_185904_a().func_76220_a() ||
                    targetState.func_185900_c(getBlockWorld(), target).field_72337_e < 0.5d;
            if (ejectIntoWorld) {
                this.getBlockWorld().func_184133_a(null, field_174879_c, SoundEvents.field_187638_cR, SoundCategory.BLOCKS, 0.2F, ((getBlockWorld().field_73012_v.nextFloat() - getBlockWorld().field_73012_v.nextFloat()) * 0.7F + 1.0F) * 2.0F);
                ejectStack(getBlockWorld(), target, facing, eject);
                inventory.extractItem(index, ejectStackSize, false);
            }
        }
    }

    public void ejectStack(World world, BlockPos pos, EnumFacing facing, ItemStack stack) {
        if (world.field_72995_K)
            return;
        Vec3i vec = new BlockPos(0, 0, 0).func_177972_a(facing);
        EntityItem item = new EntityItem(world, pos.func_177958_n() + 0.5F - (vec.func_177958_n() / 4d), pos.func_177956_o() + 0.25D, pos.func_177952_p() + 0.5D - (vec.func_177952_p() / 4d), stack);
        float velocity = 0.05F;
        item.field_70159_w = (double) (vec.func_177958_n() * velocity);
        item.field_70181_x = vec.func_177956_o() * velocity * 0.1;
        item.field_70179_y = (double) (vec.func_177952_p() * velocity);
        item.func_174869_p();
        world.func_72838_d(item);
    }

    @Override
    public void func_70296_d() {
        super.func_70296_d();
        if (this.getBlockWorld() != null) {
            IBlockState state = getBlockWorld().func_180495_p(field_174879_c);
            getBlockWorld().func_184138_a(field_174879_c, state, state, 3);
        }
    }

    public boolean isUseableByPlayer(EntityPlayer player) {
        return this.getBlockWorld().func_175625_s(this.field_174879_c) == this && player.func_70092_e(this.field_174879_c.func_177958_n() + 0.5D, this.field_174879_c.func_177956_o() + 0.5D, this.field_174879_c.func_177952_p() + 0.5D) <= 64.0D;
    }

    @Override
    public int getMaxVisibleSlots() {
        return 27;
    }

    @Override
    public int getMechanicalOutput(EnumFacing facing) {
        return 0;
    }

    @Override
    public int getMechanicalInput(EnumFacing facing) {
        return BWMAPI.IMPLEMENTATION.getPowerOutput(field_145850_b, field_174879_c.func_177972_a(facing), facing.func_176734_d());
    }

    @Override
    public int getMaximumInput(EnumFacing facing) {
        return 1;
    }

    @Override
    public int getMinimumInput(EnumFacing facing) {
        return 0;
    }

    @Override
    public World getBlockWorld() {
        return super.func_145831_w();
    }

    @Override
    public BlockPos getBlockPos() {
        return func_174877_v();
    }

    @Override
    public Block getBlock() {
        return func_145838_q();
    }

    @Override
    public int getProgress() {
        return cookProgress;
    }

    @Override
    public int getMax() {
        return cookTime;
    }

}
