package vazkii.patchouli.common.book;

import com.google.common.base.Suppliers;
import com.google.gson.JsonObject;
import vazkii.patchouli.api.PatchouliAPI;
import vazkii.patchouli.api.PatchouliConfigAccess;
import vazkii.patchouli.client.base.ClientAdvancements;
import vazkii.patchouli.client.book.*;
import vazkii.patchouli.common.base.PatchouliConfig;
import vazkii.patchouli.common.base.PatchouliSounds;
import vazkii.patchouli.common.item.ItemModBook;
import vazkii.patchouli.common.util.ItemStackUtil;
import vazkii.patchouli.common.util.SerializationUtil;
import vazkii.patchouli.xplat.XplatModContainer;

import org.jetbrains.annotations.Nullable;

import java.util.HashMap;
import java.util.Map;
import java.util.function.Supplier;
import net.minecraft.class_156;
import net.minecraft.class_1799;
import net.minecraft.class_1937;
import net.minecraft.class_2561;
import net.minecraft.class_2583;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_3518;
import net.minecraft.class_5250;
import net.minecraft.class_7887;

public class Book {

	private static final String[] ORDINAL_SUFFIXES = new String[] { "th", "st", "nd", "rd", "th", "th", "th", "th", "th", "th" };
	private static final class_2960 DEFAULT_MODEL = class_2960.method_60655(PatchouliAPI.MOD_ID, "book_brown");
	private static final class_2960 DEFAULT_BOOK_TEXTURE = class_2960.method_60655(PatchouliAPI.MOD_ID, "textures/gui/book_brown.png");
	private static final class_2960 DEFAULT_FILLER_TEXTURE = class_2960.method_60655(PatchouliAPI.MOD_ID, "textures/gui/page_filler.png");
	private static final class_2960 DEFAULT_CRAFTING_TEXTURE = class_2960.method_60655(PatchouliAPI.MOD_ID, "textures/gui/crafting.png");

	private static final Map<String, String> DEFAULT_MACROS = class_156.method_656(() -> {
		Map<String, String> ret = new HashMap<>();
		ret.put("$(list", "$(li"); //  The lack of ) is intended
		ret.put("/$", "$()");
		ret.put("<br>", "$(br)");

		ret.put("$(item)", "$(#b0b)");
		ret.put("$(thing)", "$(#490)");
		return ret;
	});

	private BookContents contents;

	private boolean wasUpdated = false;

	public final XplatModContainer owner;
	public final class_2960 id;
	private Supplier<class_1799> bookItem;

	public final int textColor, headerColor, nameplateColor, linkColor, linkHoverColor, progressBarColor, progressBarBackground;

	public final boolean isExternal;

	// JSON Loaded properties

	public final String name;
	public final String landingText;

	public final class_2960 bookTexture, fillerTexture, craftingTexture;

	public final class_2960 model;

	public final boolean useBlockyFont;

	public final class_2960 openSound, flipSound;

	public final boolean showProgress;

	public final String indexIconRaw;

	public final String version;
	public final String subtitle;

	@Nullable public final class_2960 creativeTab;

	@Nullable public final class_2960 advancementsTab;

	public final boolean noBook;

	public final boolean showToasts;

	public final boolean pauseGame;

	public final boolean isPamphlet;

	public final boolean i18n;
	@Nullable public final PatchouliConfigAccess.TextOverflowMode overflowMode;

	public final Map<String, String> macros = new HashMap<>();

	private static int parseColor(JsonObject root, String key, String defaultColor) {
		return 0xFF000000 | Integer.parseInt(class_3518.method_15253(root, key, defaultColor), 16);
	}

	public Book(JsonObject root, XplatModContainer owner, class_2960 id, boolean external) {
		this.name = class_3518.method_15265(root, "name");
		this.landingText = class_3518.method_15253(root, "landing_text", "patchouli.gui.lexicon.landing_info");
		this.bookTexture = SerializationUtil.getAsResourceLocation(root, "book_texture", DEFAULT_BOOK_TEXTURE);
		this.fillerTexture = SerializationUtil.getAsResourceLocation(root, "filler_texture", DEFAULT_FILLER_TEXTURE);
		this.craftingTexture = SerializationUtil.getAsResourceLocation(root, "crafting_texture", DEFAULT_CRAFTING_TEXTURE);
		this.model = SerializationUtil.getAsResourceLocation(root, "model", DEFAULT_MODEL).method_45138("item/");
		this.useBlockyFont = class_3518.method_15258(root, "use_blocky_font", false);

		this.owner = owner;
		this.id = id;
		this.isExternal = external;
		this.textColor = parseColor(root, "text_color", "000000");
		this.headerColor = parseColor(root, "header_color", "333333");
		this.nameplateColor = parseColor(root, "nameplate_color", "FFDD00");
		this.linkColor = parseColor(root, "link_color", "0000EE");
		this.linkHoverColor = parseColor(root, "link_hover_color", "8800EE");
		this.progressBarColor = parseColor(root, "progress_bar_color", "FFFF55");
		this.progressBarBackground = parseColor(root, "progress_bar_background", "DDDDDD");
		this.openSound = SerializationUtil.getAsResourceLocation(root, "open_sound", PatchouliSounds.BOOK_OPEN.method_14833());
		this.flipSound = SerializationUtil.getAsResourceLocation(root, "flip_sound", PatchouliSounds.BOOK_FLIP.method_14833());
		this.showProgress = class_3518.method_15258(root, "show_progress", true);
		this.indexIconRaw = class_3518.method_15253(root, "index_icon", "");
		this.version = class_3518.method_15253(root, "version", "0");
		this.subtitle = class_3518.method_15253(root, "subtitle", "");
		this.creativeTab = SerializationUtil.getAsResourceLocation(root, "creative_tab", null);
		this.advancementsTab = SerializationUtil.getAsResourceLocation(root, "advancements_tab", null);
		this.noBook = class_3518.method_15258(root, "dont_generate_book", false);
		this.showToasts = class_3518.method_15258(root, "show_toasts", true);
		this.pauseGame = class_3518.method_15258(root, "pause_game", false);
		this.isPamphlet = class_3518.method_15258(root, "pamphlet", false);
		this.i18n = class_3518.method_15258(root, "i18n", false);
		this.overflowMode = SerializationUtil.getAsEnum(root, "text_overflow_mode", PatchouliConfigAccess.TextOverflowMode.class, null);

		boolean useResourcePack = class_3518.method_15258(root, "use_resource_pack", false);
		if (!this.isExternal && !useResourcePack) {
			String message = "Book %s has use_resource_pack set to false. ".formatted(this.id)
					+ "This behaviour was removed in 1.20. "
					+ "The book author should enable this flag and move all book contents clientside to /assets/, "
					+ "leaving the book.json in /data/. See https://vazkiimods.github.io/Patchouli/docs/upgrading/upgrade-guide-120 for details.";
			throw new IllegalArgumentException(message);
		}

		// Check legacy extensions flag
		class_2960 extensionTargetID = SerializationUtil.getAsResourceLocation(root, "extend", null);
		if (extensionTargetID != null) {
			String message = "Book %s is declared to extend %s. ".formatted(this.id, extensionTargetID)
					+ "This behaviour was removed in 1.20. "
					+ "The author should simply ship the extra content they want to add or override in a resource pack.";
			throw new IllegalArgumentException(message);
		}

		var customBookItem = class_3518.method_15253(root, "custom_book_item", "");
		if (noBook) {
			// Need lazy parsing for mods that load after Patchouli, as parser looks up item and components
			// in registries; wrap in try-catch in case of faulty item definition
			bookItem = Suppliers.memoize(() -> {
				try {
					return ItemStackUtil.loadFromParsed(
							ItemStackUtil.deserializeStack(customBookItem, class_7887.method_46817()));
				} catch (Exception e) {
					PatchouliAPI.LOGGER.warn("Failed to parse item \"{}\" for book {} defined by mod {}, skipping",
							customBookItem, id, owner.getId(), e);
					return class_1799.field_8037;
				}
			});
		} else {
			bookItem = Suppliers.memoize(() -> ItemModBook.forBook(id));
		}

		macros.putAll(DEFAULT_MACROS);
		for (var e : class_3518.method_15281(root, "macros", new JsonObject()).entrySet()) {
			macros.put(e.getKey(), class_3518.method_15287(e.getValue(), "macro value"));
		}
	}

	public class_1799 getBookItem() {
		return this.bookItem.get();
	}

	public void markUpdated() {
		wasUpdated = true;
	}

	public boolean popUpdated() {
		boolean updated = wasUpdated;
		wasUpdated = false;
		return updated;
	}

	/**
	 * Must only be called on client
	 * 
	 * @param singleBook Hint that the book was reloaded through the button on the main page
	 */
	public void reloadContents(class_1937 level, boolean singleBook) {
		try {
			contents = BookContentsBuilder.loadAndBuildFor(level, this, singleBook);
		} catch (Exception e) {
			PatchouliAPI.LOGGER.error("Error loading and compiling book {}, using empty contents", id, e);
			contents = BookContents.empty(this, e);
		}
	}

	public final boolean advancementsEnabled() {
		return !PatchouliConfig.get().disableAdvancementLocking()
				&& !PatchouliConfig.get().noAdvancementBooks().contains(id.toString());
	}

	public void reloadLocks(boolean suppressToasts) {
		getContents().entries.values().forEach(BookEntry::updateLockStatus);
		getContents().categories.values().forEach(c -> c.updateLockStatus(true));

		boolean updated = popUpdated();
		if (updated && !suppressToasts && advancementsEnabled() && showToasts) {
			ClientAdvancements.sendBookToast(this);
		}
	}

	public String getOwnerName() {
		return owner.getName();
	}

	public class_2583 getFontStyle() {
		if (useBlockyFont) {
			return class_2583.field_24360;
		} else {
			return class_2583.field_24360.method_27704(class_310.field_24211);
		}
	}

	public class_5250 getSubtitle() {
		class_2561 editionStr;

		try {
			int ver = Integer.parseInt(version);
			if (ver == 0) {
				return class_2561.method_43471(subtitle);
			}

			editionStr = class_2561.method_43470(numberToOrdinal(ver));
		} catch (NumberFormatException e) {
			editionStr = class_2561.method_43471("patchouli.gui.lexicon.dev_edition");
		}

		return class_2561.method_43469("patchouli.gui.lexicon.edition_str", editionStr);
	}

	public BookIcon getIcon() {
		if (indexIconRaw == null || indexIconRaw.isEmpty()) {
			return new BookIcon.StackIcon(getBookItem());
		} else {
			return BookIcon.from(indexIconRaw);
		}
	}

	private static String numberToOrdinal(int i) {
		return i % 100 == 11 || i % 100 == 12 || i % 100 == 13 ? i + "th" : i + ORDINAL_SUFFIXES[i % 10];
	}

	public BookContents getContents() {
		return contents != null ? contents : BookContents.empty(this, null);
	}
}
