/*
 * This class is distributed as part of the Botania Mod.
 * Get the Source Code in github:
 * https://github.com/Vazkii/Botania
 *
 * Botania is Open Source and distributed under the
 * Botania License: http://botaniamod.net/license.php
 */
package vazkii.botania.common.impl.mana;

import com.google.common.collect.Iterables;

import it.unimi.dsi.fastutil.objects.Object2IntMap;
import it.unimi.dsi.fastutil.objects.Object2IntMap.Entry;
import it.unimi.dsi.fastutil.objects.Object2IntOpenHashMap;
import vazkii.botania.api.BotaniaAPI;
import vazkii.botania.api.item.ManaProficiencyArmor;
import vazkii.botania.api.mana.*;
import vazkii.botania.xplat.XplatAbstractions;

import java.util.*;
import net.minecraft.class_1263;
import net.minecraft.class_1304;
import net.minecraft.class_1657;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_1890;
import net.minecraft.class_1893;

public class ManaItemHandlerImpl implements ManaItemHandler {
	@Override
	public List<class_1799> getManaItems(class_1657 player) {
		if (player == null) {
			return Collections.emptyList();
		}

		List<class_1799> toReturn = new ArrayList<>();

		for (class_1799 stackInSlot : Iterables.concat(player.method_31548().field_7547, player.method_31548().field_7544)) {
			if (!stackInSlot.method_7960() && XplatAbstractions.INSTANCE.findManaItem(stackInSlot) != null) {
				toReturn.add(stackInSlot);
			}
		}

		XplatAbstractions.INSTANCE.fireManaItemEvent(player, toReturn);
		return toReturn;
	}

	@Override
	public List<class_1799> getManaAccesories(class_1657 player) {
		if (player == null) {
			return Collections.emptyList();
		}

		class_1263 acc = BotaniaAPI.instance().getAccessoriesInventory(player);

		List<class_1799> toReturn = new ArrayList<>(acc.method_5439());

		for (int slot = 0; slot < acc.method_5439(); slot++) {
			class_1799 stackInSlot = acc.method_5438(slot);

			if (!stackInSlot.method_7960() && XplatAbstractions.INSTANCE.findManaItem(stackInSlot) != null) {
				toReturn.add(stackInSlot);
			}
		}

		return toReturn;
	}

	@Override
	public int requestMana(class_1799 stack, class_1657 player, int manaToGet, boolean remove) {
		if (stack.method_7960()) {
			return 0;
		}

		List<class_1799> items = getManaItems(player);
		List<class_1799> acc = getManaAccesories(player);
		int manaReceived = 0;
		for (class_1799 stackInSlot : Iterables.concat(items, acc)) {
			if (stackInSlot == stack) {
				continue;
			}
			var manaItem = XplatAbstractions.INSTANCE.findManaItem(stackInSlot);
			if (manaItem.canExportManaToItem(stack) && manaItem.getMana() > 0) {
				var requestor = XplatAbstractions.INSTANCE.findManaItem(stack);
				if (requestor != null && !requestor.canReceiveManaFromItem(stackInSlot)) {
					continue;
				}

				int mana = Math.min(manaToGet - manaReceived, manaItem.getMana());

				if (remove) {
					manaItem.addMana(-mana);
				}

				manaReceived += mana;

				if (manaReceived >= manaToGet) {
					break;
				}
			}
		}

		return manaReceived;
	}

	@Override
	public boolean requestManaExact(class_1799 stack, class_1657 player, int manaToGet, boolean remove) {
		if (stack.method_7960()) {
			return false;
		}

		List<class_1799> items = getManaItems(player);
		List<class_1799> acc = getManaAccesories(player);
		int manaReceived = 0;
		Object2IntMap<ManaItem> manaToRemove = new Object2IntOpenHashMap<>();
		for (class_1799 stackInSlot : Iterables.concat(items, acc)) {
			if (stackInSlot == stack) {
				continue;
			}
			var manaItemSlot = XplatAbstractions.INSTANCE.findManaItem(stackInSlot);
			if (manaItemSlot.canExportManaToItem(stack)) {
				var manaItem = XplatAbstractions.INSTANCE.findManaItem(stack);
				if (manaItem != null && !manaItem.canReceiveManaFromItem(stackInSlot)) {
					continue;
				}

				int mana = Math.min(manaToGet - manaReceived, manaItemSlot.getMana());

				if (remove) {
					manaToRemove.put(manaItemSlot, mana);
				}

				manaReceived += mana;

				if (manaReceived >= manaToGet) {
					break;
				}
			}
		}

		if (manaReceived == manaToGet) {
			for (var e : manaToRemove.object2IntEntrySet()) {
				e.getKey().addMana(-e.getIntValue());
			}
			return true;
		}

		return false;
	}

	@Override
	public int dispatchMana(class_1799 stack, class_1657 player, int manaToSend, boolean add) {
		if (stack.method_7960()) {
			return 0;
		}

		List<class_1799> items = getManaItems(player);
		List<class_1799> acc = getManaAccesories(player);
		for (class_1799 stackInSlot : Iterables.concat(items, acc)) {
			if (stackInSlot == stack) {
				continue;
			}
			ManaItem manaItemSlot = XplatAbstractions.INSTANCE.findManaItem(stackInSlot);
			if (manaItemSlot.canReceiveManaFromItem(stack)) {
				var manaItem = XplatAbstractions.INSTANCE.findManaItem(stack);
				if (manaItem != null && !manaItem.canExportManaToItem(stackInSlot)) {
					continue;
				}

				int received;
				if (manaItemSlot.getMana() + manaToSend <= manaItemSlot.getMaxMana()) {
					received = manaToSend;
				} else {
					received = manaToSend - (manaItemSlot.getMana() + manaToSend - manaItemSlot.getMaxMana());
				}

				if (add) {
					manaItemSlot.addMana(manaToSend);
				}

				return received;
			}
		}

		return 0;
	}

	@Override
	public boolean dispatchManaExact(class_1799 stack, class_1657 player, int manaToSend, boolean add) {
		if (stack.method_7960()) {
			return false;
		}

		List<class_1799> items = getManaItems(player);
		List<class_1799> acc = getManaAccesories(player);
		for (class_1799 stackInSlot : Iterables.concat(items, acc)) {
			if (stackInSlot == stack) {
				continue;
			}
			ManaItem manaItemSlot = XplatAbstractions.INSTANCE.findManaItem(stackInSlot);
			if (manaItemSlot.getMana() + manaToSend <= manaItemSlot.getMaxMana() && manaItemSlot.canReceiveManaFromItem(stack)) {
				var manaItem = XplatAbstractions.INSTANCE.findManaItem(stack);
				if (manaItem != null && !manaItem.canExportManaToItem(stackInSlot)) {
					continue;
				}

				if (add) {
					manaItemSlot.addMana(manaToSend);
				}

				return true;
			}
		}

		return false;
	}

	private int discountManaForTool(class_1799 stack, class_1657 player, int inCost) {
		float multiplier = Math.max(0F, 1F - getFullDiscountForTools(player, stack));
		return (int) (inCost * multiplier);
	}

	@Override
	public int requestManaForTool(class_1799 stack, class_1657 player, int manaToGet, boolean remove) {
		int cost = discountManaForTool(stack, player, manaToGet);
		return requestMana(stack, player, cost, remove);
	}

	@Override
	public boolean requestManaExactForTool(class_1799 stack, class_1657 player, int manaToGet, boolean remove) {
		int cost = discountManaForTool(stack, player, manaToGet);
		return requestManaExact(stack, player, cost, remove);
	}

	@Override
	public int getInvocationCountForTool(class_1799 stack, class_1657 player, int manaToGet) {
		if (stack.method_7960()) {
			return 0;
		}

		int cost = discountManaForTool(stack, player, manaToGet);
		int invocations = 0;

		List<class_1799> items = getManaItems(player);
		List<class_1799> acc = getManaAccesories(player);
		for (class_1799 stackInSlot : Iterables.concat(items, acc)) {
			if (stackInSlot == stack) {
				continue;
			}
			ManaItem manaItemSlot = XplatAbstractions.INSTANCE.findManaItem(stackInSlot);
			int availableMana = manaItemSlot.getMana();
			if (manaItemSlot.canExportManaToItem(stack) && availableMana > cost) {
				var manaItem = XplatAbstractions.INSTANCE.findManaItem(stack);
				if (manaItem != null && !manaItem.canReceiveManaFromItem(stackInSlot)) {
					continue;
				}

				invocations += availableMana / cost;
			}
		}

		return invocations;
	}

	@Override
	public float getFullDiscountForTools(class_1657 player, class_1799 tool) {
		float discount = 0F;
		for (int i = 0; i < player.method_31548().field_7548.size(); i++) {
			class_1799 armor = player.method_31548().field_7548.get(i);
			if (!armor.method_7960() && armor.method_7909() instanceof ManaDiscountArmor discountArmor) {
				discount += discountArmor.getDiscount(armor, i, player, tool);
			}
		}

		int unbreaking = class_1890.method_8225(class_1893.field_9119, tool);
		discount += unbreaking * 0.05F;
		discount = XplatAbstractions.INSTANCE.fireManaDiscountEvent(player, discount, tool);

		return discount;
	}

	@Override
	public boolean hasProficiency(class_1657 player, class_1799 manaItem) {
		boolean proficient = false;

		for (class_1304 e : class_1304.values()) {
			if (e.method_5925() != class_1304.class_1305.field_6178) {
				continue;
			}
			class_1799 stack = player.method_6118(e);
			if (!stack.method_7960()) {
				class_1792 item = stack.method_7909();
				if (item instanceof ManaProficiencyArmor armor
						&& armor.shouldGiveProficiency(stack, e, player, manaItem)) {
					proficient = true;
					break;
				}
			}
		}

		return XplatAbstractions.INSTANCE.fireManaProficiencyEvent(player, manaItem, proficient);
	}
}
