package vazkii.botania.common.config;

import com.google.gson.Gson;
import com.google.gson.JsonElement;
import com.mojang.serialization.Codec;
import com.mojang.serialization.JsonOps;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import vazkii.botania.api.BotaniaAPI;
import vazkii.botania.api.configdata.ConfigDataManager;
import vazkii.botania.api.configdata.LooniumStructureConfiguration;
import vazkii.botania.xplat.XplatAbstractions;

import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import net.minecraft.class_2960;
import net.minecraft.class_3264;
import net.minecraft.class_3300;
import net.minecraft.class_3695;
import net.minecraft.class_4309;

import static vazkii.botania.common.lib.ResourceLocationHelper.prefix;

public class ConfigDataManagerImpl implements ConfigDataManager {
	public static void registerListener() {
		XplatAbstractions.INSTANCE.registerReloadListener(class_3264.field_14190, prefix("configdata"), new ConfigDataManagerImpl());
	}

	private final Map<class_2960, LooniumStructureConfiguration> looniumConfigs = new HashMap<>();

	@Override
	public @Nullable LooniumStructureConfiguration getEffectiveLooniumStructureConfiguration(class_2960 id) {
		LooniumStructureConfiguration configuration = this.looniumConfigs.get(id);
		return configuration != null ? configuration.getEffectiveConfig(looniumConfigs::get) : null;
	}

	private static void validateLooniumConfig(Map<class_2960, LooniumStructureConfiguration> map) {
		Set<class_2960> errorEntries = new HashSet<>();
		Set<class_2960> visitedEntries = new LinkedHashSet<>();
		do {
			errorEntries.clear();
			for (Map.Entry<class_2960, LooniumStructureConfiguration> entry : map.entrySet()) {
				class_2960 id = entry.getKey();
				class_2960 parent = entry.getValue().parent;
				if (id.equals(parent)) {
					BotaniaAPI.LOGGER.warn("Ignoring Loonium structure configuration, because it specified itself as parent: {}", id);
					errorEntries.add(id);
				} else {
					visitedEntries.clear();
					if (!findTopmostParent(map, id, parent, visitedEntries)) {
						BotaniaAPI.LOGGER.warn("Ignoring Loonium structure configuration(s) without top-most parent: {}", visitedEntries);
						errorEntries.addAll(visitedEntries);
						break;
					}
				}
			}
			errorEntries.forEach(map::remove);
		} while (!errorEntries.isEmpty() && !map.isEmpty());

		if (!map.containsKey(LooniumStructureConfiguration.DEFAULT_CONFIG_ID)) {
			BotaniaAPI.LOGGER.error("Default Loonium configuration not found!");
		}
	}

	private static boolean findTopmostParent(Map<class_2960, LooniumStructureConfiguration> map,
			class_2960 id, class_2960 parent, Set<class_2960> visitedEntries) {
		if (!visitedEntries.add(id)) {
			BotaniaAPI.LOGGER.warn("Cyclic dependency between Loonium structure configurations detected: {}", visitedEntries);
			return false;
		}
		if (parent == null) {
			return true;
		}
		var parentConfig = map.get(parent);
		return parentConfig != null && findTopmostParent(map, parent, parentConfig.parent, visitedEntries);
	}

	private void applyLooniumConfig(Map<class_2960, LooniumStructureConfiguration> looniumConfigs) {
		BotaniaAPI.LOGGER.info("Loaded {} Loonium configurations", looniumConfigs.size());
		this.looniumConfigs.putAll(looniumConfigs);
	}

	@NotNull
	@Override
	public CompletableFuture<Void> method_25931(@NotNull class_4045 barrier, @NotNull class_3300 manager,
			@NotNull class_3695 prepProfiler, @NotNull class_3695 reloadProfiler,
			@NotNull Executor backgroundExecutor, @NotNull Executor gameExecutor) {
		var looniumTask = scheduleConfigParse(barrier, manager, backgroundExecutor, gameExecutor, ConfigDataType.LOONUIM);

		return CompletableFuture.allOf(looniumTask).thenRun(() -> BotaniaAPI.instance().setConfigData(this));
	}

	private <T> CompletableFuture<Void> scheduleConfigParse(class_4045 barrier, class_3300 manager,
			Executor backgroundExecutor, Executor gameExecutor, ConfigDataType<T> type) {
		return CompletableFuture.supplyAsync(() -> {
			Map<class_2960, JsonElement> resourceMap = new HashMap<>();
			class_4309.method_51148(manager, type.directory, new Gson(), resourceMap);
			Map<class_2960, T> configs = new HashMap<>(resourceMap.size());
			resourceMap.forEach((id, jsonElement) -> {
				BotaniaAPI.LOGGER.debug("Parsing {} config '{}'", type.directory, id);
				type.codec.parse(JsonOps.INSTANCE, jsonElement).result().ifPresent(c -> configs.put(id, c));
			});
			type.validateFunction.accept(configs);
			return configs;
		}, backgroundExecutor)
				.thenCompose(barrier::method_18352)
				.thenAcceptAsync(c -> type.applyFunction.accept(this, c), gameExecutor);
	}

	private record ConfigDataType<T> (Codec<T> codec, String directory,
			Consumer<Map<class_2960, T>> validateFunction,
			BiConsumer<ConfigDataManagerImpl, Map<class_2960, T>> applyFunction) {
		private static final ConfigDataType<LooniumStructureConfiguration> LOONUIM =
				new ConfigDataType<>(LooniumStructureConfiguration.CODEC, "loonium_config",
						ConfigDataManagerImpl::validateLooniumConfig, ConfigDataManagerImpl::applyLooniumConfig);

	}
}
