/*
 * This class is distributed as part of the Botania Mod.
 * Get the Source Code in github:
 * https://github.com/Vazkii/Botania
 *
 * Botania is Open Source and distributed under the
 * Botania License: http://botaniamod.net/license.php
 */
package vazkii.botania.common.block.block_entity.mana;

import com.google.common.base.Predicates;
import com.mojang.blaze3d.systems.RenderSystem;

import it.unimi.dsi.fastutil.ints.*;
import it.unimi.dsi.fastutil.ints.Int2ObjectMap.Entry;
import it.unimi.dsi.fastutil.objects.ObjectIterator;
import org.apache.commons.lang3.mutable.MutableInt;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.lwjgl.opengl.GL11;

import vazkii.botania.api.BotaniaAPI;
import vazkii.botania.api.BotaniaAPIClient;
import vazkii.botania.api.block.WandHUD;
import vazkii.botania.api.block.Wandable;
import vazkii.botania.api.internal.VanillaPacketDispatcher;
import vazkii.botania.api.item.ManaDissolvable;
import vazkii.botania.api.mana.*;
import vazkii.botania.api.mana.spark.ManaSpark;
import vazkii.botania.api.mana.spark.SparkAttachable;
import vazkii.botania.api.recipe.ManaInfusionRecipe;
import vazkii.botania.api.state.BotaniaStateProperties;
import vazkii.botania.client.core.helper.RenderHelper;
import vazkii.botania.client.fx.SparkleParticleData;
import vazkii.botania.client.fx.WispParticleData;
import vazkii.botania.client.gui.HUDHandler;
import vazkii.botania.common.block.BotaniaBlocks;
import vazkii.botania.common.block.block_entity.BotaniaBlockEntities;
import vazkii.botania.common.block.block_entity.BotaniaBlockEntity;
import vazkii.botania.common.block.mana.ManaPoolBlock;
import vazkii.botania.common.crafting.BotaniaRecipeTypes;
import vazkii.botania.common.handler.BotaniaSounds;
import vazkii.botania.common.handler.ManaNetworkHandler;
import vazkii.botania.common.helper.EntityHelper;
import vazkii.botania.common.item.BotaniaItems;
import vazkii.botania.common.item.ManaTabletItem;
import vazkii.botania.xplat.BotaniaConfig;
import vazkii.botania.xplat.XplatAbstractions;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import net.minecraft.class_1297;
import net.minecraft.class_1542;
import net.minecraft.class_1657;
import net.minecraft.class_1767;
import net.minecraft.class_1799;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_238;
import net.minecraft.class_243;
import net.minecraft.class_2487;
import net.minecraft.class_2586;
import net.minecraft.class_2680;
import net.minecraft.class_310;
import net.minecraft.class_332;
import net.minecraft.class_3419;
import net.minecraft.class_3532;
import net.minecraft.class_5712;

import static vazkii.botania.api.state.BotaniaStateProperties.OPTIONAL_DYE_COLOR;

public class ManaPoolBlockEntity extends BotaniaBlockEntity implements ManaPool, KeyLocked, SparkAttachable,
		ThrottledPacket, Wandable {
	public static final int PARTICLE_COLOR = 0x00C6FF;
	public static final float PARTICLE_COLOR_BLUE = (PARTICLE_COLOR & 0xFF) / 255F;
	public static final float PARTICLE_COLOR_GREEN = (PARTICLE_COLOR >> 8 & 0xFF) / 255F;
	public static final float PARTICLE_COLOR_RED = (PARTICLE_COLOR >> 16 & 0xFF) / 255F;
	public static final int MAX_MANA = 1000000;
	private static final int MAX_MANA_DILLUTED = 10000;

	private static final String TAG_MANA = "mana";
	private static final String TAG_OUTPUTTING = "outputting";
	private static final String TAG_MANA_CAP = "manaCap";
	private static final String TAG_CAN_ACCEPT = "canAccept";
	private static final String TAG_CAN_SPARE = "canSpare";
	private static final String TAG_INPUT_KEY = "inputKey";
	private static final String TAG_OUTPUT_KEY = "outputKey";
	private static final int CRAFT_EFFECT_EVENT = 0;
	private static final int CHARGE_EFFECT_EVENT = 1;
	private static final int DRAIN_EFFECT_EVENT = 2;
	private static final float CHARGING_GRAVITY = 0.003f;

	private boolean outputting = false;

	private Optional<class_1767> legacyColor = Optional.empty();
	private int mana;

	private int manaCap = -1;
	private int soundTicks = 0;
	private boolean canAccept = true;
	private boolean canSpare = true;
	boolean isDoingTransfer = false;
	int ticksDoingTransfer = 0;

	private String inputKey = "";
	private final String outputKey = "";

	private int ticks = 0;
	private boolean sendPacket = false;
	private final Int2ObjectMap<MutableInt> chargingParticles = new Int2ObjectOpenHashMap<>();
	private final Int2ObjectMap<MutableInt> drainingParticles = new Int2ObjectOpenHashMap<>();

	public ManaPoolBlockEntity(class_2338 pos, class_2680 state) {
		super(BotaniaBlockEntities.POOL, pos, state);
	}

	@Override
	public boolean isFull() {
		class_2680 stateBelow = field_11863.method_8320(field_11867.method_10074());
		return !stateBelow.method_27852(BotaniaBlocks.manaVoid) && getCurrentMana() >= getMaxMana();
	}

	@Override
	public void receiveMana(int mana) {
		int old = this.mana;
		this.mana = Math.max(0, Math.min(getCurrentMana() + mana, getMaxMana()));
		if (old != this.mana) {
			method_5431();
			markDispatchable();
		}
	}

	@Override
	public void method_11012() {
		super.method_11012();
		BotaniaAPI.instance().getManaNetworkInstance().fireManaNetworkEvent(this, ManaBlockType.POOL, ManaNetworkAction.REMOVE);
	}

	public static int calculateComparatorLevel(int mana, int max) {
		int val = (int) ((double) mana / (double) max * 15.0);
		if (mana > 0) {
			val = Math.max(val, 1);
		}
		return val;
	}

	public ManaInfusionRecipe getMatchingRecipe(@NotNull class_1799 stack, @NotNull class_2680 state) {
		List<ManaInfusionRecipe> matchingNonCatRecipes = new ArrayList<>();
		List<ManaInfusionRecipe> matchingCatRecipes = new ArrayList<>();

		for (var recipe : BotaniaRecipeTypes.getRecipes(field_11863, BotaniaRecipeTypes.MANA_INFUSION_TYPE).values()) {
			if (recipe.matches(stack)) {
				if (recipe.getRecipeCatalyst() == null) {
					matchingNonCatRecipes.add(recipe);
				} else if (recipe.getRecipeCatalyst().test(state)) {
					matchingCatRecipes.add(recipe);
				}
			}
		}

		// Recipes with matching catalyst take priority above recipes with no catalyst specified
		return !matchingCatRecipes.isEmpty() ? matchingCatRecipes.get(0) : !matchingNonCatRecipes.isEmpty() ? matchingNonCatRecipes.get(0) : null;
	}

	public boolean collideEntityItem(class_1542 item) {
		if (field_11863.field_9236 || !item.method_5805() || item.method_6983().method_7960()) {
			return false;
		}

		class_1799 stack = item.method_6983();

		if (stack.method_7909() instanceof ManaDissolvable dissolvable) {
			dissolvable.onDissolveTick(this, item);
		}

		if (XplatAbstractions.INSTANCE.itemFlagsComponent(item).manaInfusionSpawned) {
			return false;
		}

		ManaInfusionRecipe recipe = getMatchingRecipe(stack, field_11863.method_8320(field_11867.method_10074()));

		if (recipe != null) {
			int mana = recipe.getManaToConsume();
			if (getCurrentMana() >= mana) {
				receiveMana(-mana);

				class_1799 output = recipe.getRecipeOutput(field_11863.method_30349(), stack);
				EntityHelper.shrinkItem(item);
				item.method_24830(false); //Force entity collision update to run every tick if crafting is in progress

				class_1542 outputItem = new class_1542(field_11863, field_11867.method_10263() + 0.5, field_11867.method_10264() + 1.5, field_11867.method_10260() + 0.5, output);
				XplatAbstractions.INSTANCE.itemFlagsComponent(outputItem).manaInfusionSpawned = true;
				if (item.method_24921() instanceof class_1657 player) {
					player.method_51283(recipe, List.of(output));
					output.method_7982(field_11863, player, output.method_7947());
				}
				field_11863.method_8649(outputItem);

				craftingEffect(true);
				return true;
			}
		}

		return false;
	}

	public void craftingEffect(boolean playSound) {
		if (playSound && soundTicks == 0) {
			field_11863.method_8396(null, field_11867, BotaniaSounds.manaPoolCraft, class_3419.field_15245, 1F, 1F);
			soundTicks = 6;
		}

		field_11863.method_33596(null, class_5712.field_28174, method_11016());
		field_11863.method_8427(method_11016(), method_11010().method_26204(), CRAFT_EFFECT_EVENT, 0);
	}

	@Override
	public boolean method_11004(int event, int param) {
		switch (event) {
			case CRAFT_EFFECT_EVENT: {
				if (field_11863.field_9236) {
					for (int i = 0; i < 25; i++) {
						float red = (float) Math.random();
						float green = (float) Math.random();
						float blue = (float) Math.random();
						SparkleParticleData data = SparkleParticleData.sparkle((float) Math.random(), red, green, blue, 10);
						field_11863.method_8406(data, field_11867.method_10263() + 0.5 + Math.random() * 0.4 - 0.2, field_11867.method_10264() + 0.75, field_11867.method_10260() + 0.5 + Math.random() * 0.4 - 0.2, 0, 0, 0);
					}
				}

				return true;
			}
			case CHARGE_EFFECT_EVENT: {
				if (field_11863.field_9236 && BotaniaConfig.common().chargingAnimationEnabled()) {
					chargingParticles.computeIfAbsent(param, i -> new MutableInt(15)).setValue(15);
				}
				return true;
			}
			case DRAIN_EFFECT_EVENT: {
				if (field_11863.field_9236 && BotaniaConfig.common().chargingAnimationEnabled()) {
					drainingParticles.computeIfAbsent(param, i -> new MutableInt(15)).setValue(15);
				}
				return true;
			}
			default:
				return super.method_11004(event, param);
		}
	}

	private void initManaCapAndNetwork() {
		if (getMaxMana() == -1) {
			manaCap = ((ManaPoolBlock) method_11010().method_26204()).variant == ManaPoolBlock.Variant.DILUTED ? MAX_MANA_DILLUTED : MAX_MANA;
		}
		if (!ManaNetworkHandler.instance.isPoolIn(field_11863, this) && !method_11015()) {
			BotaniaAPI.instance().getManaNetworkInstance().fireManaNetworkEvent(this, ManaBlockType.POOL, ManaNetworkAction.ADD);
		}
	}

	public static void clientTick(class_1937 level, class_2338 worldPosition, class_2680 state, ManaPoolBlockEntity self) {
		self.initManaCapAndNetwork();
		double particleChance = 1F - (double) self.getCurrentMana() / (double) self.getMaxMana() * 0.1;
		if (Math.random() > particleChance) {
			WispParticleData data = WispParticleData.wisp((float) Math.random() / 3F,
					PARTICLE_COLOR_RED, PARTICLE_COLOR_GREEN, PARTICLE_COLOR_BLUE, 2F);
			level.method_8406(data, worldPosition.method_10263() + 0.3 + Math.random() * 0.5,
					worldPosition.method_10264() + 0.6 + Math.random() * 0.25, worldPosition.method_10260() + Math.random(),
					0, (float) Math.random() / 25F, 0);
		}

		if (self.getCurrentMana() == 0) {
			self.chargingParticles.clear();
		} else {
			displayChargingParticles(level, worldPosition, self, self.chargingParticles, true);
		}
		displayChargingParticles(level, worldPosition, self, self.drainingParticles, false);
	}

	private static void displayChargingParticles(class_1937 level, class_2338 worldPosition, ManaPoolBlockEntity self,
			Int2ObjectMap<MutableInt> particles, boolean charging) {
		int bellowCount = charging ? getBellowCount(level, worldPosition, self) : 0;
		float relativeMana = (float) self.getCurrentMana() / self.getMaxMana();
		var particlesIterator = particles.int2ObjectEntrySet().iterator();
		while (particlesIterator.hasNext()) {
			var entry = particlesIterator.next();
			int ticksRemaining = entry.getValue().decrementAndGet();
			if (ticksRemaining % 2 == 0) {
				int encodedPos = entry.getIntKey();
				class_243 itemPosRelBase = decodeRelativeItemPosition(encodedPos, relativeMana);
				if (charging) {
					for (int i = 0; i <= bellowCount; i++) {
						class_243 itemPosRel = randomizeItemPos(itemPosRelBase);
						class_243 poolPosRel = new class_243(0.1 + 0.8 * Math.random(), 0.1 + 0.4 * relativeMana,
								0.1 + 0.8 * Math.random());
						addManaFlowParticle(level, worldPosition, poolPosRel, itemPosRel);
					}
				} else {
					class_243 itemPosRel = randomizeItemPos(itemPosRelBase);
					class_243 poolPosRel =
							new class_243(0.05 + 0.9 * Math.random(), 0.35 * relativeMana, 0.05 + 0.9 * Math.random());
					addManaFlowParticle(level, worldPosition, itemPosRel, poolPosRel);
				}
			}
			if (ticksRemaining <= 0) {
				particlesIterator.remove();
			}
		}
	}

	@NotNull
	private static class_243 randomizeItemPos(class_243 itemPosRelBase) {
		return itemPosRelBase.method_1031(0.1 * Math.random() - 0.05, 0.1 * Math.random() + 0.25, 0.1 * Math.random() - 0.05);
	}

	private static int getBellowCount(class_1937 level, class_2338 worldPosition, ManaPoolBlockEntity self) {
		int bellowCount = 0;
		for (class_2350 dir : class_2350.class_2353.field_11062) {
			class_2586 tile = level.method_8321(worldPosition.method_10093(dir));
			if (tile instanceof BellowsBlockEntity bellows && bellows.getLinkedTile() == self) {
				bellowCount++;
			}
		}
		return bellowCount;
	}

	private static void addManaFlowParticle(class_1937 level, class_2338 worldPosition, class_243 startPos, class_243 endPos) {
		double maxHeight = Math.max(startPos.field_1351, endPos.field_1351) - endPos.field_1351 + 0.05 * Math.random();
		class_243 horizontalDiff = new class_243(endPos.field_1352 - startPos.field_1352, 0, endPos.field_1350 - startPos.field_1350);
		double horizontalDistance = horizontalDiff.method_37267();
		class_243 horizontalDir = horizontalDiff.method_1021(1 / horizontalDistance);
		double startHeight = startPos.field_1351 - endPos.field_1351;
		double vY0Squared = 2 * CHARGING_GRAVITY * (maxHeight - startHeight);
		double vY0 = Math.sqrt(vY0Squared);
		double lifetime = (vY0 + Math.sqrt(vY0Squared + 2 * CHARGING_GRAVITY * startHeight)) / CHARGING_GRAVITY;
		double vX0 = horizontalDistance / lifetime;
		class_243 v0 = horizontalDir.method_1021(vX0).method_38499(class_2350.class_2351.field_11052, vY0);

		WispParticleData data = WispParticleData.wisp(0.1f, PARTICLE_COLOR_RED, PARTICLE_COLOR_GREEN,
				PARTICLE_COLOR_BLUE, (float) (0.025 * lifetime), CHARGING_GRAVITY).withNoClip(true);
		level.method_8406(data, worldPosition.method_10263() + startPos.field_1352, worldPosition.method_10264() + startPos.field_1351,
				worldPosition.method_10260() + startPos.field_1350, v0.field_1352, v0.field_1351, v0.field_1350);
	}

	public static void serverTick(class_1937 level, class_2338 worldPosition, class_2680 state, ManaPoolBlockEntity self) {

		// Legacy color format
		if (self.legacyColor.isPresent()) {
			self.setColor(self.legacyColor);
			self.legacyColor = Optional.empty();
		}

		self.initManaCapAndNetwork();
		boolean wasDoingTransfer = self.isDoingTransfer;
		self.isDoingTransfer = false;

		if (self.soundTicks > 0) {
			self.soundTicks--;
		}

		if (self.sendPacket && self.ticks % 10 == 0) {
			VanillaPacketDispatcher.dispatchTEToNearbyPlayers(self);
			self.sendPacket = false;
		}

		List<class_1542> items = level.method_18467(class_1542.class, new class_238(worldPosition, worldPosition.method_10069(1, 1, 1)));
		for (class_1542 item : items) {
			if (!item.method_5805()) {
				continue;
			}

			class_1799 stack = item.method_6983();
			var mana = XplatAbstractions.INSTANCE.findManaItem(stack);
			if (!stack.method_7960() && mana != null) {
				if (self.outputting && mana.canReceiveManaFromPool(self) || !self.outputting && mana.canExportManaToPool(self)) {
					boolean didSomething = false;

					int bellowCount = self.outputting ? getBellowCount(level, worldPosition, self) : 0;
					int transfRate = 1000 * (bellowCount + 1);

					if (self.outputting) {
						if (self.canSpare) {
							if (self.getCurrentMana() > 0 && mana.getMana() < mana.getMaxMana()) {
								didSomething = true;
							}

							int manaVal = Math.min(transfRate, Math.min(self.getCurrentMana(), mana.getMaxMana() - mana.getMana()));
							mana.addMana(manaVal);
							self.receiveMana(-manaVal);
						}
					} else {
						if (self.canAccept) {
							if (mana.getMana() > 0 && !self.isFull()) {
								didSomething = true;
							}

							int manaVal = Math.min(transfRate, Math.min(self.getMaxMana() - self.getCurrentMana(), mana.getMana()));
							if (manaVal == 0 && self.field_11863.method_8320(worldPosition.method_10074()).method_27852(BotaniaBlocks.manaVoid)) {
								manaVal = Math.min(transfRate, mana.getMana());
							}
							mana.addMana(-manaVal);
							self.receiveMana(manaVal);
						}
					}

					if (didSomething) {
						if (BotaniaConfig.common().chargingAnimationEnabled() && self.ticks % 10 == 0) {
							level.method_8427(worldPosition, state.method_26204(),
									self.outputting ? CHARGE_EFFECT_EVENT : DRAIN_EFFECT_EVENT,
									encodeRelativeItemPosition(worldPosition, item));
						}
						EntityHelper.syncItem(item);
						self.isDoingTransfer = self.outputting;
					}
				}
			}
		}

		if (self.isDoingTransfer) {
			self.ticksDoingTransfer++;
		} else {
			self.ticksDoingTransfer = 0;
			if (wasDoingTransfer) {
				VanillaPacketDispatcher.dispatchTEToNearbyPlayers(self);
			}
		}

		self.ticks++;
	}

	/**
	 * Somehow squeeze a relative position within a block cube into 8 bits.
	 * This conversion reserves 2 bits (i.e. four different values) for the vertical position and two times 3 bits
	 * (i.e. eight different values per axis) for the horizontal position. The vertical position is assumed to be either
	 * on the bottom of the pool or at least about halfway up, so the four possible values are distributed accordingly.
	 */
	private static int encodeRelativeItemPosition(class_2338 worldPosition, class_1542 item) {
		double relX = class_3532.method_15350(item.method_19538().method_10216() - worldPosition.method_10263(), 0, 1);
		double relY = class_3532.method_15350(0.125 + 0.875 * (item.method_19538().method_10214() - worldPosition.method_10264()), 0.125, 0.9);
		double relZ = class_3532.method_15350(item.method_19538().method_10215() - worldPosition.method_10260(), 0, 1);

		int compressedX = (int) Math.round(7.0 * relX);
		int compressedY = 4 - class_3532.method_15342(14 - (int) (14.0 * relY));
		int compressedZ = (int) Math.round(7.0 * relZ);

		return compressedX | compressedY << 3 | compressedZ << 5;
	}

	/**
	 * Decodes a position from the parameter that roughly matches the originally encoded one.
	 */
	private static class_243 decodeRelativeItemPosition(int param, float relativeMana) {
		int compressedX = param & 0x7;
		int compressedY = param >> 3 & 0x3;
		int compressedZ = param >> 5 & 0x7;

		double relX = compressedX / 7.0;
		double relY = 1.0 - (14.0 / 16.0) / (1 << compressedY);
		double relZ = compressedZ / 7.0;

		return new class_243(relX, Math.max(relY, 0.5 * relativeMana), relZ);
	}

	@Override
	public void writePacketNBT(class_2487 cmp) {
		cmp.method_10569(TAG_MANA, getCurrentMana());
		cmp.method_10556(TAG_OUTPUTTING, outputting);

		cmp.method_10569(TAG_MANA_CAP, getMaxMana());
		cmp.method_10556(TAG_CAN_ACCEPT, canAccept);
		cmp.method_10556(TAG_CAN_SPARE, canSpare);

		cmp.method_10582(TAG_INPUT_KEY, inputKey);
		cmp.method_10582(TAG_OUTPUT_KEY, outputKey);
	}

	@Override
	public void readPacketNBT(class_2487 cmp) {
		mana = cmp.method_10550(TAG_MANA);
		outputting = cmp.method_10577(TAG_OUTPUTTING);

		// Legacy color format
		if (cmp.method_10545("color")) {
			class_1767 color = class_1767.method_7791(cmp.method_10550("color"));
			// White was previously used as "no color"
			if (color != class_1767.field_7952) {
				legacyColor = Optional.of(color);
			} else {
				legacyColor = Optional.empty();
			}
		}
		if (cmp.method_10545(TAG_MANA_CAP)) {
			manaCap = cmp.method_10550(TAG_MANA_CAP);
		}
		if (cmp.method_10545(TAG_CAN_ACCEPT)) {
			canAccept = cmp.method_10577(TAG_CAN_ACCEPT);
		}
		if (cmp.method_10545(TAG_CAN_SPARE)) {
			canSpare = cmp.method_10577(TAG_CAN_SPARE);
		}

		if (cmp.method_10545(TAG_INPUT_KEY)) {
			inputKey = cmp.method_10558(TAG_INPUT_KEY);
		}
		if (cmp.method_10545(TAG_OUTPUT_KEY)) {
			inputKey = cmp.method_10558(TAG_OUTPUT_KEY);
		}

	}

	@Override
	public boolean onUsedByWand(@Nullable class_1657 player, class_1799 stack, class_2350 side) {
		if (player == null || player.method_5715()) {
			outputting = !outputting;
			VanillaPacketDispatcher.dispatchTEToNearbyPlayers(this);
		}
		return true;
	}

	public static class WandHud implements WandHUD {
		private final ManaPoolBlockEntity pool;

		public WandHud(ManaPoolBlockEntity pool) {
			this.pool = pool;
		}

		@Override
		public void renderHUD(class_332 gui, class_310 mc) {
			class_1799 poolStack = new class_1799(pool.method_11010().method_26204());
			String name = poolStack.method_7964().getString();

			int centerX = mc.method_22683().method_4486() / 2;
			int centerY = mc.method_22683().method_4502() / 2;

			int width = Math.max(102, mc.field_1772.method_1727(name)) + 4;

			RenderHelper.renderHUDBox(gui, centerX - width / 2, centerY + 8, centerX + width / 2, centerY + 48);

			BotaniaAPIClient.instance().drawSimpleManaHUD(gui, 0x0095FF, pool.getCurrentMana(), pool.getMaxMana(), name);

			RenderSystem.enableBlend();
			RenderSystem.blendFunc(GL11.GL_SRC_ALPHA, GL11.GL_ONE_MINUS_SRC_ALPHA);

			int arrowU = pool.outputting ? 22 : 0;
			int arrowV = 38;
			RenderHelper.drawTexturedModalRect(gui, HUDHandler.manaBar, centerX - 11, centerY + 30, arrowU, arrowV, 22, 15);
			RenderSystem.setShaderColor(1F, 1F, 1F, 1F);

			class_1799 tablet = new class_1799(BotaniaItems.manaTablet);
			ManaTabletItem.setStackCreative(tablet);

			gui.method_51427(tablet, centerX - 31, centerY + 30);
			gui.method_51427(poolStack, centerX + 15, centerY + 30);

			RenderSystem.disableBlend();
		}
	}

	@Override
	public boolean canReceiveManaFromBursts() {
		return true;
	}

	@Override
	public boolean isOutputtingPower() {
		return outputting;
	}

	@Override
	public class_1937 getManaReceiverLevel() {
		return method_10997();
	}

	@Override
	public class_2338 getManaReceiverPos() {
		return method_11016();
	}

	@Override
	public int getCurrentMana() {
		if (method_11010().method_26204() instanceof ManaPoolBlock pool) {
			return pool.variant == ManaPoolBlock.Variant.CREATIVE ? MAX_MANA : mana;
		}
		return 0;
	}

	@Override
	public int getMaxMana() {
		return manaCap;
	}

	@Override
	public String getInputKey() {
		return inputKey;
	}

	@Override
	public String getOutputKey() {
		return outputKey;
	}

	@Override
	public boolean canAttachSpark(class_1799 stack) {
		return true;
	}

	@Override
	public ManaSpark getAttachedSpark() {
		List<class_1297> sparks = field_11863.method_8390(class_1297.class, new class_238(field_11867.method_10084(), field_11867.method_10084().method_10069(1, 1, 1)), Predicates.instanceOf(ManaSpark.class));
		if (sparks.size() == 1) {
			class_1297 e = sparks.get(0);
			return (ManaSpark) e;
		}

		return null;
	}

	@Override
	public boolean areIncomingTranfersDone() {
		return false;
	}

	@Override
	public int getAvailableSpaceForMana() {
		int space = Math.max(0, getMaxMana() - getCurrentMana());
		if (space > 0) {
			return space;
		} else if (field_11863.method_8320(field_11867.method_10074()).method_27852(BotaniaBlocks.manaVoid)) {
			return getMaxMana();
		} else {
			return 0;
		}
	}

	@Override
	public Optional<class_1767> getColor() {
		return method_11010().method_11654(OPTIONAL_DYE_COLOR).toDyeColor();
	}

	@Override
	public void setColor(Optional<class_1767> color) {
		field_11863.method_8501(field_11867, method_11010().method_11657(OPTIONAL_DYE_COLOR, BotaniaStateProperties.OptionalDyeColor.fromOptionalDyeColor(color)));
	}

	@Override
	public void markDispatchable() {
		sendPacket = true;
	}
}
