package vazkii.botania.api.configdata;

import com.google.common.collect.ImmutableList;
import com.mojang.serialization.Codec;
import com.mojang.serialization.DataResult;
import com.mojang.serialization.codecs.RecordCodecBuilder;
import vazkii.botania.api.BotaniaAPI;

import java.util.List;
import java.util.Optional;
import java.util.function.Function;
import net.minecraft.class_2960;
import net.minecraft.class_5699;
import net.minecraft.class_6012;
import net.minecraft.class_7061;

public class LooniumStructureConfiguration {
	public static final int DEFAULT_COST = 35000;
	public static final int DEFAULT_MAX_NEARBY_MOBS = 10;
	public static final Codec<LooniumStructureConfiguration> CODEC = class_5699.method_48112(
			RecordCodecBuilder.create(
					instance -> instance.group(
							class_2960.field_25139.optionalFieldOf("parent")
									.forGetter(lsc -> Optional.ofNullable(lsc.parent)),
							class_5699.field_33441.optionalFieldOf("manaCost")
									.forGetter(lsc -> Optional.ofNullable(lsc.manaCost)),
							class_5699.field_33442.optionalFieldOf("maxNearbyMobs")
									.forGetter(lsc -> Optional.ofNullable(lsc.maxNearbyMobs)),
							class_7061.class_7062.field_37202
									.optionalFieldOf("boundingBoxType")
									.forGetter(lsc -> Optional.ofNullable(lsc.boundingBoxType)),
							class_6012.method_34991(LooniumMobSpawnData.CODEC)
									.optionalFieldOf("spawnedMobs")
									.forGetter(lsc -> Optional.ofNullable(lsc.spawnedMobs)),
							Codec.list(LooniumMobAttributeModifier.CODEC)
									.optionalFieldOf("attributeModifiers")
									.forGetter(lsc -> Optional.ofNullable(lsc.attributeModifiers)),
							Codec.list(LooniumMobEffectToApply.CODEC)
									.optionalFieldOf("effectsToApply")
									.forGetter(lsc -> Optional.ofNullable(lsc.effectsToApply))
					).apply(instance, LooniumStructureConfiguration::create)
			), lsc -> {
				if (lsc.parent == null && (lsc.manaCost == null || lsc.boundingBoxType == null || lsc.spawnedMobs == null)) {
					return DataResult.error(() -> "Mana cost, bounding box type, and spawned mobs must be specified if there is no parent configuration");
				}
				if (lsc.spawnedMobs != null && lsc.spawnedMobs.method_34993()) {
					return DataResult.error(() -> "Spawned mobs cannot be empty");
				}
				if (lsc.manaCost != null && lsc.manaCost > DEFAULT_COST) {
					return DataResult.error(() -> "Mana costs above %d are currently not supported"
							.formatted(DEFAULT_COST));
				}
				return DataResult.success(lsc);
			});
	public static final class_2960 DEFAULT_CONFIG_ID = new class_2960(BotaniaAPI.MODID, "default");

	public final Integer manaCost;
	public final Integer maxNearbyMobs;
	public final class_7061.class_7062 boundingBoxType;
	public final class_6012<LooniumMobSpawnData> spawnedMobs;
	public final List<LooniumMobAttributeModifier> attributeModifiers;
	public final List<LooniumMobEffectToApply> effectsToApply;
	public final class_2960 parent;

	private LooniumStructureConfiguration(class_2960 parent, Integer manaCost, Integer maxNearbyMobs,
			class_7061.class_7062 boundingBoxType, class_6012<LooniumMobSpawnData> spawnedMobs,
			List<LooniumMobAttributeModifier> attributeModifiers, List<LooniumMobEffectToApply> effectsToApply) {
		this.manaCost = manaCost;
		this.maxNearbyMobs = maxNearbyMobs;
		this.spawnedMobs = spawnedMobs;
		this.boundingBoxType = boundingBoxType;
		this.attributeModifiers = attributeModifiers != null ? ImmutableList.copyOf(attributeModifiers) : null;
		this.effectsToApply = effectsToApply != null ? ImmutableList.copyOf(effectsToApply) : null;
		this.parent = parent;
	}

	public static Builder builder() {
		return new Builder();
	}

	public static Builder forParent(class_2960 parent) {
		return builder().parent(parent);
	}

	public LooniumStructureConfiguration getEffectiveConfig(
			Function<class_2960, LooniumStructureConfiguration> parentSupplier) {
		if (parent == null) {
			return this;
		}
		var parentConfig = parentSupplier.apply(parent).getEffectiveConfig(parentSupplier);

		return new LooniumStructureConfiguration(null,
				manaCost != null ? manaCost : parentConfig.manaCost,
				maxNearbyMobs != null ? maxNearbyMobs : parentConfig.maxNearbyMobs,
				boundingBoxType != null ? boundingBoxType : parentConfig.boundingBoxType,
				spawnedMobs != null ? spawnedMobs : parentConfig.spawnedMobs,
				attributeModifiers != null ? attributeModifiers : parentConfig.attributeModifiers,
				effectsToApply != null ? effectsToApply : parentConfig.effectsToApply);
	}

	@Override
	public String toString() {
		return "LooniumStructureConfiguration{" +
				"manaCost=" + manaCost +
				", maxNearbyMobs=" + maxNearbyMobs +
				", boundingBoxType=" + boundingBoxType +
				", spawnedMobs=" + (spawnedMobs != null ? spawnedMobs.method_34994() : null) +
				", attributeModifiers=" + attributeModifiers +
				", effectsToApply=" + effectsToApply +
				", parent=" + parent +
				'}';
	}

	// Codecs don't support setting null as intentional default value for optional fields, so we do this.
	// (blame com.mojang.datafixers.util.Either::getLeft using Optional::of instead Optional.ofNullable)
	@SuppressWarnings("OptionalUsedAsFieldOrParameterType")
	private static LooniumStructureConfiguration create(Optional<class_2960> parent,
			Optional<Integer> manaCost, Optional<Integer> maxNearbyMobs,
			Optional<class_7061.class_7062> boundingBoxType,
			Optional<class_6012<LooniumMobSpawnData>> spawnedMobs,
			Optional<List<LooniumMobAttributeModifier>> attributeModifiers,
			Optional<List<LooniumMobEffectToApply>> effectsToApply) {
		return new LooniumStructureConfiguration(
				parent.orElse(null), manaCost.orElse(null), maxNearbyMobs.orElse(null),
				boundingBoxType.orElse(null), spawnedMobs.orElse(null),
				attributeModifiers.orElse(null), effectsToApply.orElse(null));
	}

	public static class Builder {
		private class_2960 parent;
		private Integer manaCost;
		private Integer maxNearbyMobs;
		private class_7061.class_7062 boundingBoxType;
		private class_6012<LooniumMobSpawnData> spawnedMobs;
		private List<LooniumMobAttributeModifier> attributeModifiers;
		private List<LooniumMobEffectToApply> effectsToApply;

		private Builder() {}

		private Builder parent(class_2960 parent) {
			this.parent = parent;
			return this;
		}

		public Builder manaCost(Integer manaCost) {
			this.manaCost = manaCost;
			return this;
		}

		public Builder maxNearbyMobs(Integer maxNearbyMobs) {
			this.maxNearbyMobs = maxNearbyMobs;
			return this;
		}

		public Builder boundingBoxType(class_7061.class_7062 boundingBoxType) {
			this.boundingBoxType = boundingBoxType;
			return this;
		}

		public Builder spawnedMobs(LooniumMobSpawnData... spawnedMobs) {
			this.spawnedMobs = class_6012.method_34989(spawnedMobs);
			return this;
		}

		public Builder attributeModifiers(LooniumMobAttributeModifier... attributeModifiers) {
			this.attributeModifiers = List.of(attributeModifiers);
			return this;
		}

		public Builder effectsToApply(LooniumMobEffectToApply... effectsToApply) {
			this.effectsToApply = List.of(effectsToApply);
			return this;
		}

		public LooniumStructureConfiguration build() {
			return new LooniumStructureConfiguration(parent, manaCost, maxNearbyMobs, boundingBoxType, spawnedMobs,
					attributeModifiers, effectsToApply);
		}
	}
}
