/*
 * This class is distributed as part of the Botania Mod.
 * Get the Source Code in github:
 * https://github.com/Vazkii/Botania
 *
 * Botania is Open Source and distributed under the
 * Botania License: http://botaniamod.net/license.php
 */
package vazkii.botania.common.crafting;

import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParseException;
import com.mojang.serialization.Dynamic;
import com.mojang.serialization.JsonOps;
import org.jetbrains.annotations.Nullable;

import vazkii.botania.api.recipe.StateIngredient;
import vazkii.botania.common.helper.ItemNBTHelper;

import java.util.*;
import net.minecraft.class_2246;
import net.minecraft.class_2248;
import net.minecraft.class_2487;
import net.minecraft.class_2509;
import net.minecraft.class_2512;
import net.minecraft.class_2540;
import net.minecraft.class_2680;
import net.minecraft.class_2960;
import net.minecraft.class_3518;
import net.minecraft.class_6862;
import net.minecraft.class_7923;

public class StateIngredientHelper {
	public static StateIngredient of(class_2248 block) {
		return new BlockStateIngredient(block);
	}

	public static StateIngredient of(class_2680 state) {
		return new BlockStateStateIngredient(state);
	}

	public static StateIngredient of(class_6862<class_2248> tag) {
		return of(tag.comp_327());
	}

	public static StateIngredient of(class_2960 id) {
		return new TagStateIngredient(id);
	}

	public static StateIngredient of(Collection<class_2248> blocks) {
		return new BlocksStateIngredient(blocks);
	}

	// Can't be 'of' because of type erasure.
	public static StateIngredient compound(Collection<StateIngredient> ingredients) {
		return new CompoundStateIngredient(ingredients);
	}

	public static StateIngredient combine(StateIngredient firstIngredient, StateIngredient secondIngredient) {
		List<StateIngredient> ingredients = new ArrayList<>();
		// Flatten the ingredients
		if (firstIngredient instanceof CompoundStateIngredient compound) {
			ingredients.addAll(compound.getIngredients());
		} else {
			ingredients.add(firstIngredient);
		}
		if (secondIngredient instanceof CompoundStateIngredient compound) {
			ingredients.addAll(compound.getIngredients());
		} else {
			ingredients.add(secondIngredient);
		}

		return new CompoundStateIngredient(ingredients);
	}

	public static StateIngredient tagExcluding(class_6862<class_2248> tag, StateIngredient... excluded) {
		return new TagExcludingStateIngredient(tag.comp_327(), List.of(excluded));
	}

	public static StateIngredient deserialize(JsonObject object) {
		switch (class_3518.method_15265(object, "type")) {
			case "tag":
				return new TagStateIngredient(new class_2960(class_3518.method_15265(object, "tag")));
			case "block":
				return new BlockStateIngredient(class_7923.field_41175.method_10223(new class_2960(class_3518.method_15265(object, "block"))));
			case "state":
				return new BlockStateStateIngredient(readBlockState(object));
			case "blocks":
				List<class_2248> blocks = new ArrayList<>();
				for (JsonElement element : class_3518.method_15261(object, "blocks")) {
					blocks.add(class_7923.field_41175.method_10223(new class_2960(element.getAsString())));
				}
				return new BlocksStateIngredient(blocks);
			case "tag_excluding":
				class_2960 tag = new class_2960(class_3518.method_15265(object, "tag"));
				List<StateIngredient> ingr = new ArrayList<>();
				for (JsonElement element : class_3518.method_15261(object, "exclude")) {
					ingr.add(deserialize(class_3518.method_15295(element, "exclude entry")));
				}
				return new TagExcludingStateIngredient(tag, ingr);
			case "compound":
				List<StateIngredient> stateIngredients = new ArrayList<>();
				for (JsonElement ingredient : class_3518.method_15261(object, "ingredients")) {
					if (!ingredient.isJsonObject()) {
						throw new JsonParseException("Unknown ingredient in compound state ingredient: " + ingredient);
					}
					stateIngredients.add(deserialize(ingredient.getAsJsonObject()));
				}
				return new CompoundStateIngredient(stateIngredients);
			default:
				throw new JsonParseException("Unknown type!");
		}
	}

	/**
	 * Deserializes a state ingredient, but removes air from its data,
	 * and returns null if the ingredient only matched air.
	 */
	@Nullable
	public static StateIngredient tryDeserialize(JsonObject object) {
		return clearTheAir(deserialize(object));
	}

	public static StateIngredient clearTheAir(StateIngredient ingredient) {
		if (ingredient instanceof BlockStateIngredient || ingredient instanceof BlockStateStateIngredient) {
			if (ingredient.test(class_2246.field_10124.method_9564())) {
				return null;
			}
		} else if (ingredient instanceof BlocksStateIngredient sib) {
			Collection<class_2248> blocks = sib.blocks;
			List<class_2248> list = new ArrayList<>(blocks);
			if (list.removeIf(b -> b == class_2246.field_10124)) {
				if (list.size() == 0) {
					return null;
				}
				return of(list);
			}
		} else if (ingredient instanceof CompoundStateIngredient sic) {
			List<StateIngredient> newIngredients = sic.getIngredients().stream().map(StateIngredientHelper::clearTheAir).filter(Objects::nonNull).toList();
			if (newIngredients.isEmpty()) {
				return null;
			}
			return compound(newIngredients);
		}
		return ingredient;
	}

	public static StateIngredient read(class_2540 buffer) {
		switch (buffer.method_10816()) {
			case 0:
				int count = buffer.method_10816();
				Set<class_2248> set = new HashSet<>();
				for (int i = 0; i < count; i++) {
					int id = buffer.method_10816();
					class_2248 block = class_7923.field_41175.method_10200(id);
					set.add(block);
				}
				return new BlocksStateIngredient(set);
			case 1:
				return new BlockStateIngredient(class_7923.field_41175.method_10200(buffer.method_10816()));
			case 2:
				return new BlockStateStateIngredient(class_2248.method_9531(buffer.method_10816()));
			case 3:
				int ingredientCount = buffer.method_10816();
				Set<StateIngredient> ingredientSet = new HashSet<>();
				for (int i = 0; i < ingredientCount; i++) {
					ingredientSet.add(read(buffer));
				}
				return new CompoundStateIngredient(ingredientSet);
			default:
				throw new IllegalArgumentException("Unknown input discriminator!");
		}
	}

	/**
	 * Writes data about the block state to the provided json object.
	 */
	public static JsonObject serializeBlockState(class_2680 state) {
		class_2487 nbt = class_2512.method_10686(state);
		ItemNBTHelper.renameTag(nbt, "Name", "name");
		ItemNBTHelper.renameTag(nbt, "Properties", "properties");
		Dynamic<net.minecraft.class_2520> dyn = new Dynamic<>(class_2509.field_11560, nbt);
		return dyn.convert(JsonOps.INSTANCE).getValue().getAsJsonObject();
	}

	/**
	 * Reads the block state from the provided json object.
	 */
	public static class_2680 readBlockState(JsonObject object) {
		class_2487 nbt = (class_2487) new Dynamic<>(JsonOps.INSTANCE, object).convert(class_2509.field_11560).getValue();
		ItemNBTHelper.renameTag(nbt, "name", "Name");
		ItemNBTHelper.renameTag(nbt, "properties", "Properties");
		String name = nbt.method_10558("Name");
		class_2960 id = class_2960.method_12829(name);
		if (id == null || class_7923.field_41175.method_17966(id).isEmpty()) {
			throw new IllegalArgumentException("Invalid or unknown block ID: " + name);
		}
		return class_2512.method_10681(class_7923.field_41175.method_46771(), nbt);
	}
}
