package vazkii.botania.client.integration.shared;

import it.unimi.dsi.fastutil.ints.IntIntImmutablePair;
import it.unimi.dsi.fastutil.ints.IntIntPair;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import vazkii.botania.api.recipe.OrechidRecipe;
import vazkii.botania.common.handler.OrechidManager;

import java.text.NumberFormat;
import java.util.stream.Stream;
import net.minecraft.class_124;
import net.minecraft.class_2338;
import net.minecraft.class_2561;
import net.minecraft.class_2680;
import net.minecraft.class_310;
import net.minecraft.class_3532;
import net.minecraft.class_3956;
import net.minecraft.class_5321;
import net.minecraft.class_638;
import net.minecraft.class_746;

/**
 * Shared helper methods for the various recipe display mod plugins.
 */
public class OrechidUIHelper {
	/**
	 * How far off from the actual chance an approximated outputs/inputs ratio calculated by
	 * {@link #getRatioForChance(double)} should be at most.
	 */
	private static final float MAX_ACCEPTABLE_RATIO_ERROR = 0.05f;

	/**
	 * How many input blocks an approximated outputs/inputs ratio calculated by
	 * {@link #getRatioForChance(double)} should have at most if the
	 * number of outputs in the ratio is greater than 1.
	 */
	private static final int MAX_NUM_INPUTS_FOR_RATIO = 16;

	@NotNull
	public static class_2561 getPercentageComponent(double chance) {
		final String chanceText = LocaleHelper.formatAsPercentage(chance, 1);
		return class_2561.method_43470(chanceText);
	}

	@NotNull
	public static class_2561 getRatioTooltipComponent(@NotNull IntIntPair ratio) {
		final NumberFormat formatter = LocaleHelper.getIntegerFormat();
		return class_2561.method_43469("botaniamisc.conversionRatio",
				formatter.format(ratio.secondInt()),
				formatter.format(ratio.firstInt()));
	}

	@NotNull
	public static class_2561 getBiomeChanceTooltipComponent(double chance, @NotNull String biomeTranslationKey) {
		return class_2561.method_43469("botaniamisc.conversionChanceBiome",
				getPercentageComponent(chance),
				class_2561.method_43471(biomeTranslationKey).method_27692(class_124.field_1056)
		).method_27692(class_124.field_1080);
	}

	@NotNull
	public static Stream<class_2561> getBiomeChanceAndRatioTooltipComponents(double chance, OrechidRecipe recipe) {
		final var biomeTranslationKey = OrechidUIHelper.getPlayerBiomeTranslationKey();
		final var player = class_310.method_1551().field_1724;
		if (biomeTranslationKey == null || player == null) {
			return Stream.empty();
		}

		final var biomeChance = OrechidUIHelper.getChance(recipe, player.method_24515());
		if (biomeChance == null || class_3532.method_20390(chance, biomeChance)) {
			return Stream.empty();
		}

		final var biomeRatio = OrechidUIHelper.getRatioForChance(biomeChance);
		return Stream.of(
				OrechidUIHelper.getBiomeChanceTooltipComponent(biomeChance, biomeTranslationKey),
				class_2561.method_43470("(")
						.method_10852(OrechidUIHelper.getRatioTooltipComponent(biomeRatio))
						.method_27693(")")
						.method_27692(class_124.field_1080)
		);
	}

	@Nullable
	public static <T extends OrechidRecipe> Double getChance(T recipe, @Nullable class_2338 pos) {
		final var level = class_310.method_1551().field_1687;
		if (level == null) {
			return null;
		}
		final var type = recipe.method_17716();
		final var state = recipe.getInput().getDisplayed().get(0);
		final int totalWeight = OrechidManager.getTotalDisplayWeightAt(level, type, state, pos);
		final int weight = pos != null
				? recipe.getWeight(level, pos)
				: recipe.getWeight();
		return (double) weight / totalWeight;
	}

	/**
	 * Determines a "visually pleasing" ratio to be expected between input and output that is not too far off the
	 * precise ratio.
	 *
	 * @param actualRatio The actual ratio.
	 * @return A pair of ints, first int being the number of input blocks, and second int being the expected number of
	 *         output blocks.
	 */
	@NotNull
	public static IntIntPair getRatioForChance(double actualRatio) {
		// First shot: 1 desired output from N input blocks
		int bestNumOutputs = 1;
		int bestNumInputs = (int) Math.round(1 / actualRatio);
		double bestError = calcError(actualRatio, bestNumOutputs, bestNumInputs);

		// Now try to bring the error below an acceptable margin, but only with relatively small integer ratios.
		if (bestNumInputs < MAX_NUM_INPUTS_FOR_RATIO && bestError > MAX_ACCEPTABLE_RATIO_ERROR) {
			// This calculates an approximation for outputs/inputs for the given chance using continued fractions.
			// (also see https://en.wikipedia.org/wiki/Continued_fraction#Infinite_continued_fractions_and_convergents)
			int numOutputsNminus1 = 1;
			int numOutputsNminus2 = 0;
			int numInputsNminus1 = 0;
			int numInputsNminus2 = 1;
			double remainderN = actualRatio;
			do {
				int coefficientN = (int) Math.floor(remainderN);
				int numOutputsN = coefficientN * numOutputsNminus1 + numOutputsNminus2;
				int numInputsN = coefficientN * numInputsNminus1 + numInputsNminus2;

				if (numInputsN > MAX_NUM_INPUTS_FOR_RATIO) {
					// numbers are getting too big
					break;
				}

				final double errorN = calcError(actualRatio, numOutputsN, numInputsN);
				if (errorN < bestError) {
					bestNumOutputs = numOutputsN;
					bestNumInputs = numInputsN;
					bestError = errorN;
				}

				// shift values for next iteration
				numOutputsNminus2 = numOutputsNminus1;
				numOutputsNminus1 = numOutputsN;
				numInputsNminus2 = numInputsNminus1;
				numInputsNminus1 = numInputsN;
				remainderN = 1 / (remainderN - coefficientN);

			} while (numInputsNminus1 != 0 && bestError > MAX_ACCEPTABLE_RATIO_ERROR);
		}

		return IntIntImmutablePair.of(bestNumInputs, bestNumOutputs);
	}

	private static double calcError(double chance, int numOutputs, int numInputs) {
		return Math.abs((double) numOutputs / numInputs - chance) / chance;
	}

	public static String getPlayerBiomeTranslationKey() {
		final var player = class_310.method_1551().field_1724;
		if (player == null) {
			return null;
		}
		final var biomeKey = player.method_37908().method_23753(player.method_24515()).method_40230().orElse(null);
		if (biomeKey == null) {
			return "argument.id.invalid";
		}
		return String.format("biome.%s.%s", biomeKey.method_29177().method_12836(), biomeKey.method_29177().method_12832());
	}
}
