/*
 * GNU Lesser General Public License v3
 * Copyright (C) 2024 Tschipp
 * mrtschipp@gmail.com
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

package tschipp.carryon.common.config;

import tschipp.carryon.Constants;
import tschipp.carryon.utils.StringHelper;

import java.util.*;
import java.util.stream.Collectors;
import net.minecraft.class_1297;
import net.minecraft.class_1299;
import net.minecraft.class_2248;
import net.minecraft.class_2378;
import net.minecraft.class_2769;
import net.minecraft.class_2960;
import net.minecraft.class_6862;
import net.minecraft.class_7923;

public class ListHandler {

    private static Set<String> FORBIDDEN_TILES = new HashSet<>();
    private static Set<String> FORBIDDEN_ENTITIES = new HashSet<>();
    private static Set<String> ALLOWED_ENTITIES = new HashSet<>();
    private static Set<String> ALLOWED_TILES = new HashSet<>();
    private static Set<String> FORBIDDEN_STACKING = new HashSet<>();
    private static Set<String> ALLOWED_STACKING = new HashSet<>();

    private static List<class_6862<class_2248>> FORBIDDEN_TILES_TAGS = new ArrayList<>();
    private static List<class_6862<class_1299<?>>> FORBIDDEN_ENTITIES_TAGS = new ArrayList<>();
    private static List<class_6862<class_1299<?>>> ALLOWED_ENTITIES_TAGS = new ArrayList<>();
    private static List<class_6862<class_2248>> ALLOWED_TILES_TAGS = new ArrayList<>();
    private static List<class_6862<class_1299<?>>> FORBIDDEN_STACKING_TAGS = new ArrayList<>();
    private static List<class_6862<class_1299<?>>> ALLOWED_STACKING_TAGS = new ArrayList<>();

    private static Set<class_2769<?>> PROPERTY_EXCEPTION_CLASSES = new HashSet<>();

    public static boolean isPermitted(class_2248 block)
    {
        if(Constants.COMMON_CONFIG.settings.useWhitelistBlocks)
            return doCheck(block, ALLOWED_TILES, ALLOWED_TILES_TAGS);
        else
            return !doCheck(block, FORBIDDEN_TILES, FORBIDDEN_TILES_TAGS);
    }

    public static boolean isPermitted(class_1297 entity)
    {
        if(Constants.COMMON_CONFIG.settings.useWhitelistEntities)
            return doCheck(entity, ALLOWED_ENTITIES, ALLOWED_ENTITIES_TAGS);
        else
            return !doCheck(entity, FORBIDDEN_ENTITIES, FORBIDDEN_ENTITIES_TAGS);
    }

    public static boolean isStackingPermitted(class_1297 entity)
    {
        if(Constants.COMMON_CONFIG.settings.useWhitelistStacking)
            return doCheck(entity, ALLOWED_STACKING, ALLOWED_STACKING_TAGS);
        else
            return !doCheck(entity, FORBIDDEN_STACKING, FORBIDDEN_STACKING_TAGS);
    }

    public static boolean isPropertyException(class_2769<?> prop)
    {
        return PROPERTY_EXCEPTION_CLASSES.contains(prop);
    }

    private static boolean doCheck(class_2248 block, Set<String> regular, List<class_6862<class_2248>> tags)
    {
        String name = class_7923.field_41175.method_10221(block).toString();
        if(regular.contains(name))
            return true;
        for(class_6862<class_2248> tag : tags)
            if(block.method_9564().method_26164(tag))
                return true;
        return false;
    }

    private static boolean doCheck(class_1297 entity, Set<String> regular, List<class_6862<class_1299<?>>> tags)
    {
        String name = class_7923.field_41177.method_10221(entity.method_5864()).toString();
        if(regular.contains(name))
            return true;
        for(class_6862<class_1299<?>> tag : tags)
            if(entity.method_5864().method_20210(tag))
                return true;
        return false;
    }

    public static void initConfigLists()
    {
        FORBIDDEN_ENTITIES.clear();
        FORBIDDEN_ENTITIES_TAGS.clear();
        FORBIDDEN_STACKING.clear();
        FORBIDDEN_STACKING_TAGS.clear();
        FORBIDDEN_TILES.clear();
        FORBIDDEN_TILES_TAGS.clear();
        ALLOWED_ENTITIES.clear();
        ALLOWED_ENTITIES_TAGS.clear();
        ALLOWED_STACKING.clear();
        ALLOWED_STACKING_TAGS.clear();
        ALLOWED_TILES.clear();
        ALLOWED_TILES_TAGS.clear();
        PROPERTY_EXCEPTION_CLASSES.clear();

        Map<class_2960, class_6862<class_2248>> blocktags = class_7923.field_41175.method_40273().collect(Collectors.toMap(t -> t.comp_327(), t -> t));
        Map<class_2960, class_6862<class_1299<?>>> entitytags = class_7923.field_41177.method_40273().collect(Collectors.toMap(t -> t.comp_327(), t -> t));

        List<String> forbidden = new ArrayList<>(List.of(Constants.COMMON_CONFIG.blacklist.forbiddenTiles));
        forbidden.add("#carryon:block_blacklist");
        addWithWildcards(forbidden, FORBIDDEN_TILES, class_7923.field_41175, blocktags, FORBIDDEN_TILES_TAGS);

        List<String> forbiddenEntity = new ArrayList<>(List.of(Constants.COMMON_CONFIG.blacklist.forbiddenEntities));
        forbiddenEntity.add("#carryon:entity_blacklist");
        addWithWildcards(forbiddenEntity, FORBIDDEN_ENTITIES, class_7923.field_41177, entitytags, FORBIDDEN_ENTITIES_TAGS);

        List<String> allowedEntities = new ArrayList<>(List.of(Constants.COMMON_CONFIG.whitelist.allowedEntities));
        allowedEntities.add("#carryon:entity_whitelist");
        addWithWildcards(allowedEntities, ALLOWED_ENTITIES, class_7923.field_41177, entitytags, ALLOWED_ENTITIES_TAGS);

        List<String> allowedBlocks = new ArrayList<>(List.of(Constants.COMMON_CONFIG.whitelist.allowedBlocks));
        allowedBlocks.add("#carryon:block_whitelist");
        addWithWildcards(allowedBlocks, ALLOWED_TILES, class_7923.field_41175, blocktags, ALLOWED_TILES_TAGS);

        List<String> forbiddenStacking = new ArrayList<>(List.of(Constants.COMMON_CONFIG.blacklist.forbiddenStacking));
        forbiddenStacking.add("#carryon:stacking_blacklist");
        addWithWildcards(forbiddenStacking, FORBIDDEN_STACKING, class_7923.field_41177, entitytags, FORBIDDEN_STACKING_TAGS);

        List<String> allowedStacking = new ArrayList<>(List.of(Constants.COMMON_CONFIG.whitelist.allowedStacking));
        allowedStacking.add("#carryon:stacking_whitelist");
        addWithWildcards(allowedStacking, ALLOWED_STACKING, class_7923.field_41177, entitytags, ALLOWED_STACKING_TAGS);

        for(String propString : Constants.COMMON_CONFIG.settings.placementStateExceptions)
        {
            if(!propString.contains("[") || !propString.contains("]"))
                continue;
            String name = propString.substring(0, propString.indexOf("["));
            String props = propString.substring(propString.indexOf("[") + 1, propString.indexOf("]"));
            class_2248 blk = class_7923.field_41175.method_10223(class_2960.method_60654(name));
            for(String propName : props.split(",")) {
                for (class_2769<?> prop : blk.method_9564().method_28501()) {
                    if (prop.method_11899().equals(propName))
                        PROPERTY_EXCEPTION_CLASSES.add(prop);
                }
            }
        }
    }

    private static <T> void addTag(String tag, Map<class_2960, class_6862<T>> tagMap, List<class_6862<T>> tags) {
        String sub = tag.substring(1);
        class_6862<T> t = tagMap.get(class_2960.method_60654(sub));
        if (t != null)
            tags.add(t);
    }

    private static <T> void addWithWildcards(List<String> entries, Set<String> toAddTo, class_2378<T> registry, Map<class_2960, class_6862<T>> tags, List<class_6862<T>> toAddTags) {

        class_2960[] keys = registry.method_10235().toArray(new class_2960[0]);
        for (int i = 0; i < entries.size(); i++)
        {
            String curr = entries.get(i);
            if (!curr.startsWith("#"))
            {
                if (curr.contains("*"))
                {
                    String[] filter = curr.replace("*", ",").split(",");

                    for (class_2960 key : keys)
                    {

                        if (containsAll(key.toString(), filter))
                        {
                            toAddTo.add(key.toString());
                        }
                    }
                }
                else
                    toAddTo.add(curr);
            }
            else
                addTag(curr, tags, toAddTags);
        }
    }

    public static boolean containsAll(String str, String... strings)
    {
        return StringHelper.matchesWildcards(str, strings);
    }

    public static void addForbiddenTiles(String toAdd)
    {
        FORBIDDEN_TILES.add(toAdd);
    }

    public static void addForbiddenEntities(String toAdd)
    {
        FORBIDDEN_ENTITIES.add(toAdd);
    }

    public static void addForbiddenStacking(String toAdd)
    {
        FORBIDDEN_STACKING.add(toAdd);
    }

    public static void addAllowedTiles(String toAdd)
    {
        ALLOWED_TILES.add(toAdd);
    }

    public static void addAllowedEntities(String toAdd)
    {
        ALLOWED_ENTITIES.add(toAdd);
    }

    public static void addAllowedStacking(String toAdd)
    {
        ALLOWED_ENTITIES.add(toAdd);
    }

}
