/*
 * GNU Lesser General Public License v3
 * Copyright (C) 2024 Tschipp
 * mrtschipp@gmail.com
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

package tschipp.carryon.common.carry;

import com.mojang.serialization.DataResult;
import tschipp.carryon.Constants;
import tschipp.carryon.common.scripting.CarryOnScript;

import javax.annotation.Nullable;
import net.minecraft.class_1295;
import net.minecraft.class_1297;
import net.minecraft.class_1299;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_2487;
import net.minecraft.class_2509;
import net.minecraft.class_2512;
import net.minecraft.class_2520;
import net.minecraft.class_2586;
import net.minecraft.class_2680;
import net.minecraft.class_2741;
import net.minecraft.class_7225;
import net.minecraft.class_7923;
import java.util.Optional;

public class CarryOnData {

    private CarryType type;
    private class_2487 nbt;
    private boolean keyPressed = false;
    private CarryOnScript activeScript;
    private int selectedSlot = 0;

    public CarryOnData(class_2487 data)
    {
        if(data.method_10545("type"))
            this.type = CarryType.valueOf(data.method_10558("type"));
        else
            this.type = CarryType.INVALID;

        this.nbt = data;

        if(data.method_10545("keyPressed"))
            this.keyPressed = data.method_10577("keyPressed");

        if(data.method_10545("activeScript"))
        {
            DataResult<CarryOnScript> res = CarryOnScript.CODEC.parse(class_2509.field_11560, data.method_10580("activeScript"));
            this.activeScript = res.getOrThrow((s) -> {throw new RuntimeException("Failed to decode activeScript during CarryOnData serialization: " + s);});
        }

        if(data.method_10545("selected"))
            this.selectedSlot = data.method_10550("selected");

    }

    public class_2487 getNbt()
    {
        nbt.method_10582("type", type.toString());
        nbt.method_10556("keyPressed", keyPressed);
        if(activeScript != null)
        {
            DataResult<class_2520> res = CarryOnScript.CODEC.encodeStart(class_2509.field_11560, activeScript);
            class_2520 tag = res.getOrThrow((s) -> {throw new RuntimeException("Failed to encode activeScript during CarryOnData serialization: " + s);});
            nbt.method_10566("activeScript", tag);
        }
        nbt.method_10569("selected", this.selectedSlot);
        return nbt;
    }

    public class_2487 getContentNbt()
    {
        if(type == CarryType.BLOCK && nbt.method_10545("block"))
            return nbt.method_10562("block");
        else if(type == CarryType.ENTITY && nbt.method_10545("entity"))
            return nbt.method_10562("entity");
        return null;
    }

    public void setBlock(class_2680 state, @Nullable class_2586 tile)
    {
        this.type = CarryType.BLOCK;

        if(state.method_28498(class_2741.field_12508))
            state = state.method_11657(class_2741.field_12508, false);

        class_2487 stateData = class_2512.method_10686(state);
        nbt.method_10566("block", stateData);

        if(tile != null)
        {
            class_2487 tileData = tile.method_38243(tile.method_10997().method_30349());
            nbt.method_10566("tile", tileData);
        }
    }

    public class_2680 getBlock()
    {
        if(this.type != CarryType.BLOCK)
            throw new IllegalStateException("Called getBlock on data that contained " + this.type);

        return class_2512.method_10681(class_7923.field_41175.method_46771(), nbt.method_10562("block"));
    }

    @Nullable
    public class_2586 getBlockEntity(class_2338 pos, class_7225.class_7874 lookup)
    {
        if(this.type != CarryType.BLOCK)
            throw new IllegalStateException("Called getBlockEntity on data that contained " + this.type);

        if(!nbt.method_10545("tile"))
            return null;

        return class_2586.method_11005(pos, this.getBlock(), nbt.method_10562("tile"), lookup);
    }

    public void setEntity(class_1297 entity)
    {
        this.type = CarryType.ENTITY;
        class_2487 entityData = new class_2487();
        entity.method_5662(entityData);
        nbt.method_10566("entity", entityData);
    }

    public class_1297 getEntity(class_1937 level)
    {
        if(this.type != CarryType.ENTITY)
            throw new IllegalStateException("Called getEntity on data that contained " + this.type);

        var optionalEntity = class_1299.method_5892(nbt.method_10562("entity"), level);
        if(optionalEntity.isPresent())
            return optionalEntity.get();

        Constants.LOG.error("Called EntityType#create even though no entity data was present. Data: " + nbt.toString());
        this.clear();
        return new class_1295(level, 0, 0, 0);
    }

    public Optional<CarryOnScript> getActiveScript()
    {
        if(activeScript == null)
            return Optional.empty();
        return Optional.of(activeScript);
    }

    public void setActiveScript(CarryOnScript script)
    {
        this.activeScript = script;
    }

    public void setCarryingPlayer() {
        this.type = CarryType.PLAYER;
    }

    public boolean isCarrying()
    {
        return this.type != CarryType.INVALID;
    }

    public boolean isCarrying(CarryType type)
    {
        return this.type == type;
    }

    public boolean isKeyPressed() {return this.keyPressed;}

    public void setKeyPressed(boolean val) {
        this.keyPressed = val;
        this.nbt.method_10556("keyPressed", val);
    }

    public void setSelected(int selectedSlot) {
        this.selectedSlot = selectedSlot;
    }

    public int getSelected() {
        return this.selectedSlot;
    }

    public void clear()
    {
        this.type = CarryType.INVALID;
        this.nbt = new class_2487();
        this.activeScript = null;
    }

    public CarryOnData clone() {
        return new CarryOnData(nbt.method_10553());
    }

    public int getTick()
    {
        if(!this.nbt.method_10545("tick"))
            return -1;
        return this.nbt.method_10550("tick");
    }

    public enum CarryType {
        BLOCK,
        ENTITY,
        PLAYER,
        INVALID
    }
}
