package rearth.oritech.api.recipe.util;

import com.google.common.base.Optional;
import com.google.common.base.Predicates;
import com.google.common.collect.Iterables;
import rearth.oritech.Oritech;
import rearth.oritech.api.recipe.*;
import rearth.oritech.init.FluidContent;

import java.util.Arrays;
import java.util.List;
import net.minecraft.class_1792;
import net.minecraft.class_1856;
import net.minecraft.class_1935;
import net.minecraft.class_3612;
import net.minecraft.class_6862;
import net.minecraft.class_7800;
import net.minecraft.class_8790;

public class MetalProcessingChainBuilder {
    private String metalName;
    private String resourcePath = "";
    // ingredient should generally be used for recipe inputs and item for recipe output
    // wherever possible, use ConventionalItemTags (Fabric) or Tags.Items (Neoforge) for ingredients
    private class_1856 ore;
    private class_1856 rawOreIngredient;
    private class_1792 rawOreItem;
    // should be a raw ore, secondary raw ore given when grinding ore blocks
    private class_1792 rawOreByproduct;
    private class_1856 ingotIngredient;
    private class_1792 ingotItem;
    private class_1856 nuggetIngredient;
    private class_1792 nuggetItem;
    private class_1856 clumpIngredient;
    private class_1792 clumpItem;
    private class_1792 smallClumpItem;
    private class_1792 dustItem;
    private class_1792 smallDustItem;
    private class_1792 centrifugeResult;
    private int centrifugeAmount;
    // usually a small dust (or nugget) given as a byproduct from the grinder or centrifuge
    private class_1792 dustByproduct;
    private class_1792 clumpByproduct;
    private int byproductAmount = 3;
    private class_1856 gemIngredient;
    private class_1792 gemItem;
    private class_1856 gemCatalyst;
    private float timeMultiplier = 1f;
    // for compat use. no need to add vanilla processing for other mods' ores
    private boolean vanillaProcessing = false;
    private boolean skipCompactingRecipes = false;
    
    private MetalProcessingChainBuilder(String metalName) {
        this.metalName = metalName;
    }
    
    public static MetalProcessingChainBuilder build(String metalName) {
        return new MetalProcessingChainBuilder(metalName);
    }
    
    public MetalProcessingChainBuilder resourcePath(String resourcePath) {
        this.resourcePath = resourcePath;
        return this;
    }
    
    public MetalProcessingChainBuilder ore(class_1856 ore) {
        this.ore = ore;
        return this;
    }
    
    public MetalProcessingChainBuilder ore(class_6862<class_1792> oreTag) {
        return ore(class_1856.method_8106(oreTag));
    }
    
    public MetalProcessingChainBuilder ore(class_1935 ore) {
        return ore(class_1856.method_8091(ore));
    }
    
    public MetalProcessingChainBuilder rawOre(class_1856 rawOreIngredient, class_1792 rawOre) {
        this.rawOreIngredient = rawOreIngredient;
        this.rawOreItem = rawOre;
        return this;
    }
    
    public MetalProcessingChainBuilder rawOre(class_6862<class_1792> rawOreTag, class_1792 rawOre) {
        return rawOre(class_1856.method_8106(rawOreTag), rawOre);
    }
    
    public MetalProcessingChainBuilder rawOre(class_1792 rawOre) {
        return rawOre(class_1856.method_8091(rawOre), rawOre);
    }
    
    public MetalProcessingChainBuilder rawOreByproduct(class_1792 byproduct) {
        this.rawOreByproduct = byproduct;
        return this;
    }
    
    public MetalProcessingChainBuilder ingot(class_1856 ingotIngredient, class_1792 ingot) {
        this.ingotIngredient = ingotIngredient;
        this.ingotItem = ingot;
        return this;
    }
    
    public MetalProcessingChainBuilder ingot(class_6862<class_1792> ingotTag, class_1792 ingot) {
        return ingot(class_1856.method_8106(ingotTag), ingot);
    }
    
    public MetalProcessingChainBuilder ingot(class_1792 ingot) {
        return ingot(class_1856.method_8091(ingot), ingot);
    }
    
    public MetalProcessingChainBuilder nugget(class_1856 nuggetIngredient, class_1792 nugget) {
        this.nuggetIngredient = nuggetIngredient;
        this.nuggetItem = nugget;
        return this;
    }
    
    public MetalProcessingChainBuilder nugget(class_6862<class_1792> nuggetTag, class_1792 nugget) {
        return nugget(class_1856.method_8106(nuggetTag), nugget);
    }
    
    public MetalProcessingChainBuilder nugget(class_1792 nugget) {
        return nugget(class_1856.method_8091(nugget), nugget);
    }
    
    public MetalProcessingChainBuilder clump(class_1856 clumpIngredient, class_1792 clump) {
        this.clumpIngredient = clumpIngredient;
        this.clumpItem = clump;
        return this;
    }
    
    public MetalProcessingChainBuilder clump(class_6862<class_1792> clumpTag, class_1792 clump) {
        return clump(class_1856.method_8106(clumpTag), clump);
    }
    
    public MetalProcessingChainBuilder clump(class_1792 clump) {
        return clump(class_1856.method_8091(clump), clump);
    }
    
    public MetalProcessingChainBuilder smallClump(class_1792 smallClump) {
        this.smallClumpItem = smallClump;
        return this;
    }
    
    public MetalProcessingChainBuilder centrifugeResult(class_1792 result, int amount) {
        this.centrifugeResult = result;
        this.centrifugeAmount = amount;
        return this;
    }
    
    public MetalProcessingChainBuilder centrifugeResult(class_1792 result) {
        return centrifugeResult(result, 1);
    }
    
    public MetalProcessingChainBuilder clumpByproduct(class_1792 byproduct) {
        this.clumpByproduct = byproduct;
        return this;
    }
    
    public MetalProcessingChainBuilder dustByproduct(class_1792 byproduct) {
        this.dustByproduct = byproduct;
        return this;
    }
    
    public MetalProcessingChainBuilder byproductAmount(int amount) {
        this.byproductAmount = amount;
        return this;
    }
    
    public MetalProcessingChainBuilder dust(class_1792 dust) {
        this.dustItem = dust;
        return this;
    }
    
    public MetalProcessingChainBuilder smallDust(class_1792 smallDust) {
        this.smallDustItem = smallDust;
        return this;
    }
    
    public MetalProcessingChainBuilder gem(class_1856 gemIngredient, class_1792 gem) {
        this.gemIngredient = gemIngredient;
        this.gemItem = gem;
        return this;
    }
    
    public MetalProcessingChainBuilder gem(class_6862<class_1792> gemTag, class_1792 gem) {
        return gem(class_1856.method_8106(gemTag), gem);
    }
    
    public MetalProcessingChainBuilder gem(class_1792 gem) {
        return gem(class_1856.method_8091(gem), gem);
    }
    
    public MetalProcessingChainBuilder gemCatalyst(class_1856 gemCatalyst) {
        this.gemCatalyst = gemCatalyst;
        return this;
    }
    
    public MetalProcessingChainBuilder gemCatalyst(class_6862<class_1792> gemCatalyst) {
        return gemCatalyst(class_1856.method_8106(gemCatalyst));
    }
    
    public MetalProcessingChainBuilder gemCatalyst(class_1792 gemCatalyst) {
        return gemCatalyst(class_1856.method_8091(gemCatalyst));
    }
    
    public MetalProcessingChainBuilder timeMultiplier(float timeMultiplier) {
        this.timeMultiplier = timeMultiplier;
        return this;
    }
    
    public MetalProcessingChainBuilder vanillaProcessing() {
        this.vanillaProcessing = true;
        return this;
    }
    
    public MetalProcessingChainBuilder skipCompacting() {
        this.skipCompactingRecipes = true;
        return this;
    }
    
    private void validate(String path) throws IllegalStateException {
        if (ore == null)
            throw new IllegalStateException("ore is required for metal processing chain " + path);
        if (rawOreItem == null)
            throw new IllegalStateException("raw ore is required for metal processing chain " + path);
        if ((dustItem != null || vanillaProcessing == true) && ingotItem == null)
            throw new IllegalStateException("ingot is required if dust is provided or vanilla processing is required for metal processing chain " + path);
        if ((smallClumpItem != null || smallDustItem != null) && nuggetItem == null)
            throw new IllegalStateException("nugget item is required if small clump or small dust are provided for metal processing chain " + path);
        if (centrifugeResult != null && centrifugeAmount < 1)
            throw new IllegalStateException("centrifugeAmount must be >= 1 if centrifugeOutput is provided for metal processing chain " + path);
        if (clumpItem != null && (centrifugeResult == null && gemItem == null))
            throw new IllegalStateException("either centrifugeResult or gemItem is required if clump is provided for metal processing chain " + path);
    }
    
    public void export(class_8790 exporter) {
        validate(resourcePath + "ore/" + metalName);
        
        // ore block -> raw ores
        PulverizerRecipeBuilder.build().input(ore).result(rawOreItem, 2).timeMultiplier(timeMultiplier).export(exporter, resourcePath + "ore/" + metalName);
        var grinderOreRecipe = GrinderRecipeBuilder.build().input(ore).result(rawOreItem, 2).time(140).timeMultiplier(timeMultiplier);
        if (rawOreByproduct != null)
            grinderOreRecipe.result(rawOreByproduct);
        grinderOreRecipe.export(exporter, resourcePath + "ore/" + metalName);
        
        // raw ores -> dusts in pulverizer
        if (dustItem != null) {
            PulverizerRecipeBuilder.build()
              .input(rawOreIngredient)
              .result(dustItem)
              .result(firstNonNullOptional(smallDustItem, nuggetItem), 3)
              .timeMultiplier(timeMultiplier)
              .export(exporter, resourcePath + "raw/" + metalName);
        }
        
        // raw ores -> clumps (falling back to dusts) in grinder
        if (clumpItem != null || dustItem != null) {
            GrinderRecipeBuilder.build()
              .input(rawOreIngredient)
              .result(firstNonNull(clumpItem, dustItem))
              .result(firstNonNullOptional(smallClumpItem, smallDustItem, nuggetItem), 3)
              .result(Optional.fromNullable(clumpByproduct), byproductAmount)
              .time(140).timeMultiplier(timeMultiplier)
              .export(exporter, resourcePath + "raw/" + metalName);
        }
        
        // raw ores -> clumps (falling back to dusts) in refinery with sheol fire
        if (clumpItem != null || dustItem != null) {
            RefineryRecipeBuilder.build()
              .input(rawOreIngredient)
              .fluidInput(FluidContent.STILL_SHEOL_FIRE.get(), 0.25f)
              .result(firstNonNull(clumpItem, dustItem), 2)
              .fluidOutput(class_3612.field_15908, 0.1f)
              .timeMultiplier(timeMultiplier)
              .export(exporter, resourcePath + "rawsheol/" + metalName);
        }
        
        // clump processing into gems in centrifuge
        if (clumpItem != null) {
            // dry variant
            CentrifugeRecipeBuilder.build()
              .input(clumpIngredient)
              .result(firstNonNull(centrifugeResult, gemItem), centrifugeResult != null ? centrifugeAmount : 1)
              .result(Optional.fromNullable(dustByproduct), byproductAmount)
              .timeMultiplier(timeMultiplier)
              .export(exporter, resourcePath + "clump/" + metalName);
            // water washed
            CentrifugeFluidRecipeBuilder.build()
              .input(clumpIngredient)
              .fluidInput(class_3612.field_15910)
              .result(firstNonNull(centrifugeResult, gemItem), centrifugeResult != null ? centrifugeAmount * 2 : 2)
              .timeMultiplier(timeMultiplier * 1.5f)
              .export(exporter, resourcePath + "clump/" + metalName);
            // sulfuric acid washing
            CentrifugeFluidRecipeBuilder.build()
              .input(clumpIngredient)
              .fluidInput(FluidContent.STILL_SULFURIC_ACID.get())
              .result(firstNonNull(centrifugeResult, gemItem), centrifugeResult != null ? centrifugeAmount * 3 : 3)
              .fluidOutput(FluidContent.STILL_MINERAL_SLURRY.get(), 0.25f)
              .timeMultiplier(timeMultiplier * 1.5f)
              .export(exporter, resourcePath + "clumpacid/" + metalName);
        }
        
        // gems to dust (doubling)
        if (gemIngredient != null) {
            // atomic forge: 1 gem -> 2 ingots
            AtomicForgeRecipeBuilder.build().input(gemIngredient).input(gemCatalyst).input(gemCatalyst).result(dustItem, 2).time(20).export(exporter, resourcePath + "dust/" + metalName);
            
            // foundry alternative: 2 gems -> 3 ingots
            FoundryRecipeBuilder.build().input(gemIngredient).input(gemIngredient).result(ingotItem, 3).export(exporter, resourcePath + "gem/" + metalName);
        }
        
        // ingots/nuggets to dust
        if (dustItem != null)
            RecipeHelpers.addDustRecipe(exporter, ingotIngredient, dustItem, resourcePath + "dust/" + metalName);
        if (smallDustItem != null)
            RecipeHelpers.addDustRecipe(exporter, nuggetIngredient, smallDustItem, resourcePath + "smalldust/" + metalName);
        
        // smelting/compacting
        // Using item instead of ingredient for recipe inputs, as that's what the offerSmelting/offerBlasting methods accept
        // This should be fine, because any mod that adds ores, dusts, etc. will provide their own smelting/blasting recipes
        if (vanillaProcessing) {
            if (dustItem != null) {
                OritechRecipeGenerator.method_36233(exporter, List.of(dustItem), class_7800.field_40642, ingotItem, 1f, 200, Oritech.MOD_ID);
                OritechRecipeGenerator.method_36234(exporter, List.of(dustItem), class_7800.field_40642, ingotItem, 1f, 100, Oritech.MOD_ID);
                OritechRecipeGenerator.method_47522(exporter, class_7800.field_40642, dustItem, smallDustItem);
            }
            if (smallDustItem != null) {
                OritechRecipeGenerator.method_36233(exporter, List.of(smallDustItem), class_7800.field_40642, nuggetItem, 0.5f, 50, Oritech.MOD_ID);
                OritechRecipeGenerator.method_36234(exporter, List.of(smallDustItem), class_7800.field_40642, nuggetItem, 0.5f, 25, Oritech.MOD_ID);
            }
            if (gemItem != null) {
                OritechRecipeGenerator.method_36233(exporter, List.of(gemItem), class_7800.field_40642, ingotItem, 1f, 200, Oritech.MOD_ID);
                OritechRecipeGenerator.method_36234(exporter, List.of(gemItem), class_7800.field_40642, ingotItem, 1f, 100, Oritech.MOD_ID);
            }
            if (clumpItem != null && smallClumpItem != null)
                OritechRecipeGenerator.method_47522(exporter, class_7800.field_40642, clumpItem, smallClumpItem);
            if (nuggetItem != null && !skipCompactingRecipes)    // to avoid duplicate vanilla nugget -> item recipes
                OritechRecipeGenerator.method_47522(exporter, class_7800.field_40642, ingotItem, nuggetItem);
        }
    }
    
    private class_1792 firstNonNull(class_1792... items) {
        return Iterables.find(Arrays.asList(items), Predicates.notNull());
    }
    
    private Optional<class_1792> firstNonNullOptional(class_1792... items) {
        return Iterables.tryFind(Arrays.asList(items), Predicates.notNull());
    }
}
