package rearth.oritech.block.base.entity;

import rearth.oritech.Oritech;
import rearth.oritech.api.networking.NetworkedBlockEntity;
import rearth.oritech.api.networking.SyncField;
import rearth.oritech.api.networking.SyncType;
import rearth.oritech.block.base.block.FrameInteractionBlock;
import rearth.oritech.client.init.ParticleContent;
import rearth.oritech.init.BlockContent;
import rearth.oritech.util.Geometry;

import java.util.HashMap;
import java.util.Objects;
import net.minecraft.class_1799;
import net.minecraft.class_1937;
import net.minecraft.class_2246;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2382;
import net.minecraft.class_243;
import net.minecraft.class_2487;
import net.minecraft.class_2591;
import net.minecraft.class_2680;
import net.minecraft.class_2741;
import net.minecraft.class_3545;
import net.minecraft.class_7225;

import static rearth.oritech.util.Geometry.*;


public abstract class FrameInteractionBlockEntity extends NetworkedBlockEntity {
    
    private static final int MAX_SEARCH_LENGTH = Oritech.CONFIG.processingMachines.machineFrameMaxLength();
    private static final HashMap<class_2382, HashMap<class_2382, class_2382>> occupiedAreas = new HashMap<>();
    
    @SyncField({SyncType.INITIAL, SyncType.SPARSE_TICK})
    private class_2338 areaMin;       // both min and max are inclusive
    @SyncField({SyncType.INITIAL, SyncType.SPARSE_TICK})
    private class_2338 areaMax;
    @SyncField({SyncType.INITIAL, SyncType.SPARSE_TICK})
    public boolean disabledViaRedstone;
    
    @SyncField
    private class_2338 currentTarget;
    @SyncField
    private class_2338 lastTarget;
    @SyncField
    private boolean moving;
    @SyncField
    private float currentProgress;
    
    private class_2382 currentDirection = new class_2382(1, 0, 0);    // not synced
    public long lastWorkedAt;   // not synced
    
    // for smooth client rendering only
    public class_243 lastRenderedPosition = new class_243(0, 0, 0);
    
    public FrameInteractionBlockEntity(class_2591<?> type, class_2338 pos, class_2680 state) {
        super(type, pos, state);
    }
    
    public boolean tryFindFrame() {
        
        Oritech.LOGGER.debug("searching machine frame");
        
        // select block on back (or based on offset of machine)
        // from there on move right, till no more frame blocks are found
        // then move back, searching again till end
        // then move left, searching again till end
        // then move forward, searching again till end
        // then move right again, searching till start position
        
        var facing = getFacing();
        var backRelative = new class_2382(getFrameOffset(), 0, 0);
        var searchStart = (class_2338) Geometry.offsetToWorldPosition(facing, backRelative, field_11867);
        
        var endRightFront = searchFrameLine(searchStart, getRight(facing));
        if (endRightFront.equals(class_2338.field_10980)) {
            highlightBlock(searchStart);
            return false;
        }
        
        var endRightBack = searchFrameLine(endRightFront, getBackward(facing));
        if (endRightBack.equals(endRightFront)) {
            highlightBlock(endRightFront.method_10081(getRight(facing)));
            highlightBlock(endRightFront.method_10081(getBackward(facing)));
            return false;
        }
        
        var endLeftBack = searchFrameLine(endRightBack, getLeft(facing));
        if (endLeftBack.equals(endRightBack)) {
            highlightBlock(endRightBack.method_10081(getBackward(facing)));
            highlightBlock(endRightBack.method_10081(getLeft(facing)));
            return false;
        }
        
        var endLeftFront = searchFrameLine(endLeftBack, getForward(facing));
        if (endLeftFront.equals(endLeftBack)) {
            highlightBlock(endLeftBack.method_10081(getLeft(facing)));
            highlightBlock(endLeftBack.method_10081(getForward(facing)));
            return false;
        }
        
        var endMiddleFront = searchFrameLineEnd(endLeftFront, getRight(facing), searchStart);
        if (endMiddleFront.equals(endLeftFront)) {
            highlightBlock(endMiddleFront.method_10081(getForward(facing)));
            highlightBlock(endMiddleFront.method_10081(getRight(facing)));
            return false;
        }
        if (!endMiddleFront.equals(searchStart)) {
            highlightBlock(endMiddleFront.method_10081(getRight(facing)));
            return false;
        }
        
        var innerValid = checkInnerEmpty(endLeftBack, endRightFront);
        if (!innerValid) return false;
        
        // offset values by 1 to define the working area instead of bounds
        var startX = Math.min(endLeftFront.method_10263(), endRightBack.method_10263()) + 1;
        var startZ = Math.min(endLeftFront.method_10260(), endRightBack.method_10260()) + 1;
        areaMin = new class_2338(startX, method_11016().method_10264(), startZ);
        
        var endX = Math.max(endLeftFront.method_10263(), endRightBack.method_10263()) - 1;
        var endZ = Math.max(endLeftFront.method_10260(), endRightBack.method_10260()) - 1;
        areaMax = new class_2338(endX, method_11016().method_10264(), endZ);
        
        if (currentTarget == null || !isInBounds(currentTarget)) {
            currentTarget = areaMin;
            lastTarget = areaMin;
        }
        this.method_5431();
        sendUpdate(SyncType.INITIAL);
        
        return true;
    }
    
    protected class_2350 getFacing() {
        return Objects.requireNonNull(field_11863).method_8320(method_11016()).method_11654(class_2741.field_12481);
    }
    
    private boolean checkInnerEmpty(class_2338 leftBack, class_2338 rightFront) {
        assert field_11863 != null;
        
        var lengthX = Math.abs(leftBack.method_10263() - rightFront.method_10263());
        var lengthZ = Math.abs(leftBack.method_10260() - rightFront.method_10260());
        
        var dirX = leftBack.method_10263() - rightFront.method_10263() > 0 ? -1 : 1;
        var dirZ = leftBack.method_10260() - rightFront.method_10260() > 0 ? -1 : 1;
        
        var valid = true;
        
        for (int x = 1; x < lengthX; x++) {
            for (int z = 1; z < lengthZ; z++) {
                var offset = new class_2338(dirX * x, 0, dirZ * z);
                var checkPos = leftBack.method_10081(offset);
                var foundBlock = field_11863.method_8320(checkPos).method_26204();
                if (!foundBlock.equals(class_2246.field_10124)) {
                    highlightBlock(checkPos);
                    valid = false;
                }
            }
        }
        
        
        return valid;
    }
    
    private class_2338 searchFrameLine(class_2338 searchStart, class_2382 direction) {
        
        var lastPosition = class_2338.field_10980;        // yes this will break if the frame starts at 0/0/0, however I'm willing to accept this
        
        for (int i = 0; i < MAX_SEARCH_LENGTH; i++) {
            var checkPos = searchStart.method_10081(direction.method_35862(i));
            if (testForFrame(checkPos)) {
                lastPosition = checkPos;
            } else {
                break;
            }
        }
        
        return lastPosition;
    }
    
    private class_2338 searchFrameLineEnd(class_2338 searchStart, class_2382 direction, class_2338 searchEnd) {
        
        var lastPosition = class_2338.field_10980;        // yes this will break if the frame starts at 0/0/0, however I'm willing to accept this
        
        for (int i = 0; i < MAX_SEARCH_LENGTH; i++) {
            var checkPos = searchStart.method_10081(direction.method_35862(i));
            if (testForFrame(checkPos)) {
                
                if (checkPos.equals(searchEnd)) {
                    Oritech.LOGGER.debug("found start, machine is valid");
                    return checkPos;
                }
                
                lastPosition = checkPos;
            } else {
                break;
            }
        }
        
        return lastPosition;
    }
    
    @SuppressWarnings("DataFlowIssue")
    private boolean testForFrame(class_2338 pos) {
        var found = field_11863.method_8320(pos).method_26204();
        return found.equals(BlockContent.MACHINE_FRAME_BLOCK);
    }
    
    @Override
    public void serverTick(class_1937 world, class_2338 pos, class_2680 state, NetworkedBlockEntity blockEntity) {
        if (!isActive(state) || !state.method_11654(FrameInteractionBlock.HAS_FRAME) || getAreaMin() == null)
            return;
        
        if (!canProgress()) return;
        
        while (currentProgress > 0.01) {
            if (!moving && currentProgress >= getWorkTime()) {
                method_5431();
                if (startBlockMove()) { // only complete work if we can move to the next position
                    currentProgress -= getWorkTime();
                    finishBlockWork(lastTarget);
                    updateToolPosInFrame();
                    moving = true;
                } else {
                    break;  // next pos is blocked. Keep current progress, but dont perform any more actions.
                }
            } else if (moving && currentProgress >= getMoveTime()) {
                method_5431();
                if (hasWorkAvailable(currentTarget)) {
                    moving = false;
                    currentProgress -= getMoveTime();
                } else if (startBlockMove()) {
                    updateToolPosInFrame();
                    currentProgress -= getMoveTime();
                } else {
                    break;
                }
                
            } else {
                break;
            }
        }
        
        doProgress(moving);
        currentProgress++;
        lastWorkedAt = world.method_8510();
    }
    
    private boolean isBlockAvailable(class_2338 target) {
        if (!occupiedAreas.containsKey(areaMin)) {
            occupiedAreas.put(areaMin, new HashMap<>(1));
            return true;
        }
        
        var frameEntries = occupiedAreas.get(areaMin);
        return !frameEntries.containsValue(target);
    }
    
    private void updateToolPosInFrame() {
        var frameEntries = occupiedAreas.get(areaMin);
        frameEntries.put(field_11867, currentTarget);
    }
    
    public void cleanup() {
        var frameEntries = occupiedAreas.get(areaMin);
        if (frameEntries != null)
            frameEntries.remove(field_11867);
    }
    
    @SuppressWarnings("BooleanMethodIsAlwaysInverted")
    protected abstract boolean hasWorkAvailable(class_2338 toolPosition);
    
    protected abstract void doProgress(boolean moving);
    
    protected abstract boolean canProgress();
    
    public abstract void finishBlockWork(class_2338 processed);
    
    @Override
    protected void method_11007(class_2487 nbt, class_7225.class_7874 registryLookup) {
        super.method_11007(nbt, registryLookup);
        if (method_11010().method_11654(FrameInteractionBlock.HAS_FRAME) && areaMin != null) {
            nbt.method_10544("areaMin", areaMin.method_10063());
            nbt.method_10544("areaMax", areaMax.method_10063());
            nbt.method_10544("currentTarget", currentTarget.method_10063());
            nbt.method_10544("currentDirection", new class_2338(currentDirection).method_10063());
            nbt.method_10569("progress", (int) currentProgress);
            nbt.method_10556("moving", moving);
        }
    }
    
    @Override
    protected void method_11014(class_2487 nbt, class_7225.class_7874 registryLookup) {
        super.method_11014(nbt, registryLookup);
        if (method_11010().method_11654(FrameInteractionBlock.HAS_FRAME)) {
            areaMin = class_2338.method_10092(nbt.method_10537("areaMin"));
            areaMax = class_2338.method_10092(nbt.method_10537("areaMax"));
            currentTarget = class_2338.method_10092(nbt.method_10537("currentTarget"));
            currentDirection = class_2338.method_10092(nbt.method_10537("currentDirection"));
            lastTarget = currentTarget;
            currentProgress = nbt.method_10550("progress");
            moving = nbt.method_10577("moving");
        }
    }
    
    private boolean startBlockMove() {
        
        var nextPos = currentTarget.method_10081(currentDirection);
        var nextDir = currentDirection;
        if (!isInBounds(nextPos)) {
            nextPos = currentTarget.method_10069(0, 0, 1);
            nextDir = currentDirection.method_35862(-1);
            if (!isInBounds(nextPos)) {
                var data = resetWorkPosition();
                nextPos = data.method_15442();
                nextDir = data.method_15441();
            }
        }
        
        // tries to not put 2 tool heads in the same spot, but also allow overtaking if previous machine is too slow
        if (!isBlockAvailable(nextPos) && currentProgress <= getWorkTime() * getSpeedMultiplier() * 2 + 4) return false;
        
        lastTarget = currentTarget;
        currentTarget = nextPos;
        currentDirection = nextDir;
        
        return true;
    }
    
    // return start position + direction
    private class_3545<class_2338, class_2338> resetWorkPosition() {
        return new class_3545<>(areaMin, new class_2338(1, 0, 0));
    }
    
    private boolean isInBounds(class_2338 pos) {
        return pos.method_10263() >= areaMin.method_10263() && pos.method_10263() <= areaMax.method_10263()
                 && pos.method_10260() >= areaMin.method_10260() && pos.method_10260() <= areaMax.method_10260();
    }
    
    private void highlightBlock(class_2338 block) {
        ParticleContent.HIGHLIGHT_BLOCK.spawn(field_11863, class_243.method_24954(block), null);
    }
    
    public abstract class_2680 getMachineHead();
    
    public int getFrameOffset() {
        return 1;
    }
    
    public float getSpeedMultiplier() {
        return 1f;
    }
    
    public class_2338 getAreaMin() {
        return areaMin;
    }
    
    public void setAreaMin(class_2338 areaMin) {
        this.areaMin = areaMin;
    }
    
    public class_2338 getAreaMax() {
        return areaMax;
    }
    
    public void setAreaMax(class_2338 areaMax) {
        this.areaMax = areaMax;
    }
    
    public class_2338 getCurrentTarget() {
        return currentTarget;
    }
    
    public void setCurrentTarget(class_2338 currentTarget) {
        this.currentTarget = currentTarget;
    }
    
    public class_2338 getLastTarget() {
        return lastTarget;
    }
    
    public void setLastTarget(class_2338 lastTarget) {
        this.lastTarget = lastTarget;
    }
    
    public int getCurrentProgress() {
        return (int) currentProgress;
    }
    
    public void setCurrentProgress(int currentProgress) {
        this.currentProgress = currentProgress;
    }
    
    public boolean isActive(class_2680 state) {
        return true;
    }
    
    public boolean isMoving() {
        return moving;
    }
    
    public void setMoving(boolean moving) {
        this.moving = moving;
    }
    
    public class_2382 getCurrentDirection() {
        return currentDirection;
    }
    
    public void setCurrentDirection(class_2382 currentDirection) {
        this.currentDirection = currentDirection;
    }
    
    public abstract float getMoveTime();
    
    public abstract float getWorkTime();
    
    public class_1799 getToolheadAdditionalRender() {
        return null;
    }
    
    @Override
    public void sendUpdate(SyncType type) {
        if (currentTarget == null || lastTarget == null) return;
        super.sendUpdate(type);
    }
}
