package rearth.oritech.util;

import rearth.oritech.Oritech;
import rearth.oritech.api.energy.containers.DynamicEnergyStorage;
import rearth.oritech.api.item.ItemApi;
import rearth.oritech.block.blocks.addons.MachineAddonBlock;
import rearth.oritech.block.blocks.addons.MachineAddonBlock.AddonSettings;
import rearth.oritech.block.entity.addons.AddonBlockEntity;
import java.util.*;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2382;
import net.minecraft.class_2487;
import net.minecraft.class_2499;
import net.minecraft.class_2520;
import net.minecraft.class_2680;

public interface MachineAddonController {
    
    // list of where actually connected addons are
    List<class_2338> getConnectedAddons();
    
    // a list of where addons could be placed
    List<class_2338> getOpenAddonSlots();
    
    class_2338 getPosForAddon();
    
    class_1937 getWorldForAddon();
    
    class_2350 getFacingForAddon();
    
    DynamicEnergyStorage getStorageForAddon();
    
    ItemApi.InventoryStorage getInventoryForAddon();
    
    ScreenProvider getScreenProvider();
    
    List<class_2382> getAddonSlots();
    
    BaseAddonData getBaseAddonData();
    
    void setBaseAddonData(BaseAddonData data);
    
    long getDefaultCapacity();
    
    long getDefaultInsertRate();
    
    default float getCoreQuality() {
        return 1f;
    }
    
    // to initialize everything, should be called when right-clicked
    default void initAddons(class_2338 brokenAddon) {
        
        var foundAddons = getAllAddons(brokenAddon);
        
        gatherAddonStats(foundAddons);
        writeAddons(foundAddons);
        updateEnergyContainer();
        removeOldAddons(foundAddons);
        
        getConnectedAddons().clear();
        updateEnergyContainer();
        
        for (var addon : foundAddons) {
            getConnectedAddons().add(addon.pos());
        }
    }
    
    private void removeOldAddons(List<AddonBlock> foundAddons) {
        // remove/reset all old addons that are not connected anymore
        for (var addon : getConnectedAddons()) {
            if (foundAddons.stream().noneMatch(newAddon -> newAddon.pos().equals(addon))) {
                var state = Objects.requireNonNull(getWorldForAddon()).method_8320(addon);
                if (state.method_26204() instanceof MachineAddonBlock) {
                    getWorldForAddon().method_8501(addon, state.method_11657(MachineAddonBlock.ADDON_USED, false));
                    getWorldForAddon().method_8452(addon, state.method_26204());
                }
            }
        }
    }
    
    default void initAddons() {
        initAddons(null);
    }
    
    // to be called if controller or one of the addons has been broken
    default void resetAddons() {
        
        for (var addon : getConnectedAddons()) {
            var state = Objects.requireNonNull(getWorldForAddon()).method_8320(addon);
            if (state.method_26204() instanceof MachineAddonBlock) {
                getWorldForAddon().method_8501(addon, state.method_11657(MachineAddonBlock.ADDON_USED, false));
                getWorldForAddon().method_8452(addon, state.method_26204());
            }
        }
        
        getConnectedAddons().clear();
        updateEnergyContainer();
    }
    
    // addon loading algorithm, called during init
    default List<AddonBlock> getAllAddons(class_2338 brokenAddon) {
        
        var useLayered = Oritech.CONFIG.layeredExtenders();
        
        var maxIterationCount = (int) getCoreQuality() + 1;
        
        // start with base slots (on machine itself)
        // repeat N times (dependent on core quality?):
        //   go through all slots
        //   check if slot is occupied by MachineAddonBlock, check if block is not used
        //   if valid and extender: add all neighboring positions to search set
        var world = getWorldForAddon();
        var pos = getPosForAddon();
        assert world != null;
        
        var openSlots = getOpenAddonSlots();
        openSlots.clear();
        
        var foundExtenders = 0;
        
        var baseSlots = getAddonSlots();    // available addon slots on machine itself (includes multiblocks)
        var searchedPositions = new HashSet<class_2338>(baseSlots.size()); // all positions ever checked, to avoid adding duplicates
        var queuedPositions = new ArrayList<class_2338>(baseSlots.size());
        var result = new ArrayList<AddonBlock>(baseSlots.size());   // results, unused addon blocks
        
        // fill initial spots
        for (var initialSpot : baseSlots) {
            queuedPositions.add((class_2338) Geometry.offsetToWorldPosition(getFacingForAddon(), initialSpot, pos));
        }
        
        // to allow loops where we modify the content
        var toAdd = new HashSet<class_2338>();
        var toRemove = new HashSet<class_2338>();
        
        //everything done in world space
        for (int i = 0; i < maxIterationCount; i++) {
            if (queuedPositions.isEmpty()) break;
            
            for (var candidatePos : queuedPositions) {
                if (searchedPositions.contains(candidatePos)) continue;
                searchedPositions.add(candidatePos);
                toRemove.add(candidatePos);
                
                var candidate = world.method_8320(candidatePos);
                var candidateEntity = world.method_8321(candidatePos);
                
                // if the candidate is the broken addon, skip it
                if (candidatePos.equals(brokenAddon)) {
                    openSlots.add(candidatePos);
                    continue;
                }
                
                // if the candidate is not an addon
                if (!(candidate.method_26204() instanceof MachineAddonBlock addonBlock) || !(candidateEntity instanceof AddonBlockEntity candidateAddonEntity)) {
                    
                    // if the block is not part of the machine itself
                    if (!candidatePos.equals(pos))
                        openSlots.add(candidatePos);
                    continue;
                }
                
                // if the candidate is in use with another controller
                if (candidate.method_11654(MachineAddonBlock.ADDON_USED) && !candidateAddonEntity.getControllerPos().equals(pos)) {
                    openSlots.add(candidatePos);
                    continue;
                }
                
                // if non-layered mode, check if we have too many extenders already
                if (addonBlock.getAddonSettings().extender() && !useLayered) {
                    if (foundExtenders < (maxIterationCount - 1)) {
                        foundExtenders++;
                    } else {
                        continue;
                    }
                }
                
                var entry = new AddonBlock(addonBlock, candidate, candidatePos, candidateAddonEntity);
                result.add(entry);
                
                if (addonBlock.getAddonSettings().extender()) {
                    var neighbors = getNeighbors(candidatePos);
                    for (var neighbor : neighbors) {
                        if (!searchedPositions.contains(neighbor)) toAdd.add(neighbor);
                    }
                }
            }
            
            queuedPositions.addAll(toAdd);
            queuedPositions.removeAll(toRemove);
            toAdd.clear();
            toRemove.clear();
        }
        
        return result;
        
    }
    
    // can be overridden to allow custom addon loading (e.g. custom stat, or check for specific addon existence)
    default void gatherAddonStats(List<AddonBlock> addons) {
        
        var speed = 1f;
        var efficiency = 1f;
        var energyAmount = 0L;
        var energyInsert = 0L;
        var extraChambers = 0;
        
        for (var addon : addons) {
            var addonSettings = addon.addonBlock().getAddonSettings();
            
            if (Oritech.CONFIG.additiveAddons()) {
                speed += 1 - addonSettings.speedMultiplier();
                efficiency += 1 - addonSettings.efficiencyMultiplier();
            } else {
                speed *= addonSettings.speedMultiplier();
                efficiency *= addonSettings.efficiencyMultiplier();
            }
            
            energyAmount += addonSettings.addedCapacity();
            energyInsert += addonSettings.addedInsert();
            extraChambers += addonSettings.chamberCount();
            
            getAdditionalStatFromAddon(addon);
        }
        
        if (Oritech.CONFIG.additiveAddons()) {
            // convert addon numbers to base (e.g. +2 (+200%) speed bonus is actually a total multiplier of 0.5) (+2 would be a speed of 3, because we start at 1)
            // efficiency change of -100% would result in efficiency multiplier of 2. -400% would be 5
            // efficiency and speed numbers < 1 here make things better.
            
            speed = 1f / speed;
            
            var efficiencyChange = efficiency - 1;
            efficiency = 1f / efficiency;
            if (efficiencyChange < 0) {
                efficiency = 1 + Math.abs(efficiencyChange);   // yes this order looks stupid, but it's easier to understand like this for me
            }
        }
        
        var baseData = new BaseAddonData(speed, efficiency, energyAmount, energyInsert, extraChambers);
        setBaseAddonData(baseData);
    }
    
    // used to check for specific addons, or do something if a specific addon has been found
    default void getAdditionalStatFromAddon(AddonBlock addonBlock) {
    
    }
    
    // update state of the found addons
    default void writeAddons(List<AddonBlock> addons) {
        
        var world = getWorldForAddon();
        var pos = getPosForAddon();
        assert world != null;
        
        for (var addon : addons) {
            var newState = addon.state()
                             .method_11657(MachineAddonBlock.ADDON_USED, true);
            // Set controller before setting block state, otherwise the addon will think
            // it's not connected to a machine the first time neighbor blocks are being updated.
            addon.addonEntity().setControllerPos(pos);
            world.method_8501(addon.pos(), newState);
        }
    }
    
    // part of init/break, updates the energy container size
    default void updateEnergyContainer() {
        var energyStorage = getStorageForAddon();
        var addonData = getBaseAddonData();
        energyStorage.capacity = getDefaultCapacity() + addonData.energyBonusCapacity;
        energyStorage.maxInsert = getDefaultInsertRate() + addonData.energyBonusTransfer;
        energyStorage.amount = Math.min(energyStorage.amount, energyStorage.capacity);
    }
    
    default void writeAddonToNbt(class_2487 nbt) {
        var data = getBaseAddonData();
        nbt.method_10548("speed", data.speed);
        nbt.method_10548("efficiency", data.efficiency);
        nbt.method_10544("energyBonusCapacity", data.energyBonusCapacity);
        nbt.method_10544("energyBonusTransfer", data.energyBonusTransfer);
        nbt.method_10569("extraChambers", data.extraChambers);
        
        var posList = new class_2499();
        for (var pos : getConnectedAddons()) {
            var posTag = new class_2487();
            posTag.method_10569("x", pos.method_10263());
            posTag.method_10569("y", pos.method_10264());
            posTag.method_10569("z", pos.method_10260());
            posList.add(posTag);
        }
        nbt.method_10566("connectedAddons", posList);
    }
    
    default void loadAddonNbtData(class_2487 nbt) {
        var data = new BaseAddonData(nbt.method_10583("speed"), nbt.method_10583("efficiency"), nbt.method_10537("energyBonusCapacity"), nbt.method_10537("energyBonusTransfer"), nbt.method_10550("extraChambers"));
        setBaseAddonData(data);
        
        var posList = nbt.method_10554("connectedAddons", class_2520.field_33260);
        var connectedAddons = getConnectedAddons();
        
        for (var posTag : posList) {
            var posCompound = (class_2487) posTag;
            var x = posCompound.method_10550("x");
            var y = posCompound.method_10550("y");
            var z = posCompound.method_10550("z");
            var pos = new class_2338(x, y, z);
            connectedAddons.add(pos);
        }
    }
    
    private static Set<class_2338> getNeighbors(class_2338 pos) {
        return Set.of(
          pos.method_10069(-1, 0, 0),
          pos.method_10069(1, 0, 0),
          pos.method_10069(0, 0, -1),
          pos.method_10069(0, 0, 1),
          pos.method_10069(0, -1, 0),
          pos.method_10069(0, 1, 0)
        );
    }
    
    record AddonBlock(MachineAddonBlock addonBlock, class_2680 state, class_2338 pos, AddonBlockEntity addonEntity) {
    }
    
    record BaseAddonData(float speed, float efficiency, long energyBonusCapacity, long energyBonusTransfer,
                         int extraChambers) {
    }
    
    BaseAddonData DEFAULT_ADDON_DATA = new BaseAddonData(1, 1, 0, 0, 0);
    
}
