package rearth.oritech.fabric;

import com.google.common.collect.Streams;
import dev.architectury.fluid.FluidStack;
import dev.architectury.hooks.fluid.fabric.FluidStackHooksFabric;
import net.fabricmc.fabric.api.transfer.v1.context.ContainerItemContext;
import net.fabricmc.fabric.api.transfer.v1.fluid.FluidVariant;
import net.fabricmc.fabric.api.transfer.v1.storage.Storage;
import net.fabricmc.fabric.api.transfer.v1.storage.StorageView;
import net.fabricmc.fabric.api.transfer.v1.transaction.Transaction;
import net.fabricmc.fabric.api.transfer.v1.transaction.TransactionContext;
import net.fabricmc.fabric.api.transfer.v1.transaction.base.SnapshotParticipant;
import net.minecraft.class_1792;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2586;
import net.minecraft.class_2591;
import net.minecraft.class_2680;
import net.minecraft.class_3612;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import rearth.oritech.Oritech;
import rearth.oritech.util.StackContext;
import rearth.oritech.api.fluid.BlockFluidApi;
import rearth.oritech.api.fluid.FluidApi;
import rearth.oritech.api.fluid.FluidApi.FluidStorage;
import rearth.oritech.api.fluid.ItemFluidApi;
import rearth.oritech.api.fluid.containers.DelegatingFluidStorage;
import rearth.oritech.api.fluid.containers.SimpleItemFluidStorage;

import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.function.Supplier;

public class FabricFluidApiImpl implements BlockFluidApi, ItemFluidApi {
    
    @SuppressWarnings("IfCanBeSwitch")
    @Override
    public void registerBlockEntity(Supplier<class_2591<?>> typeSupplier) {
        net.fabricmc.fabric.api.transfer.v1.fluid.FluidStorage.SIDED.registerForBlockEntity(
          (entity, direction) -> {
              
              var storage = ((FluidApi.BlockProvider) entity).getFluidStorage(direction);
              
              if (storage == null) return null;
              
              if (storage instanceof FluidApi.MultiSlotStorage inOutContainer) {
                  return MultiSlotWrapper.of(inOutContainer);
              } else if (storage instanceof FluidApi.SingleSlotStorage singleContainer) {
                  return SingleSlotContainerStorageWrapper.of(singleContainer);
              } else if (storage instanceof DelegatingFluidStorage delegatedStorage) {
                  return DelegatedContainerStorageWrapper.of(delegatedStorage);
              }
              
              Oritech.LOGGER.error("Error during fluid provider registration, unable to register a fluid container");
              Oritech.LOGGER.error("Erroring container type is: {}", typeSupplier.get());
              
              return null;
          }, typeSupplier.get());
    }
    
    @Override
    public void registerForItem(Supplier<class_1792> itemSupplier) {
        net.fabricmc.fabric.api.transfer.v1.fluid.FluidStorage.ITEM.registerForItems(
          (stack, context) -> SingleSlotContainerStorageWrapper.of(((FluidApi.ItemProvider) stack.method_7909()).getFluidStorage(stack)),
          itemSupplier.get()
        );
    }
    
    @Override
    public FluidApi.FluidStorage find(class_1937 world, class_2338 pos, @Nullable class_2680 state, @Nullable class_2586 entity, @Nullable class_2350 direction) {
        var candidate = net.fabricmc.fabric.api.transfer.v1.fluid.FluidStorage.SIDED.find(world, pos, state, entity, direction);
        
        return switch (candidate) {
            case null -> null;
            case SingleSlotContainerStorageWrapper wrapper -> wrapper.container;
            case MultiSlotWrapper wrapper -> wrapper.storage;
            case DelegatedContainerStorageWrapper wrapper -> wrapper.storage;
            default -> new FabricStorageWrapper(candidate, null);
        };
    }
    
    @Override
    public FluidApi.FluidStorage find(class_1937 world, class_2338 pos, @Nullable class_2350 direction) {
        return find(world, pos, null, null, direction);
    }
    
    @Override
    public FluidApi.FluidStorage find(StackContext stack) {
        if (stack.getValue().method_7947() > 1) return null;
        var context = ContainerItemContext.ofSingleSlot(new ItemStackStorage(stack));
        var candidate = net.fabricmc.fabric.api.transfer.v1.fluid.FluidStorage.ITEM.find(stack.getValue(), context);
        if (candidate == null) return null;
        if (candidate instanceof SingleSlotContainerStorageWrapper wrapper && wrapper.container instanceof SimpleItemFluidStorage itemContainer) return itemContainer.withCallback(ignored -> stack.sync());
        return new FabricStorageWrapper(candidate, stack);
    }
    
    // this is used to interact with fluid storages from other mods
    public static class FabricStorageWrapper extends FluidApi.FluidStorage {
        
        public final Storage<FluidVariant> storage;
        public final @Nullable StackContext context;
        
        public FabricStorageWrapper(Storage<FluidVariant> storage, @Nullable StackContext context) {
            this.storage = storage;
            this.context = context;
        }
        
        @Override
        public long insert(FluidStack in, boolean simulate) {
            try (var transaction = Transaction.openOuter()) {
                var inserted = storage.insert(FluidStackHooksFabric.toFabric(in), in.getAmount(), transaction);
                if (!simulate)
                    transaction.commit();
                return inserted;
            }
        }
        
        @Override
        public long extract(FluidStack out, boolean simulate) {
            try (var transaction = Transaction.openOuter()) {
                var extracted = storage.extract(FluidStackHooksFabric.toFabric(out), out.getAmount(), transaction);
                if (!simulate)
                    transaction.commit();
                return extracted;
            }
        }
        
        @Override
        public List<FluidStack> getContent() {
            return Streams
                     .stream(storage.iterator())
                     .map(FluidStackHooksFabric::fromFabric)
                     .toList();
        }
        
        @Override
        public void update() {
            if (context != null)
                context.sync();
        }
        
        @Override
        public long getCapacity() {
            Oritech.LOGGER.warn("tried to access capacity of external container");
            return 0L;  // this should not be used I guess?
        }
    }
    
    // this is used by other mods to interact with the oritech single slot fluid containers (machines/items)
    public static class SingleSlotContainerStorageWrapper extends SnapshotParticipant<FluidStack> implements Storage<FluidVariant> {
        
        public final FluidApi.SingleSlotStorage container;
        private Set<StorageView<FluidVariant>> contentView;
        
        public static SingleSlotContainerStorageWrapper of(@Nullable FluidApi.SingleSlotStorage container) {
            if (container == null) return null;
            return new SingleSlotContainerStorageWrapper(container);
        }
        
        public SingleSlotContainerStorageWrapper(FluidApi.SingleSlotStorage container) {
            this.container = container;
        }
        
        @Override
        public boolean supportsInsertion() {
            return container.supportsInsertion();
        }
        
        @Override
        public long insert(FluidVariant fluidVariant, long amount, TransactionContext transaction) {
            updateSnapshots(transaction);
            transaction.addCloseCallback((transactionContext, result) -> {
                if (result.wasCommitted()) {
                    container.update();
                }
            });
            return container.insert(FluidStackHooksFabric.fromFabric(fluidVariant, amount), false);
        }
        
        @Override
        public boolean supportsExtraction() {
            return container.supportsExtraction();
        }
        
        @Override
        public long extract(FluidVariant fluidVariant, long amount, TransactionContext transaction) {
            updateSnapshots(transaction);
            transaction.addCloseCallback((transactionContext, result) -> {
                if (result.wasCommitted()) {
                    container.update();
                }
            });
            return container.extract(FluidStackHooksFabric.fromFabric(fluidVariant, amount), false);
        }
        
        @Override
        public @NotNull Iterator<StorageView<FluidVariant>> iterator() {
            
            if (contentView != null) return contentView.iterator();
            
            contentView = Collections.singleton(new StorageView<FluidVariant>() {
                @Override
                public long extract(FluidVariant fluidVariant, long amount, TransactionContext transaction) {
                    return SingleSlotContainerStorageWrapper.this.extract(fluidVariant, amount, transaction);
                }
                
                @Override
                public boolean isResourceBlank() {
                    return container.getStack().getFluid().equals(class_3612.field_15906);
                }
                
                @Override
                public FluidVariant getResource() {
                    return FluidStackHooksFabric.toFabric(container.getStack());
                }
                
                @Override
                public long getAmount() {
                    return container.getStack().getAmount();
                }
                
                @Override
                public long getCapacity() {
                    return container.getCapacity();
                }
            }.getUnderlyingView());
            
            return contentView.iterator();
        }
        
        @Override
        protected FluidStack createSnapshot() {
            return container.getStack().copy();
        }
        
        @Override
        protected void readSnapshot(FluidStack fluidVariantResourceAmount) {
            container.setStack(fluidVariantResourceAmount);
        }
    }
    
    // this is used by other mods to interact with the oritech in/out fluid containers
    public static class MultiSlotWrapper extends SnapshotParticipant<List<FluidStack>> implements Storage<FluidVariant> {
        
        public final FluidApi.MultiSlotStorage storage;
        
        public static MultiSlotWrapper of(FluidApi.MultiSlotStorage storage) {
            if (storage == null) return null;
            return new MultiSlotWrapper(storage);
        }
        
        private MultiSlotWrapper(FluidApi.MultiSlotStorage storage) {
            this.storage = storage;
        }
        
        @Override
        public boolean supportsInsertion() {
            return storage.supportsInsertion();
        }
        
        @Override
        public long insert(FluidVariant fluidVariant, long l, TransactionContext transaction) {
            updateSnapshots(transaction);
            transaction.addCloseCallback((transactionContext, result) -> {
                if (result.wasCommitted()) {
                    storage.update();
                }
            });
            return storage.insert(FluidStackHooksFabric.fromFabric(fluidVariant, l), false);
        }
        
        @Override
        public boolean supportsExtraction() {
            return storage.supportsExtraction();
        }
        
        @Override
        public long extract(FluidVariant fluidVariant, long l, TransactionContext transaction) {
            updateSnapshots(transaction);
            transaction.addCloseCallback((transactionContext, result) -> {
                if (result.wasCommitted()) {
                    storage.update();
                }
            });
            return storage.extract(FluidStackHooksFabric.fromFabric(fluidVariant, l), false);
        }
        
        @Override
        public @NotNull Iterator<StorageView<FluidVariant>> iterator() {
            return storage.getContent().stream().map(stack -> new StorageView<FluidVariant>() {
                
                @Override
                public long extract(FluidVariant fluidVariant, long l, TransactionContext transactionContext) {
                    return MultiSlotWrapper.this.extract(fluidVariant, l, transactionContext);
                }
                
                @Override
                public boolean isResourceBlank() {
                    return stack.isEmpty();
                }
                
                @Override
                public FluidVariant getResource() {
                    return FluidStackHooksFabric.toFabric(stack);
                }
                
                @Override
                public long getAmount() {
                    return stack.getAmount();
                }
                
                @Override
                public long getCapacity() {
                    return storage.getCapacity();
                }
            }.getUnderlyingView()).iterator();
        }
        
        @Override
        protected List<FluidStack> createSnapshot() {
            return storage.getContent().stream().map(FluidStack::copy).toList();
        }
        
        @Override
        protected void readSnapshot(List<FluidStack> fluidStacks) {
            for (int i = 0; i < fluidStacks.size(); i++) {
                storage.setStack(i, fluidStacks.get(i));
            }
        }
    }
    
    public static class DelegatedContainerStorageWrapper extends SnapshotParticipant<List<FluidStack>> implements Storage<FluidVariant> {
        
        private final DelegatingFluidStorage storage;
        
        public static DelegatedContainerStorageWrapper of(DelegatingFluidStorage storage) {
            if (storage == null) return null;
            return new DelegatedContainerStorageWrapper(storage);
        }
        
        private DelegatedContainerStorageWrapper(DelegatingFluidStorage storage) {
            this.storage = storage;
        }
        
        @Override
        public boolean supportsInsertion() {
            return storage.supportsInsertion();
        }
        
        @Override
        public long insert(FluidVariant fluidVariant, long l, TransactionContext transaction) {
            updateSnapshots(transaction);
            transaction.addCloseCallback((transactionContext, result) -> {
                if (result.wasCommitted()) {
                    storage.update();
                }
            });
            return storage.insert(FluidStackHooksFabric.fromFabric(fluidVariant, l), false);
        }
        
        @Override
        public boolean supportsExtraction() {
            return storage.supportsExtraction();
        }
        
        @Override
        public long extract(FluidVariant fluidVariant, long l, TransactionContext transaction) {
            updateSnapshots(transaction);
            transaction.addCloseCallback((transactionContext, result) -> {
                if (result.wasCommitted()) {
                    storage.update();
                }
            });
            return storage.extract(FluidStackHooksFabric.fromFabric(fluidVariant, l), false);
        }
        
        @Override
        public @NotNull Iterator<StorageView<FluidVariant>> iterator() {
            return storage.getContent().stream().map(stack -> new StorageView<FluidVariant>() {
                
                @Override
                public long extract(FluidVariant fluidVariant, long l, TransactionContext transactionContext) {
                    return DelegatedContainerStorageWrapper.this.extract(fluidVariant, l, transactionContext);
                }
                
                @Override
                public boolean isResourceBlank() {
                    return stack.isEmpty();
                }
                
                @Override
                public FluidVariant getResource() {
                    return FluidStackHooksFabric.toFabric(stack);
                }
                
                @Override
                public long getAmount() {
                    return stack.getAmount();
                }
                
                @Override
                public long getCapacity() {
                    return storage.getCapacity();
                }
            }.getUnderlyingView()).iterator();
        }
        
        @Override
        protected List<FluidStack> createSnapshot() {
            return storage.getContent().stream().map(FluidStack::copy).toList();
        }
        
        @Override
        protected void readSnapshot(List<FluidStack> fluidStacks) {
            storage.setContent(fluidStacks);
        }
    }
    
}
