package rearth.oritech.block.entity.interaction;

import dev.architectury.registry.menu.ExtendedMenuProvider;
import io.netty.buffer.ByteBuf;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import rearth.oritech.Oritech;
import rearth.oritech.api.energy.EnergyApi;
import rearth.oritech.api.energy.containers.DelegatingEnergyStorage;
import rearth.oritech.api.energy.containers.DynamicStatisticEnergyStorage;
import rearth.oritech.api.item.ItemApi;
import rearth.oritech.api.item.containers.SimpleInventoryStorage;
import rearth.oritech.api.networking.NetworkedBlockEntity;
import rearth.oritech.api.networking.SyncField;
import rearth.oritech.api.networking.SyncType;
import rearth.oritech.api.networking.UpdatableField;
import rearth.oritech.block.base.entity.ExpandableEnergyStorageBlockEntity;
import rearth.oritech.block.blocks.processing.MachineCoreBlock;
import rearth.oritech.client.init.ModScreens;
import rearth.oritech.client.ui.UpgradableMachineScreenHandler;
import rearth.oritech.init.BlockEntitiesContent;
import rearth.oritech.init.SoundContent;
import rearth.oritech.util.InventoryInputMode;
import rearth.oritech.util.MultiblockMachineController;
import rearth.oritech.util.ScreenProvider;

import java.util.*;
import net.minecraft.class_1263;
import net.minecraft.class_1657;
import net.minecraft.class_1661;
import net.minecraft.class_1703;
import net.minecraft.class_18;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2382;
import net.minecraft.class_2398;
import net.minecraft.class_2487;
import net.minecraft.class_2499;
import net.minecraft.class_2520;
import net.minecraft.class_2540;
import net.minecraft.class_2561;
import net.minecraft.class_2680;
import net.minecraft.class_2741;
import net.minecraft.class_2960;
import net.minecraft.class_3218;
import net.minecraft.class_3419;
import net.minecraft.class_3917;
import net.minecraft.class_7225;
import net.minecraft.class_9135;
import net.minecraft.class_9139;

public class PowerPoleEntity extends NetworkedBlockEntity implements MultiblockMachineController, ExtendedMenuProvider,
                                                                       ScreenProvider, EnergyApi.BlockProvider {
    
    // stores data per dimension
    public static final HashMap<class_2960, PoleNetworkData> POLE_NETWORK_DATA = new HashMap<>();
    
    // multiblock
    private final ArrayList<class_2338> coreBlocksConnected = new ArrayList<>();
    @SyncField(SyncType.GUI_OPEN)
    private float coreQuality = 1f;
    @SyncField({SyncType.INITIAL, SyncType.CUSTOM})
    private final Set<ConnectionTarget> connections = new HashSet<>();
    
    private PoleNetworkData netDataRef = null;
    
    // storage
    @SyncField(SyncType.GUI_TICK)
    public DynamicStatisticEnergyStorage.EnergyStatistics currentStats = DynamicStatisticEnergyStorage.EnergyStatistics.EMPTY;
    @SyncField({SyncType.GUI_OPEN, SyncType.GUI_TICK})
    protected final PowerPoleEnergyStorage energyStorage = new PowerPoleEnergyStorage();
    
    private final EnergyApi.EnergyStorage outputStorage = new DelegatingEnergyStorage(energyStorage, null) {
        @Override
        public boolean supportsInsertion() {
            return false;
        }
        
        @Override
        public long insert(long amount, boolean simulate) {
            return 0L;
        }
    };
    
    private final SimpleInventoryStorage basicInv = new SimpleInventoryStorage(0, this::method_5431);
    
    public PowerPoleEntity(class_2338 pos, class_2680 state) {
        super(BlockEntitiesContent.POWER_POLE_ENTITY, pos, state);
    }
    
    @Override
    public void serverTick(class_1937 world, class_2338 pos, class_2680 state, NetworkedBlockEntity blockEntity) {
        
        outputEnergy();
        
        energyStorage.tick(world.method_8510());
        
        if (world.field_9229.method_43057() > 0.95f) {
            
            var stats = this.energyStorage.getCurrentStatistics(world.method_8510());
            var moved = stats.insertedLastTickTotal() + stats.extractedLastTickTotal();
            
            if (moved > 10 && world instanceof class_3218 serverLevel) {
                var at = field_11867.method_46558().method_1031(world.field_9229.method_43057() * 0.4, world.field_9229.method_43057() * 0.4, world.field_9229.method_43057() * 0.4);
                serverLevel.method_14199(class_2398.field_29644, at.field_1352, at.field_1351, at.field_1350, 2, world.field_9229.method_43057(), world.field_9229.method_43057(), world.field_9229.method_43057(), 0.15f);
            }
        }
        
    }
    
    @Override
    public void preNetworkUpdate(SyncType type) {
        super.preNetworkUpdate(type);
        currentStats = energyStorage.getCurrentStatistics(field_11863.method_8510());
    }
    
    private void outputEnergy() {
        if (!isConnected() || energyStorage.getAmount() <= 0) return;
        
        // todo caching for targets? Used to be BlockApiCache.create()
        var target = ExpandableEnergyStorageBlockEntity.getOutputPosition(field_11867, getFacingForMultiblock().method_10160());
        var candidate = EnergyApi.BLOCK.find(field_11863, target.method_15441(), target.method_15442().method_10153());
        if (candidate != null && candidate.supportsInsertion()) {
            EnergyApi.transfer(energyStorage, candidate, Long.MAX_VALUE, false);
        }
    }
    
    public void assignNewTarget(class_2338 target, class_1657 player) {
        Oritech.LOGGER.info("Assigning new power pole target");
        
        // adjust for core blocks
        var targetState = field_11863.method_8320(target);
        if (targetState.method_26204() instanceof MachineCoreBlock && targetState.method_11654(MachineCoreBlock.USED)) {
            target = MachineCoreBlock.getControllerPos(field_11863, target);
        }
        
        var pitch = 0.85f + field_11863.field_9229.method_43057() * 0.3f;
        field_11863.method_8396(null, field_11867, SoundContent.ELECTRIC_SHOCK, class_3419.field_15248, 0.7f, pitch);
        
        var dist = target.method_19455(field_11867);
        
        if (dist < Oritech.CONFIG.poleConfig.minRange() || dist > Oritech.CONFIG.poleConfig.maxRange()) {
            player.method_43496(class_2561.method_43469("message.oritech.target_designator.pole_dist_invalid", Oritech.CONFIG.poleConfig.minRange(), Oritech.CONFIG.poleConfig.maxRange(), dist));
            return;
        }
        
        var targetEntityCandidate = field_11863.method_35230(target, BlockEntitiesContent.POWER_POLE_ENTITY);
        if (targetEntityCandidate.isEmpty() || target.equals(field_11867)) {
            player.method_43496(class_2561.method_43471("message.oritech.target_designator.pole_position_invalid"));
            return;
        }
        
        var targetEntity = targetEntityCandidate.get();
        
        if (this.connections.stream().anyMatch(elem -> elem.pos().equals(targetEntity.field_11867))) {
            this.removeIncomingConnection(target);
            targetEntity.removeIncomingConnection(field_11867);
            
            var netData = getCachedNetData();
            var net = netData.getNetwork(field_11867);
            this.updateConnectionsInState(net);
            targetEntity.updateConnectionsInState(net);
            
            netData.updateNetworkSplit(Set.of(field_11867, target), getNetwork());
            player.method_43496(class_2561.method_43471("message.oritech.target_designator.removing_pole_connection"));
            return;
        }
        
        connections.add(targetEntity.getConnectionData());
        targetEntity.assignIncomingConnection(this);
        
        var allNetworks = getCachedNetData();
        
        var ownNet = getNetwork();
        var isConnected = isConnected();
        var targetNet = targetEntity.getNetwork();
        var targetConnected = targetEntity.isConnected();
        
        if (!isConnected && targetConnected) {
            // join network of target
            joinNetwork(targetNet, allNetworks);
        } else if (isConnected && !targetConnected) {
            // join target into own network
            targetEntity.joinNetwork(ownNet, allNetworks);
        } else if (!isConnected && !targetConnected) {
            // neither connected, create new network, then let both join
            var newNet = createNetwork(allNetworks);
            this.joinNetwork(newNet, allNetworks);
            targetEntity.joinNetwork(newNet, allNetworks);
        } else if (isConnected && targetConnected) {
            if (targetNet == ownNet) {
                // in same network, nothing to do
            } else {
                // merge networks
                allNetworks.mergeNetworks(ownNet, targetNet);
            }
        } else {
            throw new IllegalStateException("This should never happen");
        }
        
        allNetworks.method_80();
        
        updateConnectionsInState(Objects.requireNonNull(getNetwork()));
        targetEntity.updateConnectionsInState(getNetwork());
        
        this.setChanged(false);
        this.sendUpdate(SyncType.CUSTOM);
        
        player.method_43496(class_2561.method_43471("message.oritech.target_designator.connected_poles"));
    }
    
    private void joinNetwork(PoleNetwork target, PoleNetworkData data) {
        data.activeNetworks.put(field_11867, target);
    }
    
    private void updateConnectionsInState(PoleNetwork network) {
        network.setPole(field_11867, connections);
    }
    
    private PoleNetwork createNetwork(PoleNetworkData data) {
        return new PoleNetwork();
    }
    
    public void assignIncomingConnection(PowerPoleEntity from) {
        this.connections.add(from.getConnectionData());
        this.setChanged(false);
        this.sendUpdate(SyncType.CUSTOM);
    }
    
    public void removeIncomingConnection(class_2338 source) {
        
        var removed = this.connections.stream().filter(elem -> elem.pos().equals(source)).toList();
        
        removed.forEach(this.connections::remove);
        
        this.setChanged(false);
        this.sendUpdate(SyncType.CUSTOM);
    }
    
    public ConnectionTarget getConnectionData() {
        return new ConnectionTarget(field_11867, getFacingForMultiblock());
    }
    
    public Set<ConnectionTarget> getConnections() {
        return connections;
    }
    
    public PoleNetworkData getCachedNetData() {
        if (netDataRef == null) {
            netDataRef = POLE_NETWORK_DATA.computeIfAbsent(field_11863.method_27983().method_29177(), data -> new PoleNetworkData());
        }
        
        return netDataRef;
    }
    
    public void onRemoved() {
        
        // remove connection from targets (if loaded)
        for (var target: connections) {
            if (field_11863.method_8477(target.pos) && field_11863.method_8321(target.pos) instanceof PowerPoleEntity powerPole) {
                powerPole.removeIncomingConnection(field_11867);
            }
        }
        
        var allNetworks = getCachedNetData();
        allNetworks.removePole(field_11867);
        allNetworks.method_80();
        
        this.setChanged(false);
        
    }
    
    @Override
    public void setChanged(boolean updateComparator) {
        super.setChanged(updateComparator);
        getCachedNetData().method_80();
    }
    
    private boolean isConnected() {
        return getNetwork() != null;
    }
    
    private PoleNetwork getNetwork() {
        return getCachedNetData().getNetwork(field_11867);
    }
    
    @Override
    protected void method_11007(@NotNull class_2487 tag, class_7225.@NotNull class_7874 registries) {
        super.method_11007(tag, registries);
        addMultiblockToNbt(tag);
        
        var connectionList = new class_2499();
        for (var connection : connections) {
            var compound = new class_2487();
            compound.method_10544("p", connection.pos().method_10063());
            compound.method_10569("d", connection.facing.ordinal());
            connectionList.add(compound);
        }
        tag.method_10566("connectionData", connectionList);
    }
    
    @Override
    protected void method_11014(@NotNull class_2487 tag, class_7225.@NotNull class_7874 registries) {
        super.method_11014(tag, registries);
        loadMultiblockNbtData(tag);
        
        if (tag.method_10545("connectionData")) {
            
            connections.clear();
            
            var nbtList = tag.method_10554("connectionData", class_2520.field_33260);
            for (var nbtElem : nbtList) {
                var elem = (class_2487) nbtElem;
                var pos = class_2338.method_10092(elem.method_10537("p"));
                var dir = class_2350.values()[elem.method_10550("d")];
                connections.add(new ConnectionTarget(pos, dir));
            }
            
        }
    }
    
    @Override
    public void saveExtraData(class_2540 buf) {
        this.sendUpdate(SyncType.GUI_OPEN);
        buf.method_10807(field_11867);
    }
    
    @Override
    public @NotNull class_2561 method_5476() {
        return class_2561.method_43473();
    }
    
    @Override
    public @Nullable class_1703 createMenu(int containerId, @NotNull class_1661 playerInventory, @NotNull class_1657 player) {
        return new UpgradableMachineScreenHandler(containerId, playerInventory, this);
    }
    
    @Override
    public EnergyApi.EnergyStorage getEnergyStorage(class_2350 direction) {
        
        if (direction != null && direction.equals(getFacingForMultiblock().method_10160()))
            return outputStorage;
        
        return energyStorage;
    }
    
    @Override
    public List<class_2382> getCorePositions() {
        return List.of(new class_2382(1, 0, 0), new class_2382(-1, 0, 0));
    }
    
    @Override
    public class_2350 getFacingForMultiblock() {
        return Objects.requireNonNull(field_11863).method_8320(method_11016()).method_11654(class_2741.field_12481).method_10153();
    }
    
    @Override
    public class_2338 getPosForMultiblock() {
        return field_11867;
    }
    
    @Override
    public class_1937 getWorldForMultiblock() {
        return field_11863;
    }
    
    @Override
    public ArrayList<class_2338> getConnectedCores() {
        return coreBlocksConnected;
    }
    
    @Override
    public void setCoreQuality(float quality) {
        this.coreQuality = quality;
    }
    
    @Override
    public float getCoreQuality() {
        return coreQuality;
    }
    
    @Override
    public ItemApi.InventoryStorage getInventoryForMultiblock() {
        return null;
    }
    
    @Override
    public EnergyApi.EnergyStorage getEnergyStorageForMultiblock(class_2350 direction) {
        return energyStorage;
    }
    
    @Override
    public void triggerSetupAnimation() {
    
    }
    
    @Override
    public List<GuiSlot> getGuiSlots() {
        return List.of();
    }
    
    @Override
    public float getDisplayedEnergyUsage() {
        return 0;
    }
    
    @Override
    public float getDisplayedEnergyTransfer() {
        return Oritech.CONFIG.poleConfig.energyCapacity();
    }
    
    @Override
    public BarConfiguration getEnergyConfiguration() {
        return new BarConfiguration(7, 6, 18, 54 + 18);
    }
    
    @Override
    public float getProgress() {
        return 0;
    }
    
    @Override
    public boolean showProgress() {
        return false;
    }
    
    @Override
    public boolean showExpansionPanel() {
        return false;
    }
    
    @Override
    public InventoryInputMode getInventoryInputMode() {
        return InventoryInputMode.FILL_LEFT_TO_RIGHT;
    }
    
    @Override
    public class_1263 getDisplayedInventory() {
        return basicInv;
    }
    
    @Override
    public class_3917<?> getScreenHandlerType() {
        return ModScreens.POWER_POLE_SCREEN;
    }
    
    protected class PowerPoleEnergyStorage extends EnergyApi.EnergyStorage implements UpdatableField<PowerPoleEnergyStorage, Long> {
        
        private long clientShownEnergy;
        
        
        private boolean isValid() {
            return field_11863 != null && PowerPoleEntity.this.isConnected();
        }
        
        @Override
        public long insert(long maxAmount, boolean simulate) {
            if (!isValid()) return 0;
            
            var insertAmount = Math.min(maxAmount, getCapacity() - getAmount());
            
            if (insertAmount > 0 && !simulate) {
                var newAmount = getAmount() + insertAmount;
                setAmount(newAmount);
                getNetwork().inserted.add(insertAmount);
            }
            
            return insertAmount;
        }
        
        @Override
        public long extract(long maxAmount, boolean simulate) {
            if (!isValid()) return 0;
            
            var extractAmount = Math.min(maxAmount, this.getAmount());
            
            if (extractAmount > 0 && !simulate) {
                var newAmount = getAmount() - extractAmount;
                setAmount(newAmount);
                getNetwork().extracted.add(extractAmount);
            }
            
            return extractAmount;
        }
        
        @Override
        public void setAmount(long amount) {
            if (!isValid()) return;
            
            if (amount > getCapacity() || amount < 0) {
                Oritech.LOGGER.error("tried setting invalid amount for pole network: " + amount);
                return;
            }
            
            var network = PowerPoleEntity.this.getNetwork();
            if (network == null) {
                Oritech.LOGGER.error("Invalid set network state for power pole entity at: {}", field_11867);
                return;
            }
            
            network.storedEnergy = amount;
            
        }
        
        @Override
        public long getAmount() {
            if (field_11863.method_8608()) return clientShownEnergy;
            
            if (!isValid()) return 0;
            
            var network = PowerPoleEntity.this.getNetwork();
            if (network == null) {
                Oritech.LOGGER.error("Invalid get network state for power pole entity at: {}", field_11867);
                return 0;
            }
            
            return network.storedEnergy;
        }
        
        @Override
        public long getCapacity() {
            return Oritech.CONFIG.poleConfig.energyCapacity();
        }
        
        @Override
        public void update() {
            if (!isValid()) return;
            PowerPoleEntity.this.setChanged(false);
        }
        
        public void tick(long worldTicks) {
            var net = getNetwork();
            
            if (worldTicks <= net.lastTickedAt) return;
            var index = (int) (worldTicks % 20);
            net.historicInsert[index] = net.inserted.stream().mapToLong(Long::longValue).sum();
            net.historicExtract[index] = net.extracted.stream().mapToLong(Long::longValue).sum();
            net.currentInsertSources = net.inserted.size();
            
            net.inserted.clear();
            net.extracted.clear();
            net.lastTickedAt = worldTicks;
        }
        
        public DynamicStatisticEnergyStorage.EnergyStatistics getCurrentStatistics(long worldTicks) {
            var index = (int) (worldTicks % 20);
            var net = getNetwork();
            
            return new DynamicStatisticEnergyStorage.EnergyStatistics(
              (float) Arrays.stream(net.historicInsert).mapToLong(Long::longValue).average().orElse(0),
              (float) Arrays.stream(net.historicExtract).mapToLong(Long::longValue).average().orElse(0),
              net.historicInsert[index],
              net.historicExtract[index],
              net.currentInsertSources,
              Arrays.stream(net.historicInsert).mapToLong(Long::longValue).max().orElse(0),
              Arrays.stream(net.historicExtract).mapToLong(Long::longValue).max().orElse(0)
            );
            
        }
        
        @Override
        public Long getDeltaData() {
            return getAmount();
        }
        
        @Override
        public PowerPoleEnergyStorage getFullData() {
            return this;
        }
        
        @Override
        public class_9139<? extends ByteBuf, Long> getDeltaCodec() {
            return class_9135.field_48551;
        }
        
        @Override
        public class_9139<? extends ByteBuf, PowerPoleEnergyStorage> getFullCodec() {
            return null;
        }
        
        @Override
        public boolean useDeltaOnly(SyncType type) {
            return true;
        }
        
        @Override
        public void handleFullUpdate(PowerPoleEnergyStorage updatedData) {
        
        }
        
        @Override
        public void handleDeltaUpdate(Long updatedData) {
            this.clientShownEnergy = updatedData;
        }
    }
    
    public record ConnectionTarget(class_2338 pos, class_2350 facing) {}
    
    // this is kept separate from the block entities (and fully decoupled) so it works well across unloaded areas,
    // even if some poles are in the middle of it
    public static class PoleNetworkData extends class_18 {
        
        // runtime lookup map. Pole positions are also stored in the network, so for saving the keys can be reconstructed later here
        private final Map<class_2338, PoleNetwork> activeNetworks = new HashMap<>();
        
        public @NotNull PoleNetwork getNetwork(class_2338 pos) {
            return activeNetworks.computeIfAbsent(pos, elem -> {
                var data = new HashMap<class_2338, Set<class_2338>>();
                data.put(elem, Set.of());
                return new PoleNetwork(data, 0);
            });
        }
        
        public static class_8645<PoleNetworkData> TYPE = new class_8645<>(PoleNetworkData::new, PoleNetworkData::fromNbt, null);
        
        @Override
        public @NotNull class_2487 method_75(@NotNull class_2487 tag, @NotNull class_7225.class_7874 registryLookup) {
            
            var networksList = new class_2499();
            
            var uniqueNetworks = new HashSet<>(activeNetworks.values());
            
            // Iterate the unique networks and save them
            for (var network : uniqueNetworks) {
                
                var networkCompound = new class_2487();
                networkCompound.method_10544("energy", network.storedEnergy);
                
                var poleList = new class_2499();
                for (var polePair : network.poles.entrySet()) {
                    var data = new class_2487();
                    data.method_10544("pos", polePair.getKey().method_10063());
                    data.method_10564("cons", polePair.getValue().stream().mapToLong(class_2338::method_10063).toArray());
                    poleList.add(data);
                }
                
                networkCompound.method_10566("poles", poleList);
                networksList.add(networkCompound);
            }
            
            tag.method_10566("networks", networksList);
            return tag;
        }
        
        public static PoleNetworkData fromNbt(class_2487 nbt, class_7225.class_7874 registryLookup) {
            
            System.out.println("reading power pole data!");
            
            var data = new PoleNetworkData();
            
            if (!nbt.method_10545("networks")) return data;
            
            var networksList = nbt.method_10554("networks", class_2520.field_33260);
            
            for (var networkTag : networksList) {
                
                var tag = (class_2487) networkTag;
                
                var energy = tag.method_10537("energy");
                var poles = new HashMap<class_2338, Set<class_2338>>();
                var poleDataList = tag.method_10554("poles", class_2520.field_33260);
                for (var poleDataTag : poleDataList) {
                    var poleData = (class_2487) poleDataTag;
                    var polePos = class_2338.method_10092(poleData.method_10537("pos"));
                    var poleConnections = new HashSet<>(Arrays.stream(poleData.method_10565("cons")).mapToObj(class_2338::method_10092).toList());
                    poles.put(polePos, poleConnections);
                }
                
                var network = new PoleNetwork(poles, energy);
                
                for (var polePos : network.getPoles())
                    data.activeNetworks.put(polePos, network);
            }
            
            return data;
        }
        
        protected void mergeNetworks(PoleNetwork netA, PoleNetwork netB) {
            
            System.out.println("merging networks");
            
            // move all from netB to netA
            netA.storedEnergy = Math.min(Oritech.CONFIG.poleConfig.energyCapacity(), netA.storedEnergy + netB.storedEnergy);
            
            netA.poles.putAll(netB.poles);
            
            for (var polePos : netB.getPoles()) {
                activeNetworks.put(polePos, netA);
            }
            
        }
        
        public void updateNetworkSplit(Set<class_2338> removedConnections, PoleNetwork existingNet) {
            
            var newNets = new HashSet<Map<class_2338, Set<class_2338>>>();
            
            for (var deletedConnection : removedConnections) {
                var newConnectionNet = FloodFillNetwork(existingNet, deletedConnection);
                newNets.add(newConnectionNet);
                
            }
            
            System.out.println("split into: " + newNets.size());
            
            if (newNets.size() == 1) return;    // no split needed, there's other connections doing the same
            
            var newNetCount = newNets.size();
            var newNetPower = existingNet.storedEnergy / newNetCount;
            
            for (var newNetData : newNets) {
                var newNet = new PoleNetwork(newNetData, newNetPower);
                for (var polePos : newNet.getPoles())
                    activeNetworks.put(polePos, newNet);
            }
        }
        
        public void removePole(class_2338 removeAt) {
            
            System.out.println("removing pole");
            
            // first, updating connections in network data
            var existingNet = activeNetworks.get(removeAt);
            if (existingNet == null) return;
            
            activeNetworks.remove(removeAt);
            
            // for all connections, remove any to the deleted pole
            for (var connections : existingNet.poles.values()) {
                connections.remove(removeAt);
            }
            
            var removedPoleConnections = existingNet.poles.remove(removeAt);
            
            if (removedPoleConnections.size() <= 1) return; // no split needed
            
            updateNetworkSplit(removedPoleConnections, existingNet);
            
        }
        
        // the network is potentially split at this stage. Returns all poles connected to the marked start
        private static Map<class_2338, Set<class_2338>> FloodFillNetwork(PoleNetwork existing, class_2338 startAt) {
            
            var maxIterations = 200;
            var result = new HashMap<class_2338, Set<class_2338>>();
            
            var openChecks = Set.of(startAt);
            
            // basically a while loop, but with an extra safety check
            for (int i = 0; i < maxIterations; i++) {
                
                var next = new HashSet<class_2338>();
                for (var openPole : openChecks) {
                    var connections = existing.getConnections(openPole);
                    result.put(openPole, connections);
                    
                    // add all connections that we dont have already
                    next.addAll(connections.stream().filter(elem -> !result.containsKey(elem)).toList());
                }
                
                if (next.isEmpty()) break;
                
                openChecks = next;
            }
            
            
            return result;
        }
        
    }
    
    // stores the energy in a network. Also includes a list of poles and their connection (only used for floodfill when splitting networks)
    public static class PoleNetwork {
        
        // contains all poles as key, and 0-N positions as value
        private final Map<class_2338, Set<class_2338>> poles;
        
        public long storedEnergy = 0L;
        
        // network stats
        private final List<Long> inserted = new ArrayList<>();  // just for this tick
        private final List<Long> extracted = new ArrayList<>();
        private final Long[] historicInsert = new Long[20];
        private final Long[] historicExtract = new Long[20];
        private int currentInsertSources = 0;
        private long lastTickedAt = 0;
        
        // constructor for codec
        private PoleNetwork(Map<class_2338, Set<class_2338>> loadedPoles, long storedEnergy) {
            this.poles = new HashMap<>(loadedPoles);
            this.storedEnergy = storedEnergy;
            Arrays.fill(historicInsert, 0L);
            Arrays.fill(historicExtract, 0L);
        }
        
        // default constructor
        public PoleNetwork() {
            this.poles = new HashMap<>();
            Arrays.fill(historicInsert, 0L);
            Arrays.fill(historicExtract, 0L);
        }
        
        public Set<class_2338> getPoles() {
            return poles.keySet();
        }
        
        public Set<class_2338> getConnections(class_2338 polePos) {
            return poles.get(polePos);
        }
        
        // adds or updates a pole in a network
        public void setPole(class_2338 pole, Set<ConnectionTarget> connections) {
            poles.put(pole, new HashSet<>(connections.stream().map(elem -> elem.pos()).toList()));
        }
    }
}
