package rearth.oritech.block.entity.interaction;

import dev.architectury.fluid.FluidStack;
import dev.architectury.hooks.fluid.FluidStackHooks;
import dev.architectury.registry.menu.ExtendedMenuProvider;
import org.jetbrains.annotations.Nullable;
import rearth.oritech.Oritech;
import rearth.oritech.api.energy.EnergyApi;
import rearth.oritech.api.energy.containers.DynamicEnergyStorage;
import rearth.oritech.api.fluid.FluidApi;
import rearth.oritech.api.fluid.containers.SimpleFluidStorage;
import rearth.oritech.api.item.ItemApi;
import rearth.oritech.api.item.containers.SimpleInventoryStorage;
import rearth.oritech.api.networking.NetworkedBlockEntity;
import rearth.oritech.api.networking.SyncField;
import rearth.oritech.api.networking.SyncType;
import rearth.oritech.block.base.entity.MachineBlockEntity;
import rearth.oritech.block.blocks.interaction.DronePortBlock;
import rearth.oritech.block.blocks.processing.MachineCoreBlock;
import rearth.oritech.block.entity.MachineCoreEntity;
import rearth.oritech.block.entity.addons.CombiAddonEntity;
import rearth.oritech.block.entity.addons.RedstoneAddonBlockEntity;
import rearth.oritech.client.init.ModScreens;
import rearth.oritech.client.ui.DroneScreenHandler;
import rearth.oritech.init.BlockContent;
import rearth.oritech.init.BlockEntitiesContent;
import rearth.oritech.init.ComponentContent;
import rearth.oritech.item.tools.LaserTargetDesignator;
import rearth.oritech.util.*;
import software.bernie.geckolib.animatable.GeoBlockEntity;
import software.bernie.geckolib.animatable.instance.AnimatableInstanceCache;
import software.bernie.geckolib.animation.AnimatableManager;
import software.bernie.geckolib.animation.AnimationController;
import software.bernie.geckolib.animation.PlayState;
import software.bernie.geckolib.animation.RawAnimation;
import software.bernie.geckolib.util.GeckoLibUtil;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import net.minecraft.class_1262;
import net.minecraft.class_1263;
import net.minecraft.class_1277;
import net.minecraft.class_1657;
import net.minecraft.class_1661;
import net.minecraft.class_1703;
import net.minecraft.class_1799;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2371;
import net.minecraft.class_2382;
import net.minecraft.class_2487;
import net.minecraft.class_2509;
import net.minecraft.class_2540;
import net.minecraft.class_2561;
import net.minecraft.class_2680;
import net.minecraft.class_2741;
import net.minecraft.class_3917;
import net.minecraft.class_7225;

import static rearth.oritech.block.base.block.MultiblockMachine.ASSEMBLED;


public class DronePortEntity extends NetworkedBlockEntity
  implements ItemApi.BlockProvider, FluidApi.BlockProvider, EnergyApi.BlockProvider,
               GeoBlockEntity, MultiblockMachineController, MachineAddonController, ExtendedMenuProvider,
               ScreenProvider, RedstoneAddonBlockEntity.RedstoneControllable, ColorableMachine {
    
    // addon data
    @SyncField(SyncType.GUI_OPEN)
    private final List<class_2338> connectedAddons = new ArrayList<>();
    @SyncField(SyncType.GUI_OPEN)
    private final List<class_2338> openSlots = new ArrayList<>();
    @SyncField(SyncType.GUI_OPEN)
    private BaseAddonData addonData = BaseAddonData.DEFAULT_ADDON_DATA;
    @SyncField({SyncType.SPARSE_TICK, SyncType.INITIAL})
    public ColorableMachine.ColorVariant currentColor = getDefaultColor();
    
    // storage
    @SyncField({SyncType.GUI_OPEN, SyncType.GUI_TICK})
    protected final DynamicEnergyStorage energyStorage = new DynamicEnergyStorage(1024 * 32, 10000, 0, this::method_5431);
    
    public final DronePortItemInventory inventory = new DronePortItemInventory(15, this::method_5431);
    
    @SyncField(SyncType.GUI_TICK)
    public final DronePortFluidStorage fluidStorage = new DronePortFluidStorage(128 * FluidStackHooks.bucketAmount(), this::method_5431);
    
    // not persisted, only to assign targets
    protected final class_1277 cardInventory = new class_1277(2) {
        @Override
        public void method_5431() {
            DronePortEntity.this.method_5431();
        }
        
        @Override
        public boolean method_27070(class_1799 stack) {
            return stack.method_7909() instanceof LaserTargetDesignator;
        }
    };
    
    @SyncField(SyncType.GUI_OPEN)
    private float coreQuality = 1f;
    
    // animation
    protected final AnimatableInstanceCache animatableInstanceCache = GeckoLibUtil.createInstanceCache(this);
    
    // multiblock
    private final ArrayList<class_2338> coreBlocksConnected = new ArrayList<>();
    
    // fluid
    @SyncField(SyncType.GUI_OPEN)
    public boolean hasFluidAddon;
    
    // redstone
    @SyncField(SyncType.GUI_OPEN)
    public boolean disabledViaRedstone;
    
    // work data
    private class_2338 targetPosition;
    private long lastSentAt;
    private DroneTransferData incomingPacket;
    private boolean receivingPackage;
    
    // config
    private final long baseEnergyUsage = 1024;
    private final int takeOffTime = 300;
    private final int landTime = 260;
    private final int totalFlightTime = takeOffTime + landTime;
    
    // client only
    @SyncField(SyncType.GUI_TICK)
    private String statusMessage = "";
    
    public DronePortEntity(class_2338 pos, class_2680 state) {
        super(BlockEntitiesContent.DRONE_PORT_ENTITY, pos, state);
    }
    
    @Override
    public void serverTick(class_1937 world, class_2338 pos, class_2680 state, NetworkedBlockEntity blockEntity) {
        checkPositionCard();
        
        if (incomingPacket != null)
            checkIncomingAnimation();
        
        if (world.method_8510() % 20 == 0) {
            if (incomingPacket != null) {
                tryReceivePacket();
            } else if (canSend()) {
                sendDrone();
            }
        }
    }
    
    private void checkPositionCard() {
        
        var source = cardInventory.method_54454().get(0);
        if (source.method_7909() instanceof LaserTargetDesignator && source.method_57826(ComponentContent.TARGET_POSITION.get())) {
            var target = source.method_57824(ComponentContent.TARGET_POSITION.get());
            setTargetFromDesignator(target);
        } else {
            return;
        }
        
        cardInventory.method_54454().set(1, source);
        cardInventory.method_54454().set(0, class_1799.field_8037);
        cardInventory.method_5431();
        this.method_5431();
        
    }
    
    @Override
    protected void method_11007(class_2487 nbt, class_7225.class_7874 registryLookup) {
        super.method_11007(nbt, registryLookup);
        class_1262.method_5427(nbt, inventory.getHeldStacks(), false, registryLookup);
        addMultiblockToNbt(nbt);
        writeAddonToNbt(nbt);
        addColorToNbt(nbt);
        fluidStorage.writeNbt(nbt, "");
        nbt.method_10556("has_fluid_addon", hasFluidAddon);
        nbt.method_10556("disabled_via_redstone", disabledViaRedstone);
        nbt.method_10544("energy_stored", energyStorage.amount);
        
        if (targetPosition != null) {
            nbt.method_10544("target_position", targetPosition.method_10063());
        }
        
        var cardCompound = new class_2487();
        class_1262.method_5427(cardCompound, cardInventory.method_54454(), false, registryLookup);
        nbt.method_10566("cards", cardCompound);
        
        if (incomingPacket != null) {
            var compound = new class_2487();
            class_2371<class_1799> list = class_2371.method_37434(incomingPacket.transferredStacks.size());
            list.addAll(incomingPacket.transferredStacks);
            class_1262.method_5427(compound, list, false, registryLookup);
            nbt.method_10566("incoming", compound);
            FluidStack.CODEC.encodeStart(class_2509.field_11560, incomingPacket.movedFluid).result().ifPresent(tag -> nbt.method_10566("fluidmoving", tag));
            nbt.method_10544("incomingTime", incomingPacket.arrivesAt);
        } else {
            nbt.method_10551("incoming");
        }
    }
    
    @Override
    protected void method_11014(class_2487 nbt, class_7225.class_7874 registryLookup) {
        super.method_11014(nbt, registryLookup);
        class_1262.method_5429(nbt, inventory.getHeldStacks(), registryLookup);
        loadMultiblockNbtData(nbt);
        loadAddonNbtData(nbt);
        loadColorFromNbt(nbt);
        fluidStorage.readNbt(nbt, "");
        
        hasFluidAddon = nbt.method_10577("has_fluid_addon");
        disabledViaRedstone = nbt.method_10577("disabled_via_redstone");
        energyStorage.amount = nbt.method_10537("energy_stored");
        targetPosition = class_2338.method_10092(nbt.method_10537("target_position"));
        
        class_1262.method_5429(nbt.method_10562("cards"), cardInventory.method_54454(), registryLookup);
        
        if (nbt.method_10545("incoming")) {
            class_2371<class_1799> list = class_2371.method_10213(15, class_1799.field_8037);
            class_1262.method_5429(nbt.method_10562("incoming"), list, registryLookup);
            var fluid = FluidStack.CODEC.parse(class_2509.field_11560, nbt.method_10580("fluidmoving")).result().orElse(FluidStack.empty());
            var arrivalTime = nbt.method_10537("incomingTime");
            incomingPacket = new DroneTransferData(list, fluid, arrivalTime);
        }
    }
    
    @Override
    public void initAddons() {
        MachineAddonController.super.initAddons();
        
        // Trigger block updates for pipes to connect
        field_11863.method_8408(field_11867, method_11010().method_26204());
        for (class_2382 corePosition : getCorePositions()) {
            var worldPos = new class_2338(Geometry.offsetToWorldPosition(getFacingForMultiblock(), corePosition, getPosForAddon()));
            field_11863.method_8408(worldPos, field_11863.method_8320(worldPos).method_26204());
        }
    }
    
    @Override
    public void getAdditionalStatFromAddon(AddonBlock addonBlock) {
        if (addonBlock.state().method_26204().equals(BlockContent.MACHINE_FLUID_ADDON) || addonBlock.addonEntity() instanceof CombiAddonEntity combi && combi.hasFluid()) {
            hasFluidAddon = true;
        }
    }
    
    @Override
    public void resetAddons() {
        MachineAddonController.super.resetAddons();
        hasFluidAddon = false;
    }
    
    private void checkIncomingAnimation() {
        if (field_11863.method_8510() == incomingPacket.arrivesAt - landTime) {
            triggerNetworkReceiveAnimation();
        }
    }
    
    private void tryReceivePacket() {
        var hasArrived = field_11863.method_8510() - incomingPacket.arrivesAt > 0;
        if (!hasArrived) return;
        
        Oritech.LOGGER.debug("receiving drone package: " + incomingPacket);
        
        receivingPackage = true;
        long totalToInsert = incomingPacket.transferredStacks.stream().mapToLong(class_1799::method_7947).sum();
        long totalInserted = 0;
        for (var stack : incomingPacket.transferredStacks) {
            totalInserted += inventory.insert(stack, false);
        }
        
        if (totalInserted != totalToInsert) {
            Oritech.LOGGER.warn("Something weird has happened with drone port item storage. Caused at: " + field_11867);
            return;
        }
        
        if (!incomingPacket.movedFluid.isEmpty()) {
            fluidStorage.insertFromDrone(incomingPacket.movedFluid, false);
        }
        
        receivingPackage = false;
        incomingPacket = null;
        method_5431();
    }
    
    private void sendDrone() {
        var targetPort = (DronePortEntity) field_11863.method_8321(targetPosition);
        var arriveTime = field_11863.method_8510() + takeOffTime + landTime;
        var data = new DroneTransferData(inventory.getHeldStacks().stream().filter(stack -> !stack.method_7960()).toList(), fluidStorage.getStack(), arriveTime);
        targetPort.setIncomingPacket(data);
        
        inventory.method_5448();
        fluidStorage.setStack(FluidStack.empty());
        lastSentAt = field_11863.method_8510();
        energyStorage.amount -= calculateEnergyUsage();
        
        triggerNetworkSendAnimation();
        targetPort.method_5431();
        this.method_5431();
        
        Oritech.LOGGER.debug("sending drone package: " + data);
    }
    
    public boolean canAcceptPayload(List<class_1799> stacks, FluidStack fluid) {
        
        // fail if items are incoming and inventory is not empty
        if (!stacks.isEmpty() && !inventory.method_5442())
            return false;
        
        // fail if fluid is incoming and would not match
        if (!fluid.isEmpty() && (!hasFluidAddon || fluidStorage.insert(fluid, true) != fluid.getAmount())) {
            return false;
        }
        
        return true;
    }
    
    /**
     * Check if the drone is currently sending a package
     * Drone will be in a sending state for a certain amount of time after sending a package
     * (time it takes to take off)
     *
     * @return true if drone is sending a package
     */
    public boolean isSendingDrone() {
        var diff = field_11863.method_8510() - lastSentAt;
        return diff < takeOffTime;
    }
    
    private boolean canSend() {
        
        if (disabledViaRedstone || targetPosition == null || (inventory.method_5442() && fluidStorage.getAmount() == 0) || energyStorage.amount < calculateEnergyUsage() || incomingPacket != null)
            return false;
        var targetEntity = field_11863.method_8321(targetPosition);
        if (!(targetEntity instanceof DronePortEntity targetPort) || targetPort.disabledViaRedstone || targetPort.getIncomingPacket() != null || !targetPort.canAcceptPayload(inventory.getHeldStacks(), fluidStorage.getStack()))
            return false;
        
        
        return !isSendingDrone();
    }
    
    private long calculateEnergyUsage() {
        if (targetPosition == null) return baseEnergyUsage;
        var distance = field_11867.method_19455(targetPosition);
        return (long) Math.sqrt(distance) * 50 + baseEnergyUsage;
    }
    
    @Override
    public ColorVariant getCurrentColor() {
        return currentColor;
    }
    
    @Override
    public void assignColor(ColorVariant color) {
        this.currentColor = color;
        
        if (this.field_11863 != null && !this.field_11863.method_8608()) {
            this.setChanged(false);
            this.sendUpdate(SyncType.SPARSE_TICK);
        }
    }
    
    private void triggerNetworkSendAnimation() {
        triggerAnim("machine", "takeoff");
    }
    
    private void triggerNetworkReceiveAnimation() {
        triggerAnim("machine", "landing");
    }
    
    public boolean setTargetFromDesignator(class_2338 targetPos) {
        
        // if target is coreblock, adjust it to point to controller if connected
        var targetState = Objects.requireNonNull(field_11863).method_8320(targetPos);
        if (targetState.method_26204() instanceof MachineCoreBlock && targetState.method_11654(MachineCoreBlock.USED)) {
            var coreEntity = (MachineCoreEntity) field_11863.method_8321(targetPos);
            var controllerPos = Objects.requireNonNull(coreEntity).getControllerPos();
            if (controllerPos != null) targetPos = controllerPos;
        }
        
        var distance = targetPos.method_19455(field_11867);
        if (distance < 50) {
            statusMessage = "message.oritech.drone.invalid_distance";
            return false;
        }
        
        if (field_11863.method_8320(targetPos).method_26204() instanceof DronePortBlock) {
            // store position
            this.targetPosition = targetPos;
            statusMessage = "message.oritech.drone.target_set";
            return true;
        }
        
        statusMessage = "message.oritech.drone.target_invalid";
        return false;
        
    }
    
    @Override
    public EnergyApi.EnergyStorage getEnergyStorage(class_2350 direction) {
        return energyStorage;
    }
    
    @Override
    public ItemApi.InventoryStorage getInventoryStorage(class_2350 direction) {
        return inventory;
    }
    
    @Override
    public List<class_2382> getCorePositions() {
        return List.of(
          new class_2382(0, 0, 1),
          new class_2382(0, 0, -1),
          new class_2382(-1, 0, 1),
          new class_2382(-1, 0, 0),
          new class_2382(-1, 0, -1),
          new class_2382(-2, 0, 1),
          new class_2382(-2, 0, 0),
          new class_2382(-2, 0, -1),
          new class_2382(0, 1, 0),
          new class_2382(0, 1, 1),
          new class_2382(-1, 1, -1)
        );
    }
    
    @Override
    public class_2350 getFacingForMultiblock() {
        return Objects.requireNonNull(field_11863).method_8320(method_11016()).method_11654(class_2741.field_12481).method_10153();
    }
    
    @Override
    public class_2338 getPosForAddon() {
        return field_11867;
    }
    
    @Override
    public class_1937 getWorldForAddon() {
        return field_11863;
    }
    
    @Override
    public ArrayList<class_2338> getConnectedCores() {
        return coreBlocksConnected;
    }
    
    @Override
    public void setCoreQuality(float quality) {
        this.coreQuality = quality;
    }
    
    @Override
    public float getCoreQuality() {
        return coreQuality;
    }
    
    @Override
    public ItemApi.InventoryStorage getInventoryForMultiblock() {
        return inventory;
    }
    
    @Override
    public EnergyApi.EnergyStorage getEnergyStorageForMultiblock(class_2350 direction) {
        return energyStorage;
    }
    
    @Override
    public @Nullable FluidApi.FluidStorage getFluidStorage(class_2350 direction) {
        return hasFluidAddon ? fluidStorage : null;
    }
    
    @Override
    public List<class_2382> getAddonSlots() {
        return List.of(
          new class_2382(3, 0, -1),
          new class_2382(2, 0, -2)
        );
    }
    
    @Override
    public long getDefaultCapacity() {
        return 1024 * 32;
    }
    
    @Override
    public long getDefaultInsertRate() {
        return 512;
    }
    
    @Override
    public ItemApi.InventoryStorage getInventoryForAddon() {
        return inventory;
    }
    
    @Override
    public ScreenProvider getScreenProvider() {
        return this;
    }
    
    public DynamicEnergyStorage getEnergyStorage() {
        return energyStorage;
    }
    
    @Override
    public List<class_2338> getConnectedAddons() {
        return connectedAddons;
    }
    
    @Override
    public List<class_2338> getOpenAddonSlots() {
        return openSlots;
    }
    
    @Override
    public class_2350 getFacingForAddon() {
        return Objects.requireNonNull(field_11863).method_8320(method_11016()).method_11654(class_2741.field_12481);
    }
    
    @Override
    public DynamicEnergyStorage getStorageForAddon() {
        return getEnergyStorage();
    }
    
    @Override
    public BaseAddonData getBaseAddonData() {
        return addonData;
    }
    
    @Override
    public void setBaseAddonData(BaseAddonData data) {
        this.addonData = data;
        this.method_5431();
    }
    
    public DroneTransferData getIncomingPacket() {
        return incomingPacket;
    }
    
    public void setIncomingPacket(DroneTransferData incomingPacket) {
        this.incomingPacket = incomingPacket;
    }
    
    public boolean isActive(class_2680 state) {
        return state.method_11654(ASSEMBLED);
    }
    
    @Override
    public void triggerSetupAnimation() {
        triggerAnim("machine", "deploy");
    }
    
    public static final RawAnimation TAKEOFF = RawAnimation.begin().thenPlay("takeoff");
    public static final RawAnimation LANDING = RawAnimation.begin().thenPlay("landing");
    
    @Override
    public void registerControllers(AnimatableManager.ControllerRegistrar controllers) {
        controllers.add(new AnimationController<>(this, "machine", 0, state -> {
            if (state.getController().getAnimationState().equals(AnimationController.State.STOPPED)) {
                var targetAnim = isActive(method_11010()) ? MachineBlockEntity.IDLE : MachineBlockEntity.PACKAGED;
                state.resetCurrentAnimation();
                return state.setAndContinue(targetAnim);
            } else {
                // playing animation, keep going
                return PlayState.CONTINUE;
            }
        })
                          .triggerableAnim("takeoff", TAKEOFF)
                          .triggerableAnim("landing", LANDING)
                          .triggerableAnim("deploy", MachineBlockEntity.SETUP)
                          .setSoundKeyframeHandler(new AutoPlayingSoundKeyframeHandler<>()));
    }
    
    @Override
    public AnimatableInstanceCache getAnimatableInstanceCache() {
        return animatableInstanceCache;
    }
    
    @Override
    public int getComparatorEnergyAmount() {
        return (int) ((energyStorage.amount / (float) energyStorage.capacity) * 15);
    }
    
    @Override
    public int getComparatorSlotAmount(int slot) {
        if (inventory.getHeldStacks().size() <= slot)
            return hasFluidAddon ? ComparatorOutputProvider.getFluidStorageComparatorOutput(fluidStorage) : 0;
        
        var stack = inventory.method_5438(slot);
        if (stack.method_7960()) return
                               hasFluidAddon ? ComparatorOutputProvider.getFluidStorageComparatorOutput(fluidStorage) : 0;
        
        return hasFluidAddon ?
                 Math.max(ComparatorOutputProvider.getItemStackComparatorOutput(stack), ComparatorOutputProvider.getFluidStorageComparatorOutput(fluidStorage)) :
                 ComparatorOutputProvider.getItemStackComparatorOutput(stack);
    }
    
    @Override
    public int getComparatorProgress() {
        if (isSendingDrone()) {
            return (int) (((field_11863.method_8510() - lastSentAt) / (float) takeOffTime) * 15);
        } else if (incomingPacket != null) {
            return (int) ((totalFlightTime + (field_11863.method_8510() - incomingPacket.arrivesAt)) / (float) (totalFlightTime) * 15);
        } else {
            return 0;
        }
    }
    
    @Override
    public int getComparatorActiveState() {
        return isSendingDrone() || incomingPacket != null ? 15 : 0;
    }
    
    @Override
    public void onRedstoneEvent(boolean isPowered) {
        this.disabledViaRedstone = isPowered;
    }
    
    @Override
    public int receivedRedstoneSignal() {
        if (disabledViaRedstone) return 15;
        return 0;
    }
    
    @Override
    public String currentRedstoneEffect() {
        if (disabledViaRedstone) return "tooltip.oritech.redstone_disabled";
        return "tooltip.oritech.redstone_enabled";
    }
    
    @Override
    public boolean hasRedstoneControlAvailable() {
        return true;
    }
    
    @Override
    public void saveExtraData(class_2540 buf) {
        sendUpdate(SyncType.GUI_OPEN);
        buf.method_10807(field_11867);
    }
    
    @Override
    public class_2561 method_5476() {
        return class_2561.method_30163("");
    }
    
    @Nullable
    @Override
    public class_1703 createMenu(int syncId, class_1661 playerInventory, class_1657 player) {
        return new DroneScreenHandler(syncId, playerInventory, this);
    }
    
    @Override
    public List<GuiSlot> getGuiSlots() {
        
        var startX = 30;
        var startY = 26;
        var distance = 18;
        
        var list = new ArrayList<GuiSlot>();
        for (int y = 0; y < 3; y++) {
            for (int x = 0; x < 5; x++) {
                var index = y * 5 + x;
                list.add(new GuiSlot(index, startX + x * distance, startY + y * distance));
            }
        }
        
        return list;
    }
    
    @Override
    public float getDisplayedEnergyUsage() {
        return calculateEnergyUsage();
    }
    
    @Override
    public float getDisplayedEnergyTransfer() {
        return energyStorage.maxInsert;
    }
    
    @Override
    public float getProgress() {
        return 0;
    }
    
    @Override
    public InventoryInputMode getInventoryInputMode() {
        return InventoryInputMode.FILL_LEFT_TO_RIGHT;
    }
    
    @Override
    public class_1263 getDisplayedInventory() {
        return inventory;
    }
    
    @Override
    public class_3917<?> getScreenHandlerType() {
        return ModScreens.DRONE_SCREEN;
    }
    
    @Override
    public boolean inputOptionsEnabled() {
        return false;
    }
    
    @Override
    public boolean showProgress() {
        return false;
    }
    
    public class_1277 getCardInventory() {
        return cardInventory;
    }
    
    public String getStatusMessage() {
        return statusMessage;
    }
    
    @Override
    public class_2338 getPosForMultiblock() {
        return field_11867;
    }
    
    @Override
    public class_1937 getWorldForMultiblock() {
        return field_11863;
    }
    
    public record DroneTransferData(List<class_1799> transferredStacks, FluidStack movedFluid, long arrivesAt) {
    }
    
    public class DronePortItemInventory extends SimpleInventoryStorage {
        
        public DronePortItemInventory(int size, Runnable onUpdate) {
            super(size, onUpdate);
        }
        
        @Override
        public int insertToSlot(class_1799 addedStack, int slot, boolean simulate) {
            if (DronePortEntity.this.incomingPacket != null && !receivingPackage) return 0;
            return super.insertToSlot(addedStack, slot, simulate);
        }
    }
    
    public class DronePortFluidStorage extends SimpleFluidStorage {
        
        public DronePortFluidStorage(Long capacity, Runnable onUpdate) {
            super(capacity, onUpdate);
        }
        
        @Override
        public long insert(FluidStack toInsert, boolean simulate) {
            if (DronePortEntity.this.incomingPacket != null) return 0;
            return super.insert(toInsert, simulate);
        }
        
        /**
         * Insert from drone, bypasses the incoming packet check
         */
        public long insertFromDrone(FluidStack toInsert, boolean simulate) {
            return super.insert(toInsert, simulate);
        }
    }
}
