package rearth.oritech.block.base.entity;

import org.jetbrains.annotations.Nullable;
import rearth.oritech.Oritech;
import rearth.oritech.api.energy.containers.DynamicEnergyStorage;
import rearth.oritech.api.item.ItemApi;
import rearth.oritech.api.networking.NetworkedBlockEntity;
import rearth.oritech.api.networking.SyncField;
import rearth.oritech.api.networking.SyncType;
import rearth.oritech.client.ui.UpgradableMachineScreenHandler;
import rearth.oritech.init.recipes.OritechRecipe;
import rearth.oritech.util.MachineAddonController;
import rearth.oritech.util.ScreenProvider;

import java.util.ArrayList;
import java.util.List;
import net.minecraft.class_1657;
import net.minecraft.class_1661;
import net.minecraft.class_1703;
import net.minecraft.class_1799;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2487;
import net.minecraft.class_2591;
import net.minecraft.class_2680;
import net.minecraft.class_7225;

public abstract class UpgradableMachineBlockEntity extends MachineBlockEntity implements MachineAddonController {
    
    @SyncField(SyncType.GUI_OPEN)
    private final List<class_2338> connectedAddons = new ArrayList<>();
    @SyncField(SyncType.GUI_OPEN)
    private final List<class_2338> openSlots = new ArrayList<>();
    @SyncField(SyncType.GUI_OPEN)
    private BaseAddonData addonData = BaseAddonData.DEFAULT_ADDON_DATA;
    
    @SyncField({SyncType.GUI_TICK, SyncType.GUI_OPEN})
    public int remainingBurstTicks = 0;
    
    public UpgradableMachineBlockEntity(class_2591<?> type, class_2338 pos, class_2680 state, int energyPerTick) {
        super(type, pos, state, energyPerTick);
    }
    
    @Override
    protected void useEnergy() {
        super.useEnergy();
        
        consumeBurstTicks();
        
    }
    
    public void consumeBurstTicks() {
        // consume burst tick with each tick that we progress (which uses energy once)
        remainingBurstTicks -= 2;
        remainingBurstTicks = Math.max(remainingBurstTicks, -addonData.maxBurstTicks());
    }
    
    @Override
    public void serverTick(class_1937 world, class_2338 pos, class_2680 state, NetworkedBlockEntity blockEntity) {
        super.serverTick(world, pos, state, blockEntity);
        
        addBurstTicks();
    }
    
    public void addBurstTicks() {
        remainingBurstTicks++;
        remainingBurstTicks = Math.min(remainingBurstTicks, addonData.maxBurstTicks());
    }
    
    @Override
    protected void craftItem(OritechRecipe activeRecipe, List<class_1799> outputInventory, List<class_1799> inputInventory) {
        super.craftItem(activeRecipe, outputInventory, inputInventory);
        
        if (supportExtraChambersAuto()) {
            var chamberCount = addonData.extraChambers();
            
            // craft N extra items if we have extra chambers
            for (int i = 0; i < chamberCount; i++) {
                var newRecipe = getRecipe();
                if (newRecipe.isEmpty() || !newRecipe.get().comp_1933().equals(currentRecipe) || !canOutputRecipe(activeRecipe) || !canProceed(activeRecipe))
                    break;
                super.craftItem(activeRecipe, outputInventory, inputInventory);
            }
        }
        
    }
    
    // this should return false if the default craftItem implementation should not handle extra chambers
    public boolean supportExtraChambersAuto() {
        return true;
    }
    
    @Override
    protected void method_11007(class_2487 nbt, class_7225.class_7874 registryLookup) {
        super.method_11007(nbt, registryLookup);
        writeAddonToNbt(nbt);
    }
    
    @Override
    protected void method_11014(class_2487 nbt, class_7225.class_7874 registryLookup) {
        super.method_11014(nbt, registryLookup);
        loadAddonNbtData(nbt);
        
        updateEnergyContainer();
    }
    
    @Override
    public List<class_2338> getConnectedAddons() {
        return connectedAddons;
    }
    
    @Override
    public List<class_2338> getOpenAddonSlots() {
        return openSlots;
    }
    
    @Override
    public class_2350 getFacingForAddon() {
        return super.getFacing();
    }
    
    @Override
    public DynamicEnergyStorage getStorageForAddon() {
        return super.getEnergyStorage();
    }
    
    @Override
    public BaseAddonData getBaseAddonData() {
        return addonData;
    }
    
    
    @Override
    public class_2338 getPosForAddon() {
        return method_11016();
    }
    
    @Override
    public class_1937 getWorldForAddon() {
        return method_10997();
    }
    
    @Override
    public void setBaseAddonData(BaseAddonData data) {
        this.addonData = data;
        this.method_5431();
    }
    
    @Nullable
    @Override
    public class_1703 createMenu(int syncId, class_1661 playerInventory, class_1657 player) {
        return new UpgradableMachineScreenHandler(syncId, playerInventory, this);
    }
    
    @Override
    public ItemApi.InventoryStorage getInventoryForAddon() {
        return inventory;
    }
    
    @Override
    public ScreenProvider getScreenProvider() {
        return this;
    }
    
    public boolean isBurstAvailable() {
        return remainingBurstTicks > 0;
    }
    
    public boolean isBurstThrottled() {
        return remainingBurstTicks < 0;
    }
    
    public float getBurstBonus() {
        if (isBurstAvailable()) {
            return 1 / Oritech.CONFIG.addonConfig.burstAddonSpeedMultiplier();
        } else if(isBurstThrottled()) {
            return Oritech.CONFIG.addonConfig.burstAddonThrottleMultiplier();
        } else {
            return 1f;
        }
    }
    
    // values smaller than 1 are faster, higher than 1 are slower
    @Override
    public float getSpeedMultiplier() {
        return getBaseAddonData().speed() * getBurstBonus();
    }
    
    // same here
    @Override
    public float getEfficiencyMultiplier() {
        return getBaseAddonData().efficiency() * getBurstBonus();
    }
    
    @Override
    public int receivedRedstoneSignal() {
        if (disabledViaRedstone) return 15;
        return 0;
    }
    
    @Override
    public String currentRedstoneEffect() {
        if (disabledViaRedstone) return "tooltip.oritech.redstone_disabled";
        return "tooltip.oritech.redstone_enabled";
    }
}
