package rearth.oritech.block.entity.processing;

import dev.architectury.fluid.FluidStack;
import dev.architectury.hooks.fluid.FluidStackHooks;
import org.jetbrains.annotations.Nullable;
import rearth.oritech.Oritech;
import rearth.oritech.api.fluid.FluidApi;
import rearth.oritech.api.fluid.FluidApi.SingleSlotStorage;
import rearth.oritech.api.fluid.containers.SimpleFluidStorage;
import rearth.oritech.api.fluid.containers.SimpleInOutFluidStorage;
import rearth.oritech.api.networking.NetworkedBlockEntity;
import rearth.oritech.api.networking.SyncField;
import rearth.oritech.api.networking.SyncType;
import rearth.oritech.block.base.entity.MultiblockMachineEntity;
import rearth.oritech.client.init.ModScreens;
import rearth.oritech.client.init.ParticleContent;
import rearth.oritech.client.ui.RefineryScreenHandler;
import rearth.oritech.init.BlockEntitiesContent;
import rearth.oritech.init.recipes.OritechRecipe;
import rearth.oritech.init.recipes.OritechRecipeType;
import rearth.oritech.init.recipes.RecipeContent;
import rearth.oritech.util.Geometry;
import rearth.oritech.util.InventorySlotAssignment;

import java.util.Comparator;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import net.minecraft.class_1657;
import net.minecraft.class_1661;
import net.minecraft.class_1703;
import net.minecraft.class_1799;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2382;
import net.minecraft.class_243;
import net.minecraft.class_2487;
import net.minecraft.class_2561;
import net.minecraft.class_2680;
import net.minecraft.class_3545;
import net.minecraft.class_3917;
import net.minecraft.class_7225;
import net.minecraft.class_8786;

public class RefineryBlockEntity extends MultiblockMachineEntity implements FluidApi.BlockProvider {
    
    // own storage is exposed through this multiblock, the other storages are exposed through the respective modules
    @SyncField({SyncType.GUI_TICK, SyncType.SPARSE_TICK, SyncType.INITIAL})
    public final SimpleInOutFluidStorage ownStorage = new SimpleInOutFluidStorage(64 * FluidStackHooks.bucketAmount(), this::method_5431);
    @SyncField({SyncType.GUI_TICK, SyncType.SPARSE_TICK, SyncType.INITIAL})
    public final SimpleFluidStorage nodeA = new SimpleFluidStorage(4 * FluidStackHooks.bucketAmount(), this::method_5431);
    @SyncField({SyncType.GUI_TICK, SyncType.SPARSE_TICK, SyncType.INITIAL})
    public final SimpleFluidStorage nodeB = new SimpleFluidStorage(4 * FluidStackHooks.bucketAmount(), this::method_5431);
    
    @SyncField(SyncType.GUI_OPEN)
    private int moduleCount;    // range 0-2
    
    public RefineryBlockEntity(class_2338 pos, class_2680 state) {
        super(BlockEntitiesContent.REFINERY_ENTITY, pos, state, Oritech.CONFIG.processingMachines.refineryData.energyPerTick());
    }
    
    @Override
    public void serverTick(class_1937 world, class_2338 pos, class_2680 state, NetworkedBlockEntity blockEntity) {
        super.serverTick(world, pos, state, blockEntity);
        
        if (world.method_8510() % 25 == 0) {
            refreshModules();
        }
    }
    
    @Override
    protected void method_11007(class_2487 nbt, class_7225.class_7874 registryLookup) {
        super.method_11007(nbt, registryLookup);
        ownStorage.writeNbt(nbt, "main");
        nodeA.writeNbt(nbt, "a");
        nodeB.writeNbt(nbt, "b");
    }
    
    @Override
    protected void method_11014(class_2487 nbt, class_7225.class_7874 registryLookup) {
        super.method_11014(nbt, registryLookup);
        ownStorage.readNbt(nbt, "main");
        nodeA.readNbt(nbt, "a");
        nodeB.readNbt(nbt, "b");
    }
    
    private void refreshModules() {
        moduleCount = 0;
        var startPos = field_11867.method_10086(2);
        
        for (int i = 0; i <= 1; i++) {
            var candidatePos = startPos.method_10069(0, i, 0);
            var candidate = field_11863.method_35230(candidatePos, BlockEntitiesContent.REFINERY_MODULE_ENTITY);
            if (candidate.isEmpty() || !candidate.get().isActive(candidate.get().method_11010())) break;
            
            moduleCount++;
            candidate.get().setOwningRefinery(this);
        }
    }
    
    public int getModuleCount() {
        return moduleCount;
    }
    
    @Override
    protected Optional<class_8786<OritechRecipe>> getRecipe() {
        
        // get recipes matching input items
        var candidates = Objects.requireNonNull(field_11863).method_8433().method_17877(getOwnRecipeType(), getInputInventory(), field_11863);
        // filter out recipes based on input tank. Have the ones with input items first.
        var fluidRecipe = candidates
                            .stream()
                            .filter(candidate -> CentrifugeBlockEntity.recipeInputMatchesTank(ownStorage.getInputContainer().getStack(), candidate.comp_1933()))
                            .sorted(Comparator.comparingInt(a -> -a.comp_1933().getInputs().size()))
                            .findAny();
        if (fluidRecipe.isPresent()) {
            return fluidRecipe;
        }
        
        return Optional.empty();
    }
    
    @Override
    protected void craftItem(OritechRecipe activeRecipe, List<class_1799> outputInventory, List<class_1799> inputInventory) {
        super.craftItem(activeRecipe, outputInventory, inputInventory);
        craftFluids(activeRecipe);
    }
    
    @Override
    public List<class_1799> getCraftingResults(OritechRecipe activeRecipe) {
        var results = activeRecipe.getResults();
        if (results.isEmpty()) return List.of();
        return List.of(results.getFirst().method_46651(results.getFirst().method_7947() * getItemOutputMultiplier(activeRecipe)));
    }
    
    private void craftFluids(OritechRecipe activeRecipe) {
        // create outputs, remove inputs
        
        // remove input fluid
        ownStorage.getInputContainer().extract(ownStorage.getInputContainer().getStack().copyWithAmount(activeRecipe.getFluidInput().amount()), false);
        
        // create output fluids
        var outputs = calculateOutputFluids(activeRecipe);
        for (int i = 0; i < outputs.size(); i++) {
            var output = outputs.get(i);
            var outputTank = getOutputStorage(i);
            outputTank.insert(output, false);
        }
        
    }
    
    private List<FluidStack> calculateOutputFluids(OritechRecipe recipe) {
        // if no modules are installed, output twice the resulting items and fluids
        // if one module is installed, output twice the output A
        // if both are installed, output all as normal
        // if the recipe also only less than 2 fluid outputs, output normal
        
        if (recipe.getFluidOutputs().isEmpty()) return List.of();
        var outA = recipe.getFluidOutputs().get(0);
        
        if (recipe.getFluidOutputs().size() == 1) return List.of(outA);
        var outB = recipe.getFluidOutputs().get(1);
        
        return switch (moduleCount) {
            case 0 -> List.of(outA.copyWithAmount(outA.getAmount() * 2));
            case 1 -> List.of(outA, outB.copyWithAmount(outB.getAmount() * 2));
            case 2 -> recipe.getFluidOutputs();
            default -> throw new IllegalStateException("more than 2 modules is not supported/allowed");
        };
    }
    
    private int getItemOutputMultiplier(OritechRecipe recipe) {
        if (recipe.getFluidOutputs().size() <= 1) return 1;
        return getModuleCount() == 0 ? 2 : 1;
    }
    
    @Override
    public boolean canOutputRecipe(OritechRecipe recipe) {
        
        // 0 = base output, 1&2 = module outputs
        // checks if all fluid outputs for the active modules fit
        var fluidOutputs = calculateOutputFluids(recipe);
        for (int i = 0; i <= moduleCount; i++) {
            if (i >= fluidOutputs.size()) break;
            var fluidOutput = fluidOutputs.get(i);
            if (fluidOutput == null || fluidOutput.isEmpty()) continue;
            var storage = getOutputStorage(i);
            var inserted = storage.insert(fluidOutput, true);
            if (inserted != fluidOutput.getAmount()) return false;
        }
        
        return super.canOutputRecipe(recipe);
    }
    
    @Nullable
    @Override
    public class_1703 createMenu(int syncId, class_1661 playerInventory, class_1657 player) {
        return new RefineryScreenHandler(syncId, playerInventory, this);
    }
    
    @Override
    public BarConfiguration getFluidConfiguration() {
        return new BarConfiguration(28, 6, 21, 74);
    }
    
    @Override
    public long getDefaultCapacity() {
        return Oritech.CONFIG.processingMachines.refineryData.energyCapacity();
    }
    
    @Override
    public long getDefaultInsertRate() {
        return Oritech.CONFIG.processingMachines.refineryData.maxEnergyInsertion();
    }
    
    @Override
    protected OritechRecipeType getOwnRecipeType() {
        return RecipeContent.REFINERY;
    }
    
    @Override
    protected void useEnergy() {
        super.useEnergy();
        
        if (field_11863.field_9229.method_43057() > 0.8) return;
        // emit particles
        var facing = getFacing();
        var offsetLocal = Geometry.rotatePosition(new class_243(0.3, 0.5, 0.3), facing);
        var emitPosition = class_243.method_24953(field_11867).method_1019(offsetLocal);
        
        ParticleContent.COOLER_WORKING.spawn(field_11863, emitPosition, 1);
        
    }
    
    @Override
    public InventorySlotAssignment getSlotAssignments() {
        return new InventorySlotAssignment(0, 1, 1, 1);
    }
    
    @Override
    public List<GuiSlot> getGuiSlots() {
        return List.of(
          new GuiSlot(0, 62, 8),
          new GuiSlot(1, 62, 61, true));
    }
    
    @Override
    public class_3917<?> getScreenHandlerType() {
        return ModScreens.REFINERY_SCREEN;
    }
    
    @Override
    public int getInventorySize() {
        return 2;
    }
    
    @Override
    public boolean inputOptionsEnabled() {
        return false;
    }
    
    // x = back
    // y = up
    // z = left
    
    @Override
    public List<class_2382> getCorePositions() {
        return List.of(
          new class_2382(0, 1, 0),    // middle
          new class_2382(0, 0, -1),    // right
          new class_2382(0, 1, -1),
          new class_2382(1, 0, -1),    // back right
          new class_2382(1, 1, -1),
          new class_2382(1, 0, 0),    // back middle
          new class_2382(1, 1, 0),
          new class_2382(2, 0, -1),    // backer middle
          new class_2382(2, 1, -1)
        );
    }
    
    @Override
    public ArrowConfiguration getIndicatorConfiguration() {
        return new ArrowConfiguration(
          Oritech.id("textures/gui/modular/arrow_empty.png"),
          Oritech.id("textures/gui/modular/arrow_full.png"),
          54, 35, 29, 16, true);
    }
    
    // x = back, // z = left
    @Override
    public List<class_2382> getAddonSlots() {
        return List.of();
    }
    
    @Override
    public FluidApi.FluidStorage getFluidStorage(@Nullable class_2350 direction) {
        return ownStorage;
    }
    
    public FluidApi.FluidStorage getFluidStorageForModule(class_2338 modulePos) {
        var yDist = modulePos.method_10264() - this.field_11867.method_10264();
        if (yDist == 2) return nodeA;
        if (yDist == 3) return nodeB;
        throw new IllegalStateException("Module needs to be either 1 or 2 blocks above");
    }
    
    @Override
    public List<class_3545<class_2561, class_2561>> getExtraExtensionLabels() {
        return List.of(new class_3545<>(class_2561.method_43470("\uD83D\uDCE6: " + moduleCount), class_2561.method_43471("tooltip.oritech.refinery_module_count")));
    }
    
    public FluidApi.SingleSlotStorage getOutputStorage(int i) {
        if (i == 0) return ownStorage.getOutputContainer();
        if (i == 1) return nodeA;
        if (i == 2) return nodeB;
        throw new IllegalArgumentException("Only has 2 storage modules, tried accessing: " + i);
    }
}
    

