/**
 * This class was created by <Vazkii>. It's distributed as
 * part of the Psi Mod. Get the Source Code in github:
 * https://github.com/Vazkii/Psi
 *
 * Psi is Open Source and distributed under the
 * Psi License: http://psi.vazkii.us/license.php
 *
 * File Created @ [10/01/2016, 15:13:46 (GMT)]
 */
package org.violetmoon.zeta.util;

import org.jetbrains.annotations.NotNull;
import org.violetmoon.zeta.block.be.ZetaBlockEntity;

import net.minecraft.core.BlockPos;
import net.minecraft.core.Direction;
import net.minecraft.core.NonNullList;
import net.minecraft.nbt.CompoundTag;
import net.minecraft.nbt.ListTag;
import net.minecraft.world.WorldlyContainer;
import net.minecraft.world.entity.player.Player;
import net.minecraft.world.item.ItemStack;
import net.minecraft.world.level.block.entity.BlockEntityType;
import net.minecraft.world.level.block.state.BlockState;
import net.minecraftforge.common.capabilities.Capability;
import net.minecraftforge.common.capabilities.ForgeCapabilities;
import net.minecraftforge.common.util.LazyOptional;
import net.minecraftforge.items.wrapper.SidedInvWrapper;

// formerly from AutoRegLib
public abstract class SimpleInventoryBlockEntity extends ZetaBlockEntity implements WorldlyContainer {
	public SimpleInventoryBlockEntity(BlockEntityType<?> tileEntityTypeIn, BlockPos pos, BlockState state) {
		super(tileEntityTypeIn, pos, state);
	}

	protected NonNullList<ItemStack> inventorySlots = NonNullList.withSize(getContainerSize(), ItemStack.EMPTY);

	@Override
	public void readSharedNBT(CompoundTag par1NBTTagCompound) {
		if(!needsToSyncInventory())
			return;

		ListTag var2 = par1NBTTagCompound.getList("Items", 10);
		clearContent();
		for(int var3 = 0; var3 < var2.size(); ++var3) {
			CompoundTag var4 = var2.getCompound(var3);
			byte var5 = var4.getByte("Slot");
			if(var5 >= 0 && var5 < inventorySlots.size())
				inventorySlots.set(var5, ItemStack.of(var4));
		}
	}

	@Override
	public void writeSharedNBT(CompoundTag par1NBTTagCompound) {
		if(!needsToSyncInventory())
			return;

		ListTag var2 = new ListTag();
		for(int var3 = 0; var3 < inventorySlots.size(); ++var3) {
			if(!inventorySlots.get(var3).isEmpty()) {
				CompoundTag var4 = new CompoundTag();
				var4.putByte("Slot", (byte) var3);
				inventorySlots.get(var3).save(var4);
				var2.add(var4);
			}
		}
		par1NBTTagCompound.put("Items", var2);
	}

	protected boolean needsToSyncInventory() {
		return true;
	}

	@NotNull
	@Override
	public ItemStack getItem(int i) {
		return inventorySlots.get(i);
	}

	@NotNull
	@Override
	public ItemStack removeItem(int i, int j) {
		if(!inventorySlots.get(i).isEmpty()) {
			ItemStack stackAt;

			if (inventorySlots.get(i).getCount() <= j) {
				stackAt = inventorySlots.get(i);
				inventorySlots.set(i, ItemStack.EMPTY);
            } else {
				stackAt = inventorySlots.get(i).split(j);

				if (inventorySlots.get(i).getCount() == 0)
					inventorySlots.set(i, ItemStack.EMPTY);
            }

            inventoryChanged(i);
            return stackAt;
        }

		return ItemStack.EMPTY;
	}

	@NotNull
	@Override
	public ItemStack removeItemNoUpdate(int i) {
		ItemStack stack = getItem(i);
		setItem(i, ItemStack.EMPTY);
		inventoryChanged(i);
		return stack;
	}

	@Override
	public void setItem(int i, @NotNull ItemStack itemstack) {
		inventorySlots.set(i, itemstack);
		inventoryChanged(i);
	}

	@Override
	public int getMaxStackSize() {
		return 64;
	}

	@Override
	public boolean isEmpty() {
		for(int i = 0; i < getContainerSize(); i++) {
			ItemStack stack = getItem(i);
			if(!stack.isEmpty())
				return false;
		}

		return true;
	}

	@Override
	public boolean stillValid(@NotNull Player entityplayer) {
		return getLevel().getBlockEntity(getBlockPos()) == this && entityplayer.distanceToSqr(worldPosition.getX() + 0.5D, worldPosition.getY() + 0.5D, worldPosition.getZ() + 0.5D) <= 64;
	}

	@SuppressWarnings("unchecked")
	@Override
	public <T> @NotNull LazyOptional<T> getCapability(@NotNull Capability<T> capability, Direction facing) {
		if(capability == ForgeCapabilities.ITEM_HANDLER)
			return (LazyOptional<T>) LazyOptional.of(() -> new SidedInvWrapper(this, facing));

		return LazyOptional.empty();
	}

	@Override
	public boolean canPlaceItem(int i, @NotNull ItemStack itemstack) {
		return true;
	}

	@Override
	public void startOpen(@NotNull Player player) {
		// NO-OP
	}

	@Override
	public void stopOpen(@NotNull Player player) {
		// NO-OP
	}

	@Override
	public void clearContent() {
		inventorySlots = NonNullList.withSize(getContainerSize(), ItemStack.EMPTY);
	}

	public void inventoryChanged(int i) {
		// NO-OP
	}

	public boolean isAutomationEnabled() {
		return true;
	}

	@Override
	public boolean canTakeItemThroughFace(int index, @NotNull ItemStack stack, @NotNull Direction direction) {
		return isAutomationEnabled();
	}

	@Override
	public boolean canPlaceItemThroughFace(int index, @NotNull ItemStack itemStackIn, Direction direction) {
		return isAutomationEnabled();
	}

	@Override
	public int @NotNull [] getSlotsForFace(@NotNull Direction side) {
		if(isAutomationEnabled()) {
			int[] slots = new int[getContainerSize()];
			for(int i = 0; i < slots.length; i++)
				slots[i] = i;
			return slots;
		}

		return new int[0];
	}
}
