package org.gtreimagined.gtlib.tool;

import it.unimi.dsi.fastutil.ints.Int2ObjectMap;
import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;
import org.gtreimagined.gtlib.data.GTMaterialTypes;
import org.gtreimagined.gtlib.material.Material;
import org.gtreimagined.gtlib.material.MaterialTags;
import org.gtreimagined.gtlib.util.TagUtils;
import net.minecraft.world.item.Tier;
import net.minecraft.world.item.crafting.Ingredient;
import org.jetbrains.annotations.NotNull;

import java.util.Objects;
import java.util.Optional;

public class GTItemTier implements Tier {

    private final Material primary;
    private final Material secondary;

    public static final GTItemTier NULL = new GTItemTier(Material.NULL, Material.NULL);

    private static final Int2ObjectMap<GTItemTier> TIERS_LOOKUP = new Int2ObjectOpenHashMap<>();

    static {
        TIERS_LOOKUP.put(NULL.hashCode(), NULL);
    }

    GTItemTier(@NotNull Material primary, @NotNull Material secondary) {
        this.primary = primary.has(MaterialTags.TOOLS) ? primary : Material.NULL;
        this.secondary = secondary;
    }

    public static Optional<GTItemTier> get(int key) {
        return Optional.ofNullable(TIERS_LOOKUP.get(key));
    }

    public static GTItemTier getOrCreate(String primaryName, String secondaryName) {
        return TIERS_LOOKUP.computeIfAbsent(Objects.hash(primaryName, secondaryName), m -> new GTItemTier(Material.get(primaryName), Material.get(secondaryName)));
    }

    public static GTItemTier getOrCreate(Material primary, Material secondary) {
        return TIERS_LOOKUP.computeIfAbsent(Objects.hash(primary.hashCode(), secondary.hashCode()), m -> new GTItemTier(primary, secondary));
    }

    @Override
    public int getUses() {
        int secondaryUses = secondary.has(MaterialTags.HANDLE) ? MaterialTags.HANDLE.get(secondary).durability() : 0;
        return MaterialTags.TOOLS.get(primary).toolDurability() + secondaryUses;
    }

    @Override
    public float getSpeed() {
        float secondarySpeed = secondary.has(MaterialTags.HANDLE) ? MaterialTags.HANDLE.get(secondary).speed() : 0f;
        return MaterialTags.TOOLS.get(primary).toolSpeed() + secondarySpeed;
    }

    @Override
    public float getAttackDamageBonus() {
        return MaterialTags.TOOLS.get(primary).toolDamage();
    }

    @Override
    public int getLevel() {
        if (primary == Material.NULL) return 3;
        return MaterialTags.TOOLS.get(primary).toolQuality();
    }

    @Override
    public int getEnchantmentValue() {
        return (int) (getLevel() + getSpeed());
    }

    @Override
    public Ingredient getRepairIngredient() {
        if (primary.has(GTMaterialTypes.GEM)) {
            return Ingredient.of(TagUtils.getForgelikeItemTag("gems/".concat(primary.getId())));
        } else if (primary.has(GTMaterialTypes.INGOT)) {
            return Ingredient.of(TagUtils.getForgelikeItemTag("ingots/".concat(primary.getId())));
        } else if (primary.has(GTMaterialTypes.DUST)) {
            return Ingredient.of(TagUtils.getForgelikeItemTag("dusts/".concat(primary.getId())));
        } //else if (ItemTags.getAllTags().getTag(new ResourceLocation("forge", "blocks/".concat(primary.getId()))) != null) {
         //   return Ingredient.of(TagUtils.getForgeItemTag("blocks/".concat(primary.getId())));
     //   }
        return Ingredient.EMPTY;
        // return null;
    }

    public Material getPrimary() {
        return primary;
    }

    public Material getSecondary() {
        return secondary;
    }

    @Override
    public boolean equals(Object obj) {
        if (obj == this) return true;
        if (!(obj instanceof GTItemTier tier)) return false;
        return primary == tier.getPrimary() && secondary == tier.getSecondary();
    }

    @Override
    public int hashCode() {
        return Objects.hash(primary.hashCode(), secondary.hashCode());
    }

}
