package org.gtreimagined.gtlib.capability.machine;

import com.google.common.collect.ImmutableList;
import it.unimi.dsi.fastutil.objects.ObjectArrayList;
import net.minecraftforge.common.capabilities.ForgeCapabilities;
import org.gtreimagined.gtlib.Ref;
import org.gtreimagined.gtlib.blockentity.BlockEntityMachine;
import org.gtreimagined.gtlib.capability.Dispatch;
import org.gtreimagined.gtlib.capability.IMachineHandler;
import org.gtreimagined.gtlib.capability.rf.FEHandler;
import org.gtreimagined.gtlib.gui.SlotType;
import org.gtreimagined.gtlib.machine.event.IMachineEvent;
import org.gtreimagined.gtlib.machine.event.MachineEvent;
import org.gtreimagined.gtlib.util.Utils;
import net.minecraft.core.Direction;
import net.minecraft.world.level.block.entity.BlockEntity;
import net.minecraftforge.common.util.LazyOptional;
import net.minecraftforge.energy.CapabilityEnergy;
import net.minecraftforge.energy.IEnergyStorage;
import org.gtreimagined.tesseract.api.fe.IExtendedEnergyStorage;

import java.util.List;
import java.util.Optional;

public class MachineFEHandler<T extends BlockEntityMachine<T>> extends FEHandler implements IMachineHandler, Dispatch.Sided<IExtendedEnergyStorage> {
    protected final T tile;
    protected List<IEnergyStorage> cachedItems = new ObjectArrayList<>();

    protected int offsetInsert = 0;
    protected int offsetExtract = 0;
    public MachineFEHandler(T tile, int energy, int capacity, int maxIn, int maxOut) {
        super(energy, capacity, maxIn, maxOut);
        this.tile = tile;
    }

    public MachineFEHandler(T tile, int capacity, boolean isGenerator) {
        this(tile, 0, capacity, isGenerator ? 0 : (int) tile.getMachineTier().getVoltage(), isGenerator ? (int) tile.getMachineTier().getVoltage() : 0);
    }

    public void onUpdate(){
        for (Direction dir : Ref.DIRS) {
            if (canExtract(dir)) {
                BlockEntity tile = this.tile.getCachedBlockEntity(dir);
                if (tile == null) continue;
                Optional<IEnergyStorage> handle = tile.getCapability(ForgeCapabilities.ENERGY, dir.getOpposite()).resolve();
                handle.ifPresent(eh -> Utils.transferEnergy(this, eh));
            }
        }
    }

    @Override
    public void init() {
        this.cachedItems = tile.itemHandler.map(MachineItemHandler::getFEChargeableItems).map(ImmutableList::copyOf).orElse(ImmutableList.of());
    }

    @Override
    public int getMaxEnergyStored() {
        if (canChargeItem()) {
            return super.getMaxEnergyStored() + (cachedItems != null ? cachedItems.stream().mapToInt(IEnergyStorage::getMaxEnergyStored).sum() : 0);
        }
        return super.getMaxEnergyStored();
    }

    @Override
    public int receiveEnergy(int maxAmount, boolean simulate) {
        int j = 0;
        int inserted = super.receiveEnergy(maxAmount, simulate);
        for (int i = offsetInsert; j < cachedItems.size(); j++, i = (i == cachedItems.size() - 1 ? 0 : (i + 1))) {
            IEnergyStorage handler = cachedItems.get(i);
            if (!handler.canReceive()) continue;
            int insert = handler.receiveEnergy(maxAmount, simulate);
            if (insert > 0) {
                offsetInsert = (offsetInsert + 1) % cachedItems.size();
                inserted += insert;
            }
        }
        if (inserted > 0) {
            tile.onMachineEvent(MachineEvent.ENERGY_INPUTTED);
        }
        return inserted;
    }

    @Override
    public int extractEnergy(int maxAmount, boolean simulate) {
        int j = 0;
        int extracted = super.extractEnergy(maxAmount, simulate);
        for (int i = offsetInsert; j < cachedItems.size(); j++, i = (i == cachedItems.size() - 1 ? 0 : (i + 1))) {
            IEnergyStorage handler = cachedItems.get(i);
            if (!handler.canExtract()) continue;
            int extract = handler.extractEnergy(maxAmount, simulate);
            if (extract > 0) {
                offsetInsert = (offsetInsert + 1) % cachedItems.size();
                extracted += extract;
            }
        }
        if (extracted > 0) {
            tile.onMachineEvent(MachineEvent.ENERGY_INPUTTED);
        }
        return extracted;
    }

    @Override
    public int getEnergyStored() {
        if (canChargeItem()) {
            return super.getEnergyStored() + (cachedItems != null ? cachedItems.stream().mapToInt(IEnergyStorage::getEnergyStored).sum() : 0);
        }
        return super.getEnergyStored();
    }

    @Override
    public boolean canReceive(Direction direction) {
        return super.canReceive(direction) && (tile.getFacing() != direction || tile.getMachineType().allowsFrontIO());
    }

    public boolean canChargeItem() {
        return true;
    }

    @Override
    public void onMachineEvent(IMachineEvent event, Object... data) {
        if (event == SlotType.ENERGY) {
            tile.itemHandler.ifPresent(h -> {
                cachedItems = h.getFEChargeableItems();
                offsetInsert = 0;
                offsetExtract = 0;
            });
            //refreshNet();
        }
    }

    @Override
    public LazyOptional<? extends IExtendedEnergyStorage> forSide(Direction side) {
        return LazyOptional.of(() -> this);
    }

    @Override
    public LazyOptional<? extends IExtendedEnergyStorage> forNullSide() {
        return LazyOptional.of(() -> this);
    }

    public void onRemove() {
    }
}
