package org.gtreimagined.gtlib.blockentity.multi;

import com.google.common.collect.Lists;
import com.gtnewhorizon.structurelib.StructureLibAPI;
import com.gtnewhorizon.structurelib.alignment.IAlignment;
import com.gtnewhorizon.structurelib.alignment.IAlignmentLimits;
import com.gtnewhorizon.structurelib.alignment.enumerable.ExtendedFacing;
import com.gtnewhorizon.structurelib.alignment.enumerable.Flip;
import com.gtnewhorizon.structurelib.alignment.enumerable.Rotation;
import com.gtnewhorizon.structurelib.structure.IStructureElement;
import it.unimi.dsi.fastutil.Pair;
import it.unimi.dsi.fastutil.longs.Long2ObjectOpenHashMap;
import it.unimi.dsi.fastutil.longs.LongList;
import it.unimi.dsi.fastutil.objects.Object2ObjectMap;
import it.unimi.dsi.fastutil.objects.Object2ObjectOpenHashMap;
import it.unimi.dsi.fastutil.objects.ObjectOpenHashSet;
import org.gtreimagined.gtlib.GTLib;
import org.gtreimagined.gtlib.GTLibConfig;
import org.gtreimagined.gtlib.block.BlockBasic;
import org.gtreimagined.gtlib.blockentity.BlockEntityMachine;
import org.gtreimagined.gtlib.blockentity.IFakeTileCap;
import org.gtreimagined.gtlib.capability.IComponentHandler;
import org.gtreimagined.gtlib.client.scene.TrackedDummyWorld;
import org.gtreimagined.gtlib.cover.CoverDynamo;
import org.gtreimagined.gtlib.cover.CoverEnergy;
import org.gtreimagined.gtlib.cover.ICover;
import org.gtreimagined.gtlib.machine.MachineState;
import org.gtreimagined.gtlib.machine.event.IMachineEvent;
import org.gtreimagined.gtlib.machine.event.MachineEvent;
import org.gtreimagined.gtlib.machine.types.BasicMultiMachine;
import org.gtreimagined.gtlib.machine.types.Machine;
import org.gtreimagined.gtlib.registration.IGTObject;
import org.gtreimagined.gtlib.structure.Structure;
import org.gtreimagined.gtlib.structure.StructureCache;
import org.gtreimagined.gtlib.structure.StructureHandle;
import org.gtreimagined.gtlib.texture.Texture;
import org.gtreimagined.gtlib.tool.GTToolType;
import org.gtreimagined.gtlib.util.Utils;
import net.minecraft.core.BlockPos;
import net.minecraft.core.Direction;
import net.minecraft.nbt.CompoundTag;
import net.minecraft.server.level.ServerLevel;
import net.minecraft.world.InteractionHand;
import net.minecraft.world.InteractionResult;
import net.minecraft.world.entity.player.Player;
import net.minecraft.world.level.Level;
import net.minecraft.world.level.block.state.BlockState;
import net.minecraft.world.phys.BlockHitResult;
import net.minecraftforge.common.capabilities.Capability;
import net.minecraftforge.common.util.LazyOptional;
import net.minecraftforge.fml.loading.FMLEnvironment;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import tesseract.api.forge.TesseractCaps;

import java.util.Collections;
import java.util.List;
import java.util.Set;

/**
 * Allows a MultiMachine to handle GUI recipes, instead of using Hatches
 **/
public class BlockEntityBasicMultiMachine<T extends BlockEntityBasicMultiMachine<T>> extends BlockEntityMachine<T>
        implements IAlignment , IFakeTileCap {

    private final Set<StructureHandle<?>> allHandlers = new ObjectOpenHashSet<>();
    protected boolean validStructure = false;

    public Long2ObjectOpenHashMap<IStructureElement<T>> structurePositions = new Long2ObjectOpenHashMap<>();

    private ExtendedFacing extendedFacing;
    private IAlignmentLimits limits = getInitialAlignmentLimits();
    /**
     * To ensure proper load from disk, do not check if INVALID_STRUCTURE is loaded
     * from disk.
     **/
    protected boolean shouldCheckFirstTick = true;
    // Number of calls into checkStructure, invalidateStructure. if > 0 ignore
    // callbacks from structurecache.
    protected int checkingStructure = 0;
    /**
     * Used whenever a machine might be rotated and is checking structure, since the
     * facing is changed before checkStructure()
     * is called and it needs to properly offset in recipeStop
     */
    public BlockState oldState;
    private Direction facingOverride;


    public Object2ObjectMap<String, List<IComponentHandler>> components = new Object2ObjectOpenHashMap<>();

    public BlockEntityBasicMultiMachine(Machine<?> type, BlockPos pos, BlockState state) {
        super(type, pos, state);
        extendedFacing = ExtendedFacing.of(getFacing(state), Rotation.NORMAL, Flip.NONE);
    }

    @Override
    public void onRemove() {
        super.onRemove();
        if (getLevel() != null && !getLevel().isClientSide()) {
            // Remove handlers from the structure cache.
            allHandlers.forEach(StructureHandle::deregister);
            invalidateStructure();
        }
    }

    @Override
    public IAlignmentLimits getAlignmentLimits() {
        return limits;
    }

    @Override
    public ExtendedFacing getExtendedFacing() {
        return extendedFacing;
    }

    @Override
    public void setExtendedFacing(ExtendedFacing extendedFacing) {
        if (this.extendedFacing != extendedFacing && extendedFacing.getDirection() == this.getFacing()){
            this.extendedFacing = extendedFacing;
            invalidateCaps();
            if (isServerSide()) {
                invalidateStructure();
                checkStructure();
                StructureLibAPI.sendAlignment(
                        this,
                        getBlockPos(), 1.0, (ServerLevel) level);
            }
        }

    }

    @Override
    public boolean setFacing(Direction side) {
        boolean facingSet = super.setFacing(side);
        if (facingSet){
            extendedFacing = ExtendedFacing.of(side, extendedFacing.getRotation(), extendedFacing.getFlip());
            if (isServerSide()) {
                invalidateStructure();
                checkStructure();
                StructureLibAPI.sendAlignment(
                        this,
                        getBlockPos(), 1.0, (ServerLevel) level);
            }
        }
        return facingSet;
    }

    protected IAlignmentLimits getInitialAlignmentLimits() {
        return (d, r, f) -> !f.isVerticallyFliped();
    }

    /**
     * How many multiblocks you can share components with.
     *
     * @return how many.
     */
    public int maxShares() {
        return Integer.MAX_VALUE;
    }

    @Override
    public void onFirstTickClient(Level level, BlockPos pos, BlockState state) {
        StructureLibAPI.queryAlignment(this);
        super.onFirstTickClient(level, pos, state);
    }

    @Override
    public void onFirstTickServer(Level level, BlockPos pos, BlockState state) {
        // Register handlers to the structure cache.
        allHandlers.forEach(StructureHandle::register);
        // if INVALID_STRUCTURE was stored to disk don't bother rechecking on first
        // tick.
        // This is not only behavioural but if INVALID_STRUCTURE are checked then
        // maxShares
        // might misbehave.
        if (!validStructure && shouldCheckFirstTick) {
            checkStructure();
        }
        super.onFirstTickServer(level, pos, state);
    }

    @Override
    public InteractionResult onInteractServer(BlockState state, Level world, BlockPos pos, Player player, InteractionHand hand, BlockHitResult hit, @Nullable GTToolType type) {
        if (!validStructure && checkingStructure == 0){
            if (checkStructure()){
                return InteractionResult.SUCCESS;
            }
        }
        return super.onInteractServer(state, world, pos, player, hand, hit, type);
    }

    @Override
    public Direction getFacing() {
        return facingOverride != null ? facingOverride : super.getFacing();
    }

    public boolean checkStructure() {
        if (level != null && isClientSide()){
            GTLib.LOGGER.warn("Checking structure on client side");
            Thread.dumpStack();
            return false;
        }
        Structure<T> structure = getMachineType().getStructure(getMachineTier());
        if (structure == null)
            return false;
        checkingStructure++;
        List<Pair<BlockPos, IStructureElement<T>>> oldPositions = structurePositions.long2ObjectEntrySet().stream().map(e -> Pair.of(BlockPos.of(e.getLongKey()), e.getValue())).toList();
        structurePositions.clear();
        components.clear();
        boolean oldValidStructure = validStructure;
        validStructure = structure.check((T)this);
        boolean[] fail = new boolean[1];
        fail[0] = false;
        structure.getMinMaxMap().forEach((s, p) -> {
            int min = p.left();
            int max = p.right();
            int size = 0;
            if (components.containsKey(s)){
                size = components.get(s).size();
            }
            if (size < min || size > max){
                fail[0] = true;
            }
        });
        if (fail[0]) validStructure = false;
        if (validStructure){
            LongList positions = LongList.of(structurePositions.keySet().toLongArray());
            if (level instanceof TrackedDummyWorld) {
                StructureCache.add(level, worldPosition, positions);
                StructureCache.validate(level, worldPosition, positions, maxShares());
                checkingStructure--;
                return true;
            } else if (onStructureFormed() && StructureCache.validate(this.getLevel(), this.getBlockPos(), positions, maxShares())){
                afterStructureFormed();
                if (isServerSide()){
                    if (machineState != MachineState.ACTIVE && machineState != MachineState.DISABLED) {
                        setMachineState(MachineState.IDLE);
                    }
                    this.recipeHandler.ifPresent(
                            t -> t.onMultiBlockStateChange(true, GTLibConfig.INPUT_RESET_MULTIBLOCK.get()));
                } else {
                    this.components.forEach((k, v) -> v.forEach(c -> {
                        Utils.markTileForRenderUpdate(c.getTile());
                    }));
                }

                StructureCache.add(level, getBlockPos(), positions);
            } else {
                validStructure = false;
                structurePositions.forEach((l, e) -> {
                    BlockPos pos = BlockPos.of(l);
                    e.onStructureFail((T)this, this.getLevel(), pos.getX(), pos.getY(), pos.getZ());
                });
            }

        }
        if (!validStructure && !oldPositions.isEmpty()){
            oldPositions.forEach(p ->{
                p.right().onStructureFail((T) this, this.getLevel(), p.left().getX(), p.left().getY(), p.left().getZ());
            });
        }
        checkingStructure--;
        if (validStructure != oldValidStructure){
            sidedSync(true);
        }
        return validStructure;
    }

    public void serverTick(Level level, BlockPos pos, BlockState state) {
        super.serverTick(level, pos, state);
        if (level.getGameTime() % 100 == 0 && !validStructure && checkingStructure == 0 && !FMLEnvironment.production){
            //checkStructure();
        }
    }

    public boolean allowsFakeTiles(){
        return false;
    }

    @Override
    public void onBlockUpdate(BlockPos pos) {
        super.onBlockUpdate(pos);
        if (!isServerSide()) return;
        if (checkingStructure > 0)
            return;
        if (validStructure) {
            long longPos = pos.asLong();
            if (structurePositions.containsKey(longPos) && !structurePositions.get(longPos).check((T) this, this.getLevel(), pos.getX(), pos.getY(), pos.getZ())) {
                invalidateStructure();
            }
        } else {
            checkStructure();
        }
    }

    @Override
    public void setMachineState(MachineState newState) {
        if (this.remove)
            return;
        super.setMachineState(newState);
    }

    @Override
    public void setBlockState(BlockState p_155251_) {
        BlockState old = this.getBlockState();
        super.setBlockState(p_155251_);
        BlockState newState = this.getBlockState();
        if (!getFacing(old).equals(getFacing(newState))) {
            if (checkingStructure > 0) return;
            invalidateStructure();
            oldState = old;
            this.facingOverride = Utils.dirFromState(oldState);
            checkStructure();
            oldState = null;
            facingOverride = null;
        }
    }

    @Override
    public void onMachineStop() {
        super.onMachineStop();

    }

    @Override
    public void onMachineEvent(IMachineEvent event, Object... data) {
        super.onMachineEvent(event, data);
        if (event == MachineEvent.FLUIDS_OUTPUTTED || event == MachineEvent.ITEMS_OUTPUTTED) {
            components.values().forEach(l -> l.forEach(i -> {
                if (i.getTile() instanceof BlockEntityHatch<?> hatch) {
                    hatch.onMachineEvent(event, data);
                }
            }));
        }
    }

    public void invalidateStructure() {
        if (level != null && isClientSide()){
            GTLib.LOGGER.warn("Invalidating structure on client side");
            Thread.dumpStack();
            return;
        }
        if (this.getLevel() instanceof TrackedDummyWorld)
            return;
        if (!validStructure) {
            return;
        }
        checkingStructure++;
        validStructure = false;
        if (isServerSide() && getMachineState() != getDefaultMachineState()) {
            resetMachine();
        }
        structurePositions.forEach((l,e) ->{
            BlockPos pos = BlockPos.of(l);
            e.onStructureFail((T) this, this.getLevel(), pos.getX(), pos.getY(), pos.getZ());
        });
        structurePositions.clear();
        onStructureInvalidated();
        if (isServerSide()) {
            recipeHandler.ifPresent(
                    t -> t.onMultiBlockStateChange(false, GTLibConfig.INPUT_RESET_MULTIBLOCK.get()));
            components.clear();
        } else {
            this.components.forEach((k, v) -> v.forEach(c -> {
                Utils.markTileForRenderUpdate(c.getTile());
            }));
            components.clear();
        }
        StructureCache.remove(level, worldPosition);
        sidedSync(true);
        checkingStructure--;
    }

    /**
     * Returns a list of Components
     **/
    public List<IComponentHandler> getComponents(IGTObject object) {
        return getComponents(object.getId());
    }

    public List<IComponentHandler> getComponents(String id) {
        List<IComponentHandler> list = components.get(id);
        return list != null ? list : Collections.emptyList();
    }

    public List<IComponentHandler> getComponentsByHandlerId(String id) {
        List<IComponentHandler> list = components.get(id);
        return list != null ? list : Collections.emptyList();
    }


    public void addComponent(String elementId, IComponentHandler component) {
        List<IComponentHandler> existing = components.get(component.getIdForHandlers());
        if (existing == null) components.put(component.getIdForHandlers(), Lists.newArrayList(component));
        else existing.add(component);
        if (!elementId.isEmpty() && !elementId.equals(component.getIdForHandlers())) {
            existing = components.get(elementId);
            if (existing == null) components.put(elementId, Lists.newArrayList(component));
            else existing.add(component);
        }
    }

    public boolean isStructureValid() {
        return validStructure;
        //return StructureCache.has(level, worldPosition);
    }

    @Override
    public void onLoad() {
        super.onLoad();
        Structure struc = getMachineType().getStructure(getMachineTier());
        if (struc != null) {
            //StructureCache.add(level, worldPosition, struc.allPositions(this));
        }
    }

    /**
     * Events
     **/
    public boolean onStructureFormed() {
        return true;
    }

    public void afterStructureFormed() {
        // NOOP
    }

    public void onStructureInvalidated() {
        // NOOP
    }

    public Texture getTextureForHatches(Direction dir, BlockPos hatchPos){
        Texture[] tex = this.getMachineType().getBaseTexture(this.getMachineTier(), this.getMachineState().getTextureState());
        if (tex.length == 1) return tex[0];
        return tex[dir.get3DDataValue()];
    }

    public BlockBasic getHatchBlock(BlockPos pos){
        if (this.getMachineType() instanceof BasicMultiMachine<?> multiMachine && multiMachine.getTextureBlock() != null){
            return multiMachine.getTextureBlock().apply(tier);
        }
        return null;
    }

    @Override
    public MachineState getDefaultMachineState() {
        // Has to be nullchecked because it can be called in a constructor.
        if (!validStructure)
            return MachineState.INVALID_STRUCTURE;
        return MachineState.IDLE;
    }

    @Override
    public void saveAdditional(CompoundTag tag) {
        super.saveAdditional(tag);
        tag.putByte("rotation", (byte) extendedFacing.getRotation().getIndex());
        tag.putByte("flip", (byte) extendedFacing.getFlip().getIndex());
    }

    @Override
    public void load(CompoundTag tag) {
        super.load(tag);
        if (getMachineState() == MachineState.INVALID_STRUCTURE) {
            shouldCheckFirstTick = false;
        }
        this.extendedFacing = ExtendedFacing.of(extendedFacing.getDirection(), Rotation.byIndex(tag.getByte("rotation")), Flip.byIndex(tag.getByte("flip")));
    }

    public void addStructureHandle(StructureHandle<?> handle) {
        this.allHandlers.add(handle);
    }

    @Override
    public <U> LazyOptional<U> getCapabilityFromFake(@NotNull Capability<U> cap, @Nullable Direction side, ICover cover) {
        if (!allowsFakeTiles()) return LazyOptional.empty();
        if ((cap == TesseractCaps.ENERGY_HANDLER_CAPABILITY) && !(cover instanceof CoverDynamo || cover instanceof CoverEnergy)) return LazyOptional.empty();
        return getCap(cap, side);
    }
}
