package org.embeddedt.embeddium.api.render.chunk;

import com.mojang.blaze3d.vertex.PoseStack;
import net.minecraft.client.renderer.RenderType;
import net.minecraft.client.resources.model.BakedModel;
import net.minecraft.core.BlockPos;
import net.minecraft.world.level.BlockAndTintGetter;
import net.minecraft.world.level.block.state.BlockState;
import net.neoforged.neoforge.client.model.data.ModelData;
import org.embeddedt.embeddium.impl.render.matrix_stack.CachingPoseStack;
import org.embeddedt.embeddium.impl.render.world.WorldSliceLocalGenerator;
import org.joml.Vector3f;
import org.joml.Vector3fc;

/**
 * Holds the context for the current block being rendered in a chunk section. This container is reused rather than
 * being freshly constructed for each block to avoid allocations.
 */
public class BlockRenderContext {
    private final EmbeddiumBlockAndTintGetter world;
    private final BlockAndTintGetter localSlice;

    private final BlockPos.MutableBlockPos pos = new BlockPos.MutableBlockPos();

    private final Vector3f origin = new Vector3f();

    private final PoseStack stack = new PoseStack();

    private BlockState state;
    private BakedModel model;

    private long seed;

    private ModelData modelData;
    private RenderType renderLayer;


    public BlockRenderContext(EmbeddiumBlockAndTintGetter world) {
        this.world = world;
        this.localSlice = WorldSliceLocalGenerator.generate(world);
        ((CachingPoseStack)this.stack).embeddium$setCachingEnabled(true);
    }

    public void update(BlockPos pos, BlockPos origin, BlockState state, BakedModel model, long seed, ModelData modelData, RenderType renderLayer) {
        this.pos.set(pos);
        this.origin.set(origin.getX(), origin.getY(), origin.getZ());

        this.state = state;
        this.model = model;

        this.seed = seed;

        this.modelData = modelData;
        this.renderLayer = renderLayer;
    }

    /**
     * @return The position (in world space) of the block being rendered
     */
    public BlockPos pos() {
        return this.pos;
    }

    /**
     * @return The world which the block is being rendered from
     */
    public EmbeddiumBlockAndTintGetter world() {
        return this.world;
    }

    /**
     * @return The world which the block is being rendered from. Guaranteed to be a new object for each subchunk.
     */
    public BlockAndTintGetter localSlice() {
        return this.localSlice;
    }

    /**
     * @return The state of the block being rendered
     */
    public BlockState state() {
        return this.state;
    }

    /**
     * @return A PoseStack for custom renderers
     */
    public PoseStack stack() {
        return this.stack;
    }

    /**
     * @return The model used for this block
     */
    public BakedModel model() {
        return this.model;
    }

    /**
     * @return The origin of the block within the model
     */
    public Vector3fc origin() {
        return this.origin;
    }

    /**
     * @return The PRNG seed for rendering this block
     */
    public long seed() {
        return this.seed;
    }

    /**
     * @return The additional data for model instance
     */
    public ModelData modelData() {
        return this.modelData;
    }

    /**
     * @return The render layer for model rendering
     */
    public RenderType renderLayer() {
        return this.renderLayer;
    }
}
