package me.jellysquid.mods.sodium.client.world;

import me.jellysquid.mods.sodium.client.world.biome.BlockColorCache;
import me.jellysquid.mods.sodium.client.world.cloned.ChunkRenderContext;
import me.jellysquid.mods.sodium.client.world.cloned.ClonedChunkSection;
import me.jellysquid.mods.sodium.client.world.cloned.ClonedChunkSectionCache;
import me.jellysquid.mods.sodium.client.world.cloned.PackedIntegerArrayExtended;
import me.jellysquid.mods.sodium.client.world.cloned.palette.ClonedPalette;
import net.minecraft.client.Minecraft;
import net.minecraft.core.BlockPos;
import net.minecraft.core.Direction;
import net.minecraft.core.Holder;
import net.minecraft.core.QuartPos;
import net.minecraft.core.Registry;
import net.minecraft.core.SectionPos;
import net.minecraft.util.Mth;
import net.minecraft.util.SimpleBitStorage;
import net.minecraft.util.math.*;
import net.minecraft.world.level.BlockAndTintGetter;
import net.minecraft.world.level.ColorResolver;
import net.minecraft.world.level.Level;
import net.minecraft.world.level.LightLayer;
import net.minecraft.world.level.biome.Biome;
import net.minecraft.world.level.biome.BiomeManager;
import net.minecraft.world.level.biome.Biomes;
import net.minecraft.world.level.block.Blocks;
import net.minecraft.world.level.block.entity.BlockEntity;
import net.minecraft.world.level.block.state.BlockState;
import net.minecraft.world.level.chunk.LevelChunk;
import net.minecraft.world.level.chunk.LevelChunkSection;
import net.minecraft.world.level.levelgen.structure.BoundingBox;
import net.minecraft.world.level.lighting.LevelLightEngine;
import net.minecraft.world.level.material.FluidState;
import org.embeddedt.embeddium.api.ChunkMeshEvent;
import org.embeddedt.embeddium.api.MeshAppender;

import java.util.Arrays;
import java.util.List;

/**
 * Takes a slice of world state (block states, biome and light data arrays) and copies the data for use in off-thread
 * operations. This allows chunk build tasks to see a consistent snapshot of chunk data at the exact moment the task was
 * created.
 *
 * World slices are not safe to use from multiple threads at once, but the data they contain is safe from modification
 * by the main client thread.
 *
 * Object pooling should be used to avoid huge allocations as this class contains many large arrays.
 */
public class WorldSlice implements BlockAndTintGetter {
    // The number of blocks in a section.
    private static final int SECTION_BLOCK_COUNT = 16 * 16 * 16;

    // The number of biomes in a section.
    private static final int SECTION_BIOME_COUNT = 4 * 4 * 4;

    // The radius of blocks around the origin chunk that should be copied.
    private static final int NEIGHBOR_BLOCK_RADIUS = 2;

    // The radius of chunks around the origin chunk that should be copied.
    private static final int NEIGHBOR_CHUNK_RADIUS = Mth.m_144941_(NEIGHBOR_BLOCK_RADIUS, 16) >> 4;

    // The number of sections on each axis of this slice.
    private static final int SECTION_LENGTH = 1 + (NEIGHBOR_CHUNK_RADIUS * 2);

    // The size of the lookup tables used for mapping values to coordinate int pairs. The lookup table size is always
    // a power of two so that multiplications can be replaced with simple bit shifts in hot code paths.
    private static final int TABLE_LENGTH = Mth.m_14125_(SECTION_LENGTH);

    // The number of bits needed for each X/Y/Z component in a lookup table.
    private static final int TABLE_BITS = Integer.bitCount(TABLE_LENGTH - 1);

    // The number of bits needed for each X/Y/Z block coordinate.
    private static final int BLOCK_BITS = 4;

    // The number of bits needed for each X/Y/Z biome coordinate.
    private static final int BIOME_BITS = 2;

    // The array size for the section lookup table.
    private static final int SECTION_TABLE_ARRAY_SIZE = TABLE_LENGTH * TABLE_LENGTH * TABLE_LENGTH;

    // The world this slice has copied data from
    private final Level world;

    // The accessor used for fetching biome data from the slice
    private final BiomeManager biomeAccess;

    // Local Section->BlockState table.
    private final BlockState[][] blockStatesArrays;

    // Local Section->Biome table.
    private final Holder<Biome>[][] biomeArrays;

    // Local section copies. Read-only.
    private ClonedChunkSection[] sections;

    // The biome blend cache
    private BlockColorCache biomeColors;

    // The starting point from which this slice captures blocks
    private int baseX, baseY, baseZ;

    // The chunk origin of this slice
    private SectionPos origin;

    // The volume of this slice
    private BoundingBox volume;

    public static ChunkRenderContext prepare(Level world, SectionPos origin, ClonedChunkSectionCache sectionCache) {
        LevelChunk chunk = world.m_6325_(origin.m_123341_(), origin.m_123343_());
        LevelChunkSection section = chunk.m_7103_()[world.m_151566_(origin.m_123342_())];

        // If the chunk section is absent or empty, simply terminate now. There will never be anything in this chunk
        // section to render, so we need to signal that a chunk render task shouldn't created. This saves a considerable
        // amount of time in queueing instant build tasks and greatly accelerates how quickly the world can be loaded.
        List<MeshAppender> meshAppenders = ChunkMeshEvent.post(world, origin);
        boolean isEmpty = (section == null || section.m_188008_()) && meshAppenders.isEmpty();

        if (isEmpty) {
            return null;
        }

        BoundingBox volume = new BoundingBox(origin.m_123229_() - NEIGHBOR_BLOCK_RADIUS,
                origin.m_123234_() - NEIGHBOR_BLOCK_RADIUS,
                origin.m_123239_() - NEIGHBOR_BLOCK_RADIUS,
                origin.m_123244_() + NEIGHBOR_BLOCK_RADIUS,
                origin.m_123247_() + NEIGHBOR_BLOCK_RADIUS,
                origin.m_123248_() + NEIGHBOR_BLOCK_RADIUS);

        // The min/max bounds of the chunks copied by this slice
        final int minChunkX = origin.m_123341_() - NEIGHBOR_CHUNK_RADIUS;
        final int minChunkY = origin.m_123342_() - NEIGHBOR_CHUNK_RADIUS;
        final int minChunkZ = origin.m_123343_() - NEIGHBOR_CHUNK_RADIUS;

        final int maxChunkX = origin.m_123341_() + NEIGHBOR_CHUNK_RADIUS;
        final int maxChunkY = origin.m_123342_() + NEIGHBOR_CHUNK_RADIUS;
        final int maxChunkZ = origin.m_123343_() + NEIGHBOR_CHUNK_RADIUS;

        ClonedChunkSection[] sections = new ClonedChunkSection[SECTION_TABLE_ARRAY_SIZE];

        for (int chunkX = minChunkX; chunkX <= maxChunkX; chunkX++) {
            for (int chunkZ = minChunkZ; chunkZ <= maxChunkZ; chunkZ++) {
                for (int chunkY = minChunkY; chunkY <= maxChunkY; chunkY++) {
                    sections[getLocalSectionIndex(chunkX - minChunkX, chunkY - minChunkY, chunkZ - minChunkZ)] =
                            sectionCache.acquire(chunkX, chunkY, chunkZ);
                }
            }
        }

        return new ChunkRenderContext(origin, sections, volume).withMeshAppenders(meshAppenders);
    }

    public WorldSlice(Level world) {
        this.world = world;

        this.biomeAccess = new BiomeManager(this::getStoredBiome, ((BiomeSeedProvider) this.world).getBiomeSeed());

        this.sections = new ClonedChunkSection[SECTION_TABLE_ARRAY_SIZE];
        this.blockStatesArrays = new BlockState[SECTION_TABLE_ARRAY_SIZE][SECTION_BLOCK_COUNT];
        for (BlockState[] blockStatesArray : this.blockStatesArrays) {
            Arrays.fill(blockStatesArray, Blocks.f_50016_.m_49966_());
        }
        this.biomeArrays = new Holder[SECTION_TABLE_ARRAY_SIZE][SECTION_BIOME_COUNT];

    }

    public void copyData(ChunkRenderContext context) {
        this.origin = context.getOrigin();
        this.sections = context.getSections();
        this.volume = context.getVolume();

        this.baseX = (this.origin.m_123341_() - NEIGHBOR_CHUNK_RADIUS) << 4;
        this.baseY = (this.origin.m_123342_() - NEIGHBOR_CHUNK_RADIUS) << 4;
        this.baseZ = (this.origin.m_123343_() - NEIGHBOR_CHUNK_RADIUS) << 4;

        for (int x = 0; x < SECTION_LENGTH; x++) {
            for (int y = 0; y < SECTION_LENGTH; y++) {
                for (int z = 0; z < SECTION_LENGTH; z++) {
                    int idx = getLocalSectionIndex(x, y, z);
                    this.unpackBlockData(this.blockStatesArrays[idx], this.sections[idx], context.getVolume());
                    this.unpackBiomeData(this.biomeArrays[idx], this.sections[idx]);
                }
            }
        }

        this.biomeColors = new BlockColorCache(this, Minecraft.m_91087_().f_91066_.m_232121_().m_231551_());
    }

    private void unpackBlockData(BlockState[] states, ClonedChunkSection section, BoundingBox box) {
        if (this.origin.equals(section.getPosition()))  {
            this.unpackBlockData(states, section);
        } else {
            this.unpackBlockDataSlow(states, section, box);
        }
    }

    private void unpackBlockDataSlow(BlockState[] states, ClonedChunkSection section, BoundingBox box) {
        Arrays.fill(states, Blocks.f_50016_.m_49966_());

        SimpleBitStorage intArray = section.getBlockData();
        ClonedPalette<BlockState> palette = section.getBlockPalette();

        SectionPos pos = section.getPosition();

        int minBlockX = Math.max(box.m_162395_(), pos.m_123229_());
        int maxBlockX = Math.min(box.m_162399_(), pos.m_123244_());

        int minBlockY = Math.max(box.m_162396_(), pos.m_123234_());
        int maxBlockY = Math.min(box.m_162400_(), pos.m_123247_());

        int minBlockZ = Math.max(box.m_162398_(), pos.m_123239_());
        int maxBlockZ = Math.min(box.m_162401_(), pos.m_123248_());

        for (int y = minBlockY; y <= maxBlockY; y++) {
            for (int z = minBlockZ; z <= maxBlockZ; z++) {
                for (int x = minBlockX; x <= maxBlockX; x++) {
                    int blockIdx = getLocalBlockIndex(x & 15, y & 15, z & 15);
                    int value = intArray.m_13514_(blockIdx);

                    states[blockIdx] = palette.get(value);
                }
            }
        }
    }

    private void unpackBlockData(BlockState[] states, ClonedChunkSection section) {
        ((PackedIntegerArrayExtended) section.getBlockData())
                .copyUsingPalette(states, section.getBlockPalette());
    }

    private void unpackBiomeData(Holder<Biome>[] biomes, ClonedChunkSection section) {
        for (int x = 0; x < 4; x++) {
            for (int y = 0; y < 4; y++) {
                for (int z = 0; z < 4; z++) {
                    biomes[getLocalBiomeIndex(x, y, z)] = section.getBiome(x, y, z);
                }
            }
        }
    }

    private static boolean blockBoxContains(BoundingBox box, int x, int y, int z) {
        return x >= box.m_162395_() &&
                x <= box.m_162399_() &&
                y >= box.m_162396_() &&
                y <= box.m_162400_() &&
                z >= box.m_162398_() &&
                z <= box.m_162401_();
    }

    @Override
    public BlockState m_8055_(BlockPos pos) {
        return this.getBlockState(pos.m_123341_(), pos.m_123342_(), pos.m_123343_());
    }

    public BlockState getBlockState(int x, int y, int z) {
        if (!blockBoxContains(this.volume, x, y, z)) {
            return Blocks.f_50016_.m_49966_();
        }

        int relX = x - this.baseX;
        int relY = y - this.baseY;
        int relZ = z - this.baseZ;

        return this.blockStatesArrays[getLocalSectionIndex(relX >> 4, relY >> 4, relZ >> 4)]
                [getLocalBlockIndex(relX & 15, relY & 15, relZ & 15)];
    }

    @Override
    public FluidState m_6425_(BlockPos pos) {
        return this.m_8055_(pos)
                .m_60819_();
    }

    @Override
    public float m_7717_(Direction direction, boolean shaded) {
        return this.world.m_7717_(direction, shaded);
    }

    @Override
    public LevelLightEngine m_5518_() {
        return this.world.m_5518_();
    }

    @Override
    public BlockEntity m_7702_(BlockPos pos) {
        return this.getBlockEntity(pos.m_123341_(), pos.m_123342_(), pos.m_123343_());
    }

    public BlockEntity getBlockEntity(int x, int y, int z) {
        if (!blockBoxContains(this.volume, x, y, z)) {
            return null;
        }

        int relX = x - this.baseX;
        int relY = y - this.baseY;
        int relZ = z - this.baseZ;

        return this.sections[getLocalSectionIndex(relX >> 4, relY >> 4, relZ >> 4)]
                .getBlockEntity(relX & 15, relY & 15, relZ & 15);
    }

    @Override
    public int m_6171_(BlockPos pos, ColorResolver resolver) {
        return this.biomeColors.getColor(resolver, pos.m_123341_(), pos.m_123342_(), pos.m_123343_());
    }

    @Override
    public int m_45517_(LightLayer type, BlockPos pos) {
        if (!blockBoxContains(this.volume, pos.m_123341_(), pos.m_123342_(), pos.m_123343_())) {
            return 0;
        }

        int relX = pos.m_123341_() - this.baseX;
        int relY = pos.m_123342_() - this.baseY;
        int relZ = pos.m_123343_() - this.baseZ;

        return this.sections[getLocalSectionIndex(relX >> 4, relY >> 4, relZ >> 4)]
                .getLightLevel(type, relX & 15, relY & 15, relZ & 15);
    }

    public SectionPos getOrigin() {
        return this.origin;
    }

    @Override
    public int m_141928_() {
        return this.world.m_141928_();
    }

    @Override
    public int m_141937_() {
        return this.world.m_141937_();
    }

    // Coordinates are in biome space!
    private Holder<Biome> getStoredBiome(int biomeX, int biomeY, int biomeZ) {
        int chunkX = (QuartPos.m_175402_(biomeX) - this.baseX) >> 4;
        int chunkY = (QuartPos.m_175402_(biomeY) - this.baseY) >> 4;
        int chunkZ = (QuartPos.m_175402_(biomeZ) - this.baseZ) >> 4;

        int sectionIndex = getLocalSectionIndex(chunkX, chunkY, chunkZ);

        if(sectionIndex < 0 || sectionIndex >= this.biomeArrays.length) {
            return this.world.m_5962_().m_175515_(Registry.f_122885_).m_206081_(Biomes.f_48202_);
        }

        return this.biomeArrays[sectionIndex]
                [getLocalBiomeIndex(biomeX & 3, biomeY & 3, biomeZ & 3)];
    }

    public BiomeManager getBiomeAccess() {
        return this.biomeAccess;
    }

    private static int getLocalBiomeIndex(int x, int y, int z) {
        return y << BIOME_BITS << BIOME_BITS | z << BIOME_BITS | x;
    }

    public static int getLocalBlockIndex(int x, int y, int z) {
        return y << BLOCK_BITS << BLOCK_BITS | z << BLOCK_BITS | x;
    }

    public static int getLocalSectionIndex(int x, int y, int z) {
        return y << TABLE_BITS << TABLE_BITS | z << TABLE_BITS | x;
    }
}
