package me.jellysquid.mods.sodium.client.render;

import com.mojang.blaze3d.platform.GlStateManager;
import com.mojang.blaze3d.platform.NativeImage;
import com.mojang.blaze3d.shaders.FogShape;
import com.mojang.blaze3d.systems.RenderSystem;
import com.mojang.blaze3d.vertex.BufferBuilder;
import com.mojang.blaze3d.vertex.DefaultVertexFormat;
import com.mojang.blaze3d.vertex.PoseStack;
import com.mojang.blaze3d.vertex.Tesselator;
import com.mojang.blaze3d.vertex.VertexBuffer;
import com.mojang.blaze3d.vertex.VertexFormat;
import com.mojang.math.Matrix4f;
import me.jellysquid.mods.sodium.client.model.vertex.VanillaVertexTypes;
import me.jellysquid.mods.sodium.client.model.vertex.VertexDrain;
import me.jellysquid.mods.sodium.client.model.vertex.formats.screen_quad.BasicScreenQuadVertexSink;
import me.jellysquid.mods.sodium.client.util.MathUtil;
import me.jellysquid.mods.sodium.client.util.color.ColorABGR;
import me.jellysquid.mods.sodium.client.util.color.ColorARGB;
import me.jellysquid.mods.sodium.client.util.color.ColorMixer;
import net.minecraft.client.Camera;
import net.minecraft.client.Minecraft;
import net.minecraft.client.multiplayer.ClientLevel;
import net.minecraft.client.player.LocalPlayer;
import net.minecraft.client.render.*;
import net.minecraft.client.renderer.FogRenderer;
import net.minecraft.client.renderer.ShaderInstance;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.server.packs.resources.Resource;
import net.minecraft.server.packs.resources.ResourceManager;
import net.minecraft.server.packs.resources.ResourceProvider;
import net.minecraft.util.Mth;
import net.minecraft.world.effect.MobEffectInstance;
import net.minecraft.world.effect.MobEffects;
import net.minecraft.world.level.material.FogType;
import net.minecraft.world.phys.Vec3;
import org.apache.commons.lang3.Validate;
import org.jetbrains.annotations.Nullable;
import org.lwjgl.opengl.GL30C;

import java.io.IOException;
import java.io.InputStream;

public class CloudRenderer {
    private static final ResourceLocation CLOUDS_TEXTURE_ID = new ResourceLocation("textures/environment/clouds.png");

    private static final int CLOUD_COLOR_NEG_Y = ColorABGR.pack(0.7F, 0.7F, 0.7F, 1.0f);
    private static final int CLOUD_COLOR_POS_Y = ColorABGR.pack(1.0f, 1.0f, 1.0f, 1.0f);
    private static final int CLOUD_COLOR_NEG_X = ColorABGR.pack(0.9F, 0.9F, 0.9F, 1.0f);
    private static final int CLOUD_COLOR_POS_X = ColorABGR.pack(0.9F, 0.9F, 0.9F, 1.0f);
    private static final int CLOUD_COLOR_NEG_Z = ColorABGR.pack(0.8F, 0.8F, 0.8F, 1.0f);
    private static final int CLOUD_COLOR_POS_Z = ColorABGR.pack(0.8F, 0.8F, 0.8F, 1.0f);

    private static final int DIR_NEG_Y = 1 << 0;
    private static final int DIR_POS_Y = 1 << 1;
    private static final int DIR_NEG_X = 1 << 2;
    private static final int DIR_POS_X = 1 << 3;
    private static final int DIR_NEG_Z = 1 << 4;
    private static final int DIR_POS_Z = 1 << 5;

    private VertexBuffer vertexBuffer;
    private CloudEdges edges;
    private ShaderInstance clouds;
    private final FogRenderer.FogData fogData = new FogRenderer.FogData(FogRenderer.FogMode.FOG_TERRAIN);

    private int prevCenterCellX, prevCenterCellY, cachedRenderDistance;

    public CloudRenderer(ResourceProvider factory) {
        this.reloadTextures(factory);
    }

    public void render(@Nullable ClientLevel world, LocalPlayer player, PoseStack matrices, Matrix4f projectionMatrix, float ticks, float tickDelta, double cameraX, double cameraY, double cameraZ) {
        if (world == null) {
            return;
        }

        float cloudHeight = world.m_104583_().m_108871_();

        if (Float.isNaN(cloudHeight)) {
            return;
        }

        Vec3 color = world.m_104808_(tickDelta);

        double cloudTime = (ticks + tickDelta) * 0.03F;
        double cloudCenterX = (cameraX + cloudTime);
        double cloudCenterZ = (cameraZ) + 0.33D;

        int renderDistance = Minecraft.m_91087_().f_91066_.m_193772_();
        int cloudDistance = Math.max(32, (renderDistance * 2) + 9);

        int centerCellX = (int) (Math.floor(cloudCenterX / 12));
        int centerCellZ = (int) (Math.floor(cloudCenterZ / 12));

        if (this.vertexBuffer == null || this.prevCenterCellX != centerCellX || this.prevCenterCellY != centerCellZ || this.cachedRenderDistance != renderDistance) {
            BufferBuilder bufferBuilder = Tesselator.m_85913_().m_85915_();
            bufferBuilder.m_166779_(VertexFormat.Mode.QUADS, DefaultVertexFormat.f_85815_);

            this.rebuildGeometry(bufferBuilder, cloudDistance, centerCellX, centerCellZ);

            if (this.vertexBuffer == null) {
                this.vertexBuffer = new VertexBuffer();
            }

            this.vertexBuffer.m_85921_();
            this.vertexBuffer.m_231221_(bufferBuilder.m_231175_());

            VertexBuffer.m_85931_();

            this.prevCenterCellX = centerCellX;
            this.prevCenterCellY = centerCellZ;
            this.cachedRenderDistance = renderDistance;
        }

        float previousEnd = RenderSystem.m_157199_();
        float previousStart = RenderSystem.m_157200_();
        fogData.f_234201_ = cloudDistance * 8;
        fogData.f_234200_ = (cloudDistance * 8) - 16;

        applyFogModifiers(world, fogData, player, cloudDistance * 8, tickDelta);


        RenderSystem.m_157443_(fogData.f_234201_);
        RenderSystem.m_157445_(fogData.f_234200_);

        float translateX = (float) (cloudCenterX - (centerCellX * 12));
        float translateZ = (float) (cloudCenterZ - (centerCellZ * 12));

        RenderSystem.m_69482_();

        this.vertexBuffer.m_85921_();

        boolean insideClouds = cameraY < cloudHeight + 4.5f && cameraY > cloudHeight - 0.5f;

        if (insideClouds) {
            RenderSystem.m_69464_();
        } else {
            RenderSystem.m_69481_();
        }

        RenderSystem.m_69472_();
        RenderSystem.m_157429_((float) color.f_82479_, (float) color.f_82480_, (float) color.f_82481_, 0.8f);

        matrices.m_85836_();

        Matrix4f modelViewMatrix = matrices.m_85850_().m_85861_();
        //Matrix4f.translate(-translateX, cloudHeight - (float) cameraY + 0.33F, -translateZ);
        // TODO
        modelViewMatrix.m_162199_(-translateX, cloudHeight - (float) cameraY + 0.33F, -translateZ);

        // PASS 1: Set up depth buffer
        RenderSystem.m_69461_();
        RenderSystem.m_69458_(true);
        RenderSystem.m_69444_(false, false, false, false);

        this.vertexBuffer.m_166867_(modelViewMatrix, projectionMatrix, clouds);

        // PASS 2: Render geometry
        RenderSystem.m_69478_();
        RenderSystem.m_69416_(GlStateManager.SourceFactor.SRC_ALPHA, GlStateManager.DestFactor.ONE_MINUS_SRC_ALPHA, GlStateManager.SourceFactor.ONE, GlStateManager.DestFactor.ONE_MINUS_SRC_ALPHA);
        RenderSystem.m_69458_(false);
        RenderSystem.m_69482_();
        RenderSystem.m_69456_(GL30C.GL_EQUAL);
        RenderSystem.m_69444_(true, true, true, true);

        this.vertexBuffer.m_166867_(modelViewMatrix, projectionMatrix, clouds);

        matrices.m_85849_();

        VertexBuffer.m_85931_();

        RenderSystem.m_69461_();
        RenderSystem.m_69456_(GL30C.GL_LEQUAL);

        RenderSystem.m_69481_();

        RenderSystem.m_157443_(previousEnd);
        RenderSystem.m_157445_(previousStart);
    }

    private void applyFogModifiers(ClientLevel world, FogRenderer.FogData fogData, LocalPlayer player, int cloudDistance, float tickDelta) {
        if (Minecraft.m_91087_().f_91063_ == null || Minecraft.m_91087_().f_91063_.m_109153_() == null) {
            return;
        }

        Camera camera = Minecraft.m_91087_().f_91063_.m_109153_();
        FogType cameraSubmersionType = camera.m_167685_();
        if (cameraSubmersionType == FogType.LAVA) {
            if (player.m_5833_()) {
                fogData.f_234200_ = -8.0f;
                fogData.f_234201_ = (cloudDistance) * 0.5f;
            } else if (player.m_21023_(MobEffects.f_19607_)) {
                fogData.f_234200_ = 0.0f;
                fogData.f_234201_ = 3.0f;
            } else {
                fogData.f_234200_ = 0.25f;
                fogData.f_234201_ = 1.0f;
            }
        } else if (cameraSubmersionType == FogType.POWDER_SNOW) {
            if (player.m_5833_()) {
                fogData.f_234200_ = -8.0f;
                fogData.f_234201_ = (cloudDistance) * 0.5f;
            } else {
                fogData.f_234200_ = 0.0f;
                fogData.f_234201_ = 2.0f;
            }
        } else if (cameraSubmersionType == FogType.WATER) {
            fogData.f_234200_ = -8.0f;
            fogData.f_234201_ = 96.0f;
            fogData.f_234201_ *= Math.max(0.25f, player.m_108639_());
            if (fogData.f_234201_ > (cloudDistance)) {
                fogData.f_234201_ = cloudDistance;
                fogData.f_234202_ = FogShape.CYLINDER;
            }
        } else if (world.m_104583_().m_5781_(Mth.m_14107_(camera.m_90583_().f_82479_), Mth.m_14107_(camera.m_90583_().f_82481_)) || Minecraft.m_91087_().f_91065_.m_93090_().m_93715_()) {
            fogData.f_234200_ = (cloudDistance) * 0.05f;
            fogData.f_234201_ = Math.min((cloudDistance), 192.0f) * 0.5f;
        }

        FogRenderer.MobEffectFogFunction fogModifier = FogRenderer.m_234165_(player, tickDelta);
        if (fogModifier != null) {
            MobEffectInstance statusEffectInstance = player.m_21124_(fogModifier.m_213948_());
            if (statusEffectInstance != null) {
                fogModifier.m_213725_(fogData, player, statusEffectInstance, (cloudDistance * 8), tickDelta);
            }
        }
    }

    private void rebuildGeometry(BufferBuilder bufferBuilder, int cloudDistance, int centerCellX, int centerCellZ) {
        var sink = VertexDrain.of(bufferBuilder)
                .createSink(VanillaVertexTypes.BASIC_SCREEN_QUADS);

        for (int offsetX = -cloudDistance; offsetX < cloudDistance; offsetX++) {
            for (int offsetZ = -cloudDistance; offsetZ < cloudDistance; offsetZ++) {
                int connectedEdges = this.edges.getEdges(centerCellX + offsetX, centerCellZ + offsetZ);

                if (connectedEdges == 0) {
                    continue;
                }

                int baseColor = this.edges.getColor(centerCellX + offsetX, centerCellZ + offsetZ);

                float x = offsetX * 12;
                float z = offsetZ * 12;

                // -Y
                if ((connectedEdges & DIR_NEG_Y) != 0) {
                    int mixedColor = ColorMixer.mulARGB(baseColor, CLOUD_COLOR_NEG_Y);
                    sink.ensureCapacity(4);
                    sink.writeQuad(x + 12, 0.0f, z + 12, mixedColor);
                    sink.writeQuad(x + 0.0f, 0.0f, z + 12, mixedColor);
                    sink.writeQuad(x + 0.0f, 0.0f, z + 0.0f, mixedColor);
                    sink.writeQuad(x + 12, 0.0f, z + 0.0f, mixedColor);
                }

                // +Y
                if ((connectedEdges & DIR_POS_Y) != 0) {
                    int mixedColor = ColorMixer.mulARGB(baseColor, CLOUD_COLOR_POS_Y);
                    sink.ensureCapacity(4);
                    sink.writeQuad(x + 0.0f, 4.0f, z + 12, mixedColor);
                    sink.writeQuad(x + 12, 4.0f, z + 12, mixedColor);
                    sink.writeQuad(x + 12, 4.0f, z + 0.0f, mixedColor);
                    sink.writeQuad(x + 0.0f, 4.0f, z + 0.0f, mixedColor);
                }

                // -X
                if ((connectedEdges & DIR_NEG_X) != 0) {
                    int mixedColor = ColorMixer.mulARGB(baseColor, CLOUD_COLOR_NEG_X);
                    sink.ensureCapacity(4);
                    sink.writeQuad(x + 0.0f, 0.0f, z + 12, mixedColor);
                    sink.writeQuad(x + 0.0f, 4.0f, z + 12, mixedColor);
                    sink.writeQuad(x + 0.0f, 4.0f, z + 0.0f, mixedColor);
                    sink.writeQuad(x + 0.0f, 0.0f, z + 0.0f, mixedColor);
                }

                // +X
                if ((connectedEdges & DIR_POS_X) != 0) {
                    int mixedColor = ColorMixer.mulARGB(baseColor, CLOUD_COLOR_POS_X);
                    sink.ensureCapacity(4);
                    sink.writeQuad(x + 12, 4.0f, z + 12, mixedColor);
                    sink.writeQuad(x + 12, 0.0f, z + 12, mixedColor);
                    sink.writeQuad(x + 12, 0.0f, z + 0.0f, mixedColor);
                    sink.writeQuad(x + 12, 4.0f, z + 0.0f, mixedColor);
                }

                // -Z
                if ((connectedEdges & DIR_NEG_Z) != 0) {
                    int mixedColor = ColorMixer.mulARGB(baseColor, CLOUD_COLOR_NEG_Z);
                    sink.ensureCapacity(4);
                    sink.writeQuad(x + 12, 4.0f, z + 0.0f, mixedColor);
                    sink.writeQuad(x + 12, 0.0f, z + 0.0f, mixedColor);
                    sink.writeQuad(x + 0.0f, 0.0f, z + 0.0f, mixedColor);
                    sink.writeQuad(x + 0.0f, 4.0f, z + 0.0f, mixedColor);
                }

                // +Z
                if ((connectedEdges & DIR_POS_Z) != 0) {
                    int mixedColor = ColorMixer.mulARGB(baseColor, CLOUD_COLOR_POS_Z);
                    sink.ensureCapacity(4);
                    sink.writeQuad(x + 12, 0.0f, z + 12, mixedColor);
                    sink.writeQuad(x + 12, 4.0f, z + 12, mixedColor);
                    sink.writeQuad(x + 0.0f, 4.0f, z + 12, mixedColor);
                    sink.writeQuad(x + 0.0f, 0.0f, z + 12, mixedColor);
                }
            }
        }

        sink.flush();
    }

    public void reloadTextures(ResourceProvider factory) {
        this.edges = createCloudEdges();

        if (clouds != null) {
            clouds.close();
        }

        try {
            this.clouds = new ShaderInstance(factory, "clouds", DefaultVertexFormat.f_85815_);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }

        if (this.vertexBuffer != null) {
            this.vertexBuffer.close();
            this.vertexBuffer = null;
        }
    }

    public void destroy() {
        clouds.close();
        if (this.vertexBuffer != null) {
            this.vertexBuffer.close();
            this.vertexBuffer = null;
        }
    }

    private static CloudEdges createCloudEdges() {
        NativeImage nativeImage;

        ResourceManager resourceManager = Minecraft.m_91087_().m_91098_();
        Resource resource = resourceManager.m_213713_(CLOUDS_TEXTURE_ID)
                .orElseThrow();

        try (InputStream inputStream = resource.m_215507_()){
            nativeImage = NativeImage.m_85058_(inputStream);
        } catch (IOException ex) {
            throw new RuntimeException("Failed to load texture data", ex);
        }

        return new CloudEdges(nativeImage);
    }

    private static class CloudEdges {
        private final byte[] edges;
        private final int[] colors;
        private final int width, height;

        public CloudEdges(NativeImage texture) {
            int width = texture.m_84982_();
            int height = texture.m_85084_();

            Validate.isTrue(MathUtil.isPowerOfTwo(width), "Texture width must be power-of-two");
            Validate.isTrue(MathUtil.isPowerOfTwo(height), "Texture height must be power-of-two");

            this.edges = new byte[width * height];
            this.colors = new int[width * height];

            this.width = width;
            this.height = height;

            for (int x = 0; x < width; x++) {
                for (int z = 0; z < height; z++) {
                    int index = index(x, z, width, height);
                    int cell = texture.m_84985_(x, z);

                    this.colors[index] = cell;

                    int edges = 0;

                    if (isOpaqueCell(cell)) {
                        edges |= DIR_NEG_Y | DIR_POS_Y;

                        int negX = texture.m_84985_(wrap(x - 1, width), wrap(z, height));

                        if (cell != negX) {
                            edges |= DIR_NEG_X;
                        }

                        int posX = texture.m_84985_(wrap(x + 1, width), wrap(z, height));

                        if (!isOpaqueCell(posX) && cell != posX) {
                            edges |= DIR_POS_X;
                        }

                        int negZ = texture.m_84985_(wrap(x, width), wrap(z - 1, height));

                        if (cell != negZ) {
                            edges |= DIR_NEG_Z;
                        }

                        int posZ = texture.m_84985_(wrap(x, width), wrap(z + 1, height));

                        if (!isOpaqueCell(posZ) && cell != posZ) {
                            edges |= DIR_POS_Z;
                        }
                    }

                    this.edges[index] = (byte) edges;
                }
            }
        }

        private static boolean isOpaqueCell(int color) {
            return ColorARGB.unpackAlpha(color) > 1;
        }

        public int getEdges(int x, int z) {
            return this.edges[index(x, z, this.width, this.height)];
        }

        public int getColor(int x, int z) {
            return this.colors[index(x, z, this.width, this.height)];
        }

        private static int index(int posX, int posZ, int width, int height) {
            return (wrap(posX, width) * width) + wrap(posZ, height);
        }

        private static int wrap(int pos, int dim) {
            return (pos & (dim - 1));
        }
    }
}