package net.zestyblaze.lootr.command;

import com.google.common.collect.Lists;
import com.mojang.authlib.GameProfile;
import com.mojang.brigadier.arguments.StringArgumentType;
import com.mojang.brigadier.builder.LiteralArgumentBuilder;
import com.mojang.brigadier.builder.RequiredArgumentBuilder;
import net.fabricmc.fabric.api.command.v2.CommandRegistrationCallback;
import net.minecraft.ChatFormatting;
import net.minecraft.commands.CommandSourceStack;
import net.minecraft.commands.Commands;
import net.minecraft.commands.SharedSuggestionProvider;
import net.minecraft.commands.arguments.ResourceLocationArgument;
import net.minecraft.commands.arguments.coordinates.BlockPosArgument;
import net.minecraft.commands.arguments.coordinates.Vec3Argument;
import net.minecraft.core.BlockPos;
import net.minecraft.core.Direction;
import net.minecraft.core.NonNullList;
import net.minecraft.network.chat.Component;
import net.minecraft.network.chat.ComponentUtils;
import net.minecraft.network.chat.Style;
import net.minecraft.network.chat.TextColor;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.server.level.ServerLevel;
import net.minecraft.server.level.ServerPlayer;
import net.minecraft.world.entity.Entity;
import net.minecraft.world.item.ItemStack;
import net.minecraft.world.level.ChunkPos;
import net.minecraft.world.level.Level;
import net.minecraft.world.level.block.*;
import net.minecraft.world.level.block.entity.*;
import net.minecraft.world.level.block.state.BlockState;
import net.minecraft.world.level.block.state.properties.BlockStateProperties;
import net.minecraft.world.level.block.state.properties.EnumProperty;
import net.minecraft.world.level.chunk.LevelChunk;
import net.minecraft.world.level.levelgen.structure.BoundingBox;
import net.minecraft.world.level.storage.loot.BuiltInLootTables;
import net.minecraft.world.phys.Vec3;
import net.zestyblaze.lootr.api.blockentity.ILootBlockEntity;
import net.zestyblaze.lootr.block.LootrBarrelBlock;
import net.zestyblaze.lootr.block.LootrChestBlock;
import net.zestyblaze.lootr.block.LootrShulkerBlock;
import net.zestyblaze.lootr.block.entities.LootrInventoryBlockEntity;
import net.zestyblaze.lootr.config.ConfigManager;
import net.zestyblaze.lootr.data.ChestData;
import net.zestyblaze.lootr.data.DataStorage;
import net.zestyblaze.lootr.data.SpecialChestInventory;
import net.zestyblaze.lootr.entity.LootrChestMinecartEntity;
import net.zestyblaze.lootr.init.ModBlocks;
import net.zestyblaze.lootr.init.ModEntities;
import net.zestyblaze.lootr.util.ChestUtil;
import net.zestyblaze.lootr.util.ServerAccessImpl;
import org.jetbrains.annotations.Nullable;

import java.util.*;
import java.util.stream.Collectors;

public class CommandLootr {
  private static final Map<String, UUID> profileMap = new HashMap<>();
  private static List<ResourceLocation> tables = null;
  private static List<String> tableNames = null;

  private static List<ResourceLocation> getTables() {
    if (tables == null) {
      tables = new ArrayList<>(BuiltInLootTables.all());
      tableNames = tables.stream().map(ResourceLocation::toString).collect(Collectors.toList());
    }
    return tables;
  }

  private static List<String> getProfiles() {
    return Lists.newArrayList(ServerAccessImpl.getServer().getProfileCache().profilesByName.keySet());
  }

  private static List<String> getTableNames() {
    getTables();
    return tableNames;
  }

  public static void createBlock(CommandSourceStack c, @Nullable Block block, @Nullable ResourceLocation incomingTable) {
    Level world = c.getLevel();
    Vec3 incomingPos = c.getPosition();
    BlockPos pos = new BlockPos((int) incomingPos.x, (int) incomingPos.y, (int) incomingPos.z);
    ResourceLocation table;
    if (incomingTable == null) {
      table = getTables().get(world.getRandom().nextInt(getTables().size()));
    } else {
      table = incomingTable;
    }
    if (block == null) {
      LootrChestMinecartEntity cart = new LootrChestMinecartEntity(ModEntities.LOOTR_MINECART_ENTITY, pos.getX() + 0.5, pos.getY() + 0.5, pos.getZ() + 0.5, world);
      Entity e = c.getEntity();
      if (e != null) {
        cart.setYRot(e.getYRot());
      }
      cart.setLootTable(table, world.getRandom().nextLong());
      world.addFreshEntity(cart);
      c.sendSuccess(() -> Component.translatable("lootr.commands.summon", ComponentUtils.wrapInSquareBrackets(Component.translatable("lootr.commands.blockpos", pos.getX(), pos.getY(), pos.getZ()).setStyle(Style.EMPTY.withColor(TextColor.fromLegacyFormat(ChatFormatting.GREEN)).withBold(true))), table.toString()), false);
    } else {
      BlockState placementState = block.defaultBlockState();
      Entity e = c.getEntity();
      if (e != null) {
        EnumProperty<Direction> prop = null;
        Direction dir = Direction.orderedByNearest(e)[0].getOpposite();
        if (placementState.hasProperty(LootrBarrelBlock.FACING)) {
          prop = LootrBarrelBlock.FACING;
        } else if (placementState.hasProperty(LootrChestBlock.FACING)) {
          prop = LootrChestBlock.FACING;
          dir = e.getDirection().getOpposite();
        } else if (placementState.hasProperty(LootrShulkerBlock.FACING)) {
          prop = LootrShulkerBlock.FACING;
        }
        if (prop != null) {
          placementState = placementState.setValue(prop, dir);
        }
      }
      world.setBlock(pos, placementState, 2);
      RandomizableContainerBlockEntity.setLootTable(world, world.getRandom(), pos, table);
      c.sendSuccess(() -> Component.translatable("lootr.commands.create", Component.translatable(block.getDescriptionId()), ComponentUtils.wrapInSquareBrackets(Component.translatable("lootr.commands.blockpos", pos.getX(), pos.getY(), pos.getZ()).setStyle(Style.EMPTY.withColor(TextColor.fromLegacyFormat(ChatFormatting.GREEN)).withBold(true))), table.toString()), false);
    }
  }

  private static RequiredArgumentBuilder<CommandSourceStack, ResourceLocation> suggestTables() {
    return Commands.argument("table", ResourceLocationArgument.id())
        .suggests((c, build) -> SharedSuggestionProvider.suggest(getTableNames(), build));
  }

  private static RequiredArgumentBuilder<CommandSourceStack, String> suggestProfiles() {
    return Commands.argument("profile", StringArgumentType.string()).suggests((c, build) -> SharedSuggestionProvider.suggest(getProfiles(), build));
  }

  public static LiteralArgumentBuilder<CommandSourceStack> builder(LiteralArgumentBuilder<CommandSourceStack> builder) {
    builder.executes(c -> {
      c.getSource().sendSuccess(() -> Component.translatable("lootr.commands.usage"), false);
      return 1;
    });
    builder.then(Commands.literal("barrel").executes(c -> {
      createBlock(c.getSource(), ModBlocks.BARREL, null);
      return 1;
    }).then(suggestTables().executes(c -> {
      createBlock(c.getSource(), ModBlocks.BARREL, ResourceLocationArgument.getId(c, "table"));
      return 1;
    })));
    builder.then(Commands.literal("trapped_chest").executes(c -> {
      createBlock(c.getSource(), ModBlocks.TRAPPED_CHEST, null);
      return 1;
    }).then(suggestTables().executes(c -> {
      createBlock(c.getSource(), ModBlocks.TRAPPED_CHEST, ResourceLocationArgument.getId(c, "table"));
      return 1;
    })));
    builder.then(Commands.literal("chest").executes(c -> {
      createBlock(c.getSource(), ModBlocks.CHEST, null);
      return 1;
    }).then(suggestTables().executes(c -> {
      createBlock(c.getSource(), ModBlocks.CHEST, ResourceLocationArgument.getId(c, "table"));
      return 1;
    })));
    builder.then(Commands.literal("shulker").executes(c -> {
      createBlock(c.getSource(), ModBlocks.SHULKER, null);
      return 1;
    }).then(suggestTables().executes(c -> {
      createBlock(c.getSource(), ModBlocks.SHULKER, ResourceLocationArgument.getId(c, "table"));
      return 1;
    })));
    builder.then(Commands.literal("clear").executes(c -> {
      c.getSource().sendSuccess(() -> Component.literal("Must provide player name."), true);
      return 1;
    }).then(suggestProfiles().executes(c -> {
      String playerName = StringArgumentType.getString(c, "profile");
      Optional<GameProfile> opt_profile = c.getSource().getServer().getProfileCache().get(playerName);
      if (!opt_profile.isPresent()) {
        c.getSource().sendFailure(Component.literal("Invalid player name: " + playerName + ", profile not found in the cache."));
        return 0;
      }
      GameProfile profile = opt_profile.get();
      c.getSource().sendSuccess(() -> Component.literal(DataStorage.clearInventories(profile.getId()) ? "Cleared stored inventories for " + playerName : "No stored inventories for " + playerName + " to clear"), true);
      return 1;
    })));
    builder.then(Commands.literal("cart").executes(c -> {
      createBlock(c.getSource(), null, null);
      return 1;
    }).then(suggestTables().executes(c -> {
      createBlock(c.getSource(), null, ResourceLocationArgument.getId(c, "table"));
      return 1;
    })));
    builder.then(Commands.literal("custom").executes(c -> {
      BlockPos pos = BlockPos.containing(c.getSource().getPosition());
      Level world = c.getSource().getLevel();
      BlockState state = world.getBlockState(pos);
      if (!state.is(Blocks.CHEST) && !state.is(Blocks.BARREL)) {
        pos = pos.below();
        state = world.getBlockState(pos);
      }
      if (!state.is(Blocks.CHEST) && !state.is(Blocks.BARREL)) {
        c.getSource().sendSuccess(() -> Component.literal("Please stand on the chest or barrel you wish to convert."), false);
      } else {
        NonNullList<ItemStack> reference;
        BlockState newState;
        if (state.is(Blocks.CHEST)) {
          reference = ((ChestBlockEntity) Objects.requireNonNull(world.getBlockEntity(pos))).items;
          newState = ModBlocks.INVENTORY.defaultBlockState().setValue(ChestBlock.FACING, state.getValue(ChestBlock.FACING)).setValue(ChestBlock.WATERLOGGED, state.getValue(ChestBlock.WATERLOGGED));
        } else {
          Direction facing = state.getValue(BarrelBlock.FACING);
          if (facing == Direction.UP || facing == Direction.DOWN) {
            facing = Direction.NORTH;
          }
          reference = ((BarrelBlockEntity) Objects.requireNonNull(world.getBlockEntity(pos))).items;
          newState = ModBlocks.INVENTORY.defaultBlockState().setValue(ChestBlock.FACING, facing);
        }
        NonNullList<ItemStack> custom = ChestUtil.copyItemList(reference);
        world.removeBlockEntity(pos);
        world.setBlockAndUpdate(pos, newState);
        BlockEntity te = world.getBlockEntity(pos);
        if (!(te instanceof LootrInventoryBlockEntity inventory)) {
          c.getSource().sendSuccess(() -> Component.literal("Unable to convert chest, BlockState is not a Lootr Inventory block."), false);
        } else {
          inventory.setCustomInventory(custom);
          inventory.setChanged();
        }
      }
      return 1;
    }));
    builder.then(Commands.literal("open_as").executes(c -> {
      c.getSource().sendSuccess(() -> Component.literal("Must provide player name."), true);
      return 1;
    }).then(suggestProfiles().executes(c -> {
      String playerName = StringArgumentType.getString(c, "profile");
      Optional<GameProfile> opt_profile = c.getSource().getServer().getProfileCache().get(playerName);
      if (!opt_profile.isPresent()) {
        c.getSource().sendFailure(Component.literal("Invalid player name: " + playerName + ", profile not found in the cache."));
        return 0;
      }
      GameProfile profile = opt_profile.get();
      BlockPos pos = BlockPos.containing(c.getSource().getPosition());
      Level level = c.getSource().getLevel();
      BlockEntity te = level.getBlockEntity(pos);
      if (!(te instanceof ILootBlockEntity)) {
        pos = pos.below();
        te = level.getBlockEntity(pos);
      }
      if (!(te instanceof ILootBlockEntity ibe)) {
        c.getSource().sendSuccess(() -> Component.literal("Please stand on a valid Lootr container."), false);
        return 0;
      }

      ChestData data = DataStorage.getContainerData((ServerLevel) level, te.getBlockPos(), ibe.getTileId());
      SpecialChestInventory inventory = data.getInventory(profile.getId());
      if (inventory == null) {
        c.getSource().sendSuccess(() -> Component.literal("No stored inventory for " + playerName + " found."), true);
        return 0;
      }

      ServerPlayer player = c.getSource().getPlayer();
      if (player == null) {
        c.getSource().sendSuccess(() -> Component.literal("Command can only be executed by a player"), false);
        return 0;
      }

      player.openMenu(inventory);

      return 1;
    })));
    builder.then(Commands.literal("open_as_uuid").executes(c -> {
      c.getSource().sendSuccess(() -> Component.literal("Must provide player UUID."), true);
      return 1;
    }).then(Commands.argument("uuid", StringArgumentType.string()).executes(c -> {
      String uuid = StringArgumentType.getString(c, "uuid");
      UUID id;
      try {
        id = UUID.fromString(uuid);
      } catch (IllegalArgumentException exception) {
        c.getSource().sendFailure(Component.literal("Invalid UUID: " + uuid));
        return 0;
      }
      BlockPos pos = BlockPos.containing(c.getSource().getPosition());
      Level level = c.getSource().getLevel();
      BlockEntity te = level.getBlockEntity(pos);
      if (!(te instanceof ILootBlockEntity)) {
        pos = pos.below();
        te = level.getBlockEntity(pos);
      }
      if (!(te instanceof ILootBlockEntity ibe)) {
        c.getSource().sendSuccess(() -> Component.literal("Please stand on a valid Lootr container."), false);
        return 0;
      }

      ChestData data = DataStorage.getContainerData((ServerLevel) level, te.getBlockPos(), ibe.getTileId());
      SpecialChestInventory inventory = data.getInventory(id);
      if (inventory == null) {
        c.getSource().sendSuccess(() -> Component.literal("No stored inventory for " + id + " found."), true);
        return 0;
      }

      ServerPlayer player = c.getSource().getPlayer();
      if (player == null) {
        c.getSource().sendSuccess(() -> Component.literal("Command can only be executed by a player"), false);
        return 0;
      }

      player.openMenu(inventory);

      return 1;
    })));
    builder.then(Commands.literal("id").executes(c -> {
      BlockPos pos = BlockPos.containing(c.getSource().getPosition());
      Level world = c.getSource().getLevel();
      BlockEntity te = world.getBlockEntity(pos);
      if (!(te instanceof ILootBlockEntity)) {
        pos = pos.below();
        te = world.getBlockEntity(pos);
      }
      if (!(te instanceof ILootBlockEntity ibe)) {
        c.getSource().sendSuccess(() -> Component.literal("Please stand on a valid Lootr container."), false);
      } else {
        c.getSource().sendSuccess(() -> Component.literal("The ID of this inventory is: ").append(ComponentUtils.copyOnClickText(ibe.getTileId().toString())), false);
      }
      return 1;
    }));
    builder.then(Commands.literal("refresh").executes(c -> {
      BlockPos pos = BlockPos.containing(c.getSource().getPosition());
      Level level = c.getSource().getLevel();
      BlockEntity be = level.getBlockEntity(pos);
      if (!(be instanceof ILootBlockEntity)) {
        pos = pos.below();
        be = level.getBlockEntity(pos);
      }
      if (be instanceof ILootBlockEntity ibe) {
        DataStorage.setRefreshing(((ILootBlockEntity) be).getTileId(), ConfigManager.get().refresh.refresh_value);
        c.getSource().sendSuccess(() -> Component.literal("Container with ID " + (ibe).getTileId() + " has been set to refresh with a delay of " + ConfigManager.get().refresh.refresh_value), false);
      } else {
        c.getSource().sendSuccess(() -> Component.literal("Please stand on a valid Lootr container."), false);
      }
      return 1;
    }));
    builder.then(Commands.literal("decay").executes(c -> {
      BlockPos pos = BlockPos.containing(c.getSource().getPosition());
      Level level = c.getSource().getLevel();
      BlockEntity be = level.getBlockEntity(pos);
      if (!(be instanceof ILootBlockEntity)) {
        pos = pos.below();
        be = level.getBlockEntity(pos);
      }
      if (be instanceof ILootBlockEntity ibe) {
        DataStorage.setDecaying((ibe).getTileId(), ConfigManager.get().decay.decay_value);
        c.getSource().sendSuccess(() -> Component.literal("Container with ID " + (ibe).getTileId() + " has been set to decay with a delay of " + ConfigManager.get().decay.decay_value), false);
      } else {
        c.getSource().sendSuccess(() -> Component.literal("Please stand on a valid Lootr container."), false);
      }
      return 1;
    }));
    builder.then(Commands.literal("openers").then(Commands.argument("location", Vec3Argument.vec3()).executes(c -> {
      BlockPos position = Vec3Argument.getCoordinates(c, "location").getBlockPos(c.getSource());
      Level world = c.getSource().getLevel();
      BlockEntity tile = world.getBlockEntity(position);
      if (tile instanceof ILootBlockEntity ibe) {
        Set<UUID> openers = ((ILootBlockEntity) tile).getOpeners();
        c.getSource().sendSuccess(() -> Component.literal("Tile at location " + position + " has " + openers.size() + " openers. UUIDs as follows:"), true);
        for (UUID uuid : openers) {
          Optional<GameProfile> prof = c.getSource().getServer().getProfileCache().get(uuid);
          c.getSource().sendSuccess(() -> Component.literal("UUID: " + uuid.toString() + ", user profile: " + (prof.isPresent() ? prof.get().getName() : "null")), true);
        }
      } else {
        c.getSource().sendSuccess(() -> Component.literal("No Lootr tile exists at location: " + position), false);
      }
      return 1;
    })));
    builder.then(Commands.literal("convert").then(Commands.argument("from", BlockPosArgument.blockPos()).then(Commands.argument("to", BlockPosArgument.blockPos()).executes(context -> {
      BoundingBox bounds = BoundingBox.fromCorners(BlockPosArgument.getLoadedBlockPos(context, "from"), BlockPosArgument.getLoadedBlockPos(context, "to"));
      ChunkPos start = new ChunkPos(new BlockPos(bounds.minX(), bounds.minY(), bounds.minZ()));
      ChunkPos stop = new ChunkPos(new BlockPos(bounds.maxX(), bounds.maxY(), bounds.maxZ()));
      List<ChunkPos> positions = new ArrayList<>();
      for (int x = start.x; x <= stop.x; x++) {
        for (int z = start.z; z <= stop.z; z++) {
          positions.add(new ChunkPos(x, z));
        }
      }
      ServerLevel level = context.getSource().getLevel();
      for (ChunkPos chunkPos : positions) {
        LevelChunk chunk = level.getChunk(chunkPos.x, chunkPos.z);
        List<BlockPos> convertableBlocks = new ArrayList<>();
        for (BlockPos pos : chunk.getBlockEntitiesPos()) {
          if (!bounds.isInside(pos)) {
            continue;
          }
          convertableBlocks.add(pos);
        }
        if (convertableBlocks.isEmpty()) {
          continue;
        }
        for (BlockPos pos : convertableBlocks) {
          BlockEntity blockEntity = chunk.getBlockEntity(pos, LevelChunk.EntityCreationType.IMMEDIATE);
          if (!(blockEntity instanceof BaseContainerBlockEntity) || blockEntity instanceof ILootBlockEntity) {
            continue;
          }
          if (blockEntity instanceof RandomizableContainerBlockEntity lootContainer) {
            if (lootContainer.lootTable != null) {
              continue;
            }
          }
          BlockState state = blockEntity.getBlockState();
          NonNullList<ItemStack> reference;
          if (blockEntity instanceof BarrelBlockEntity barrelBlock) {
            reference = barrelBlock.items;
          } else if (blockEntity instanceof ChestBlockEntity chestBlock) {
            reference = chestBlock.items;
          } else {
            continue;
          }
          BlockState newState = updateBlockState(state, ModBlocks.INVENTORY.defaultBlockState());
          NonNullList<ItemStack> custom = copyItemList(reference);
          level.removeBlockEntity(pos);
          level.setBlockAndUpdate(pos, newState);
          BlockEntity te = level.getBlockEntity(pos);
          if (!(te instanceof LootrInventoryBlockEntity inventory)) {
            context.getSource().sendSuccess(() -> Component.literal("Unable to convert chest, BlockState is not a Lootr Inventory block."), false);
          } else {
            inventory.setCustomInventory(custom);
            inventory.setChanged();
          }
        }
      }

      return 1;
    }))));
    return builder;
  }

  private static BlockState updateBlockState(BlockState oldState, BlockState newState) {
    if (oldState.hasProperty(BlockStateProperties.FACING) && newState.hasProperty(BlockStateProperties.FACING)) {
      newState = newState.setValue(BlockStateProperties.FACING, oldState.getValue(BlockStateProperties.FACING));
    }
    if (oldState.hasProperty(HorizontalDirectionalBlock.FACING) && newState.hasProperty(HorizontalDirectionalBlock.FACING)) {
      newState = newState.setValue(HorizontalDirectionalBlock.FACING, oldState.getValue(HorizontalDirectionalBlock.FACING));
    }
    if (oldState.hasProperty(BlockStateProperties.WATERLOGGED) && newState.hasProperty(BlockStateProperties.WATERLOGGED)) {
      newState = newState.setValue(BlockStateProperties.WATERLOGGED, oldState.getValue(BlockStateProperties.WATERLOGGED));
    }
    return newState;
  }

  private static NonNullList<ItemStack> copyItemList(NonNullList<ItemStack> reference) {
    NonNullList<ItemStack> contents = NonNullList.withSize(reference.size(), ItemStack.EMPTY);
    for (int i = 0; i < reference.size(); i++) {
      contents.set(i, reference.get(i).copy());
    }
    return contents;
  }

  public static void registerCommands() {
    CommandRegistrationCallback.EVENT.register(((dispatcher, reg, env) -> dispatcher.register(builder(Commands.literal("lootr").requires(p -> p.hasPermission(2))))));
  }
}
