package net.darkhax.simplelootviewer.common.impl.data;

import com.google.common.collect.ArrayListMultimap;
import net.darkhax.bookshelf.common.api.loot.LootPoolEntryDescriptions;
import net.darkhax.bookshelf.common.impl.Constants;
import net.darkhax.simplelootviewer.common.impl.SimpleLootViewer;
import net.darkhax.simplelootviewer.common.impl.data.info.Name;
import net.darkhax.simplelootviewer.common.impl.data.info.NameType;
import net.darkhax.simplelootviewer.common.impl.data.info.TableInfo;
import net.minecraft.class_1299;
import net.minecraft.class_2248;
import net.minecraft.class_2378;
import net.minecraft.class_2561;
import net.minecraft.class_2960;
import net.minecraft.class_52;
import net.minecraft.class_5321;
import net.minecraft.class_5455;
import net.minecraft.class_7923;
import net.minecraft.class_7924;
import java.text.DecimalFormat;
import java.util.Map;

public record LootTableHelper() {

    private static final DecimalFormat FORMAT = new DecimalFormat("#.##");
    private static int CACHE_VERSION = -1;
    private static ArrayListMultimap<LootType, TableInfo> CACHE;

    /**
     * Gets a cached version of the loot table data. The cache will be rebuilt if it is outdated.
     *
     * @param registries The current reloadable game registries. The loot table registry must be available.
     * @return A multimap of info sorted by type of loot table.
     */
    public static ArrayListMultimap<LootType, TableInfo> getCachedData(class_5455 registries) {
        if (CACHE == null || CACHE_VERSION != Constants.SERVER_REVISION) {
            SimpleLootViewer.LOG.info("Rebuilding table cache.");
            CACHE = collectData(registries);
            CACHE_VERSION = Constants.SERVER_REVISION;
        }
        return CACHE;
    }

    /**
     * Collects all data on loot tables and stores it in a usable format for our plugin.
     *
     * @param registries The current reloadable game registries. The loot table registry must be available.
     * @return A multimap of table info sorted by type of loot table.
     */
    public static ArrayListMultimap<LootType, TableInfo> collectData(class_5455 registries) {
        final long start = System.nanoTime();
        final ArrayListMultimap<LootType, TableInfo> info = ArrayListMultimap.create();
        final class_2378<class_52> lootTableRegistry = registries.method_33310(class_7924.field_50079).orElse(null);
        if (lootTableRegistry != null) {
            // Map loot tables to their corresponding blocks.
            final ArrayListMultimap<class_2960, class_2960> BLOCK_DROPS = ArrayListMultimap.create();
            for (Map.Entry<class_5321<class_2248>, class_2248> block : class_7923.field_41175.method_29722()) {
                BLOCK_DROPS.put(block.getValue().method_26162().method_29177(), block.getKey().method_29177());
            }
            // Map loot tables to their corresponding mobs.
            final ArrayListMultimap<class_2960, class_2960> ENTITY_DROPS = ArrayListMultimap.create();
            for (Map.Entry<class_5321<class_1299<?>>, class_1299<?>> type : class_7923.field_41177.method_29722()) {
                ENTITY_DROPS.put(type.getValue().method_16351().method_29177(), type.getKey().method_29177());
            }
            // Map tables
            for (Map.Entry<class_5321<class_52>, class_52> entry : lootTableRegistry.method_29722()) {
                final class_2960 tableKey = entry.getKey().method_29177();
                if (!SimpleLootViewer.CONFIG.get().hidden.get().contains(tableKey)) {
                    final LootType lootType = LootType.determineType(tableKey);
                    if (lootType.isEnabled()) {
                        final Name name = nameTable(tableKey, BLOCK_DROPS, ENTITY_DROPS);
                        info.put(lootType, new TableInfo(tableKey, name, LootPoolEntryDescriptions.getUniqueItems(registries, entry.getValue())));
                    }
                }
            }
        }
        else {
            SimpleLootViewer.LOG.warn("Unable to load loot tables from registry.");
        }
        final long end = System.nanoTime();
        SimpleLootViewer.LOG.info("Processed {} loot tables in {}ms", info.size(), FORMAT.format((end - start) / 1000000d));
        return info;
    }

    /**
     * Creates a custom name for a loot table.
     *
     * @param tableKey     The ID of the loot table.
     * @param blockTables  A map of blocks that produce each loot table.
     * @param entityTables A map of entities that produce each loot table.
     * @return A name for the loot table.
     */
    public static Name nameTable(class_2960 tableKey, ArrayListMultimap<class_2960, class_2960> blockTables, ArrayListMultimap<class_2960, class_2960> entityTables) {
        if (blockTables.containsKey(tableKey)) {
            return new Name(NameType.BLOCK, blockTables.get(tableKey).toArray(class_2960[]::new));
        }
        else if (entityTables.containsKey(tableKey)) {
            return new Name(NameType.ENTITY, entityTables.get(tableKey).toArray(class_2960[]::new));
        }
        else {
            return new Name(NameType.TABLE, tableKey);
        }
    }

    /**
     * Creates a translatable name based on the ID of the loot table. If the name is not translatable we will fall back
     * to the ID of the table.
     *
     * @param tableKey The ID of the loot table.
     * @return The name of the table.
     */
    public static class_2561 nameTableFromId(class_2960 tableKey) {
        return class_2561.method_48321("table." + tableKey.method_12836() + "." + tableKey.method_12832().replaceAll("/", ".") + ".name", tableKey.toString());
    }
}