package net.darkhax.botanypots.common.api.data.recipes;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Multimap;
import net.darkhax.bookshelf.common.api.function.ReloadableCache;
import net.darkhax.bookshelf.common.api.function.SidedReloadableCache;
import net.darkhax.botanypots.common.api.data.context.BotanyPotContext;
import net.darkhax.botanypots.common.impl.BotanyPotsMod;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_1937;
import net.minecraft.class_2960;
import net.minecraft.class_3956;
import net.minecraft.class_7923;
import net.minecraft.class_8786;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.function.Supplier;

/**
 * An experimental cache that makes looking up botany pot recipes faster.
 *
 * @param <T> The type of recipe held by the cache.
 */
public final class RecipeCache<T extends BotanyPotRecipe> {

    private final Supplier<class_3956<T>> recipeType;
    private final ReloadableCache<Map<class_2960, class_8786<T>>> recipeCache;
    private final Multimap<class_1792, class_8786<T>> lookupCache = ArrayListMultimap.create();
    private final List<class_8786<T>> uncached = new LinkedList<>();

    private RecipeCache(Supplier<class_3956<T>> recipeType) {
        this.recipeType = recipeType;
        this.recipeCache = ReloadableCache.recipes(this.recipeType);
    }

    /**
     * Checks if an item exists in the cache.
     *
     * @param stack The item to test.
     * @return If the item has at least one match in the lookup cache.
     */
    public boolean isCached(class_1799 stack) {
        return !stack.method_7960() && !lookupCache.get(stack.method_7909()).isEmpty();
    }

    /**
     * Gets a map of all cached values.
     *
     * @return The lookup cache.
     */
    public Multimap<class_1792, class_8786<T>> getCachedValues() {
        return this.lookupCache;
    }

    /**
     * Looks up an item in the cache.
     *
     * @param stack   The item to lookup.
     * @param context The context of the lookup.
     * @param level   The current game level.
     * @return The recipe that best matches the provided information.
     */
    @Nullable
    public class_8786<T> lookup(class_1799 stack, BotanyPotContext context, class_1937 level) {
        if (stack.method_7960()) {
            return null;
        }
        final Collection<class_8786<T>> cachedRecipes = lookupCache.get(stack.method_7909());
        for (class_8786<T> cached : lookupCache.get(stack.method_7909())) {
            if (cached.comp_1933().couldMatch(stack, context, level)) {
                return cached;
            }
        }
        for (class_8786<T> holder : this.uncached) {
            if (holder.comp_1933().couldMatch(stack, context, level)) {
                return holder;
            }
        }
        return null;
    }

    private void buildCache(class_1937 level) {
        long startTime = System.nanoTime();
        final Map<class_2960, class_8786<T>> recipes = recipeCache.apply(level);
        if (recipes == null) {
            BotanyPotsMod.LOG.error("Could not build {} cache. Entries do not exist?", this.recipeType.get());
            return;
        }
        uncached.clear();
        uncached.addAll(recipes.values());
        if (!recipes.isEmpty()) {
            for (class_1792 item : class_7923.field_41178) {
                final class_1799 defaultStack = item.method_7854();
                for (class_8786<T> recipe : recipes.values()) {
                    if (recipe.comp_1933() instanceof CacheableRecipe cacheable && cacheable.canBeCached() && cacheable.isCacheKey(defaultStack)) {
                        this.lookupCache.put(item, recipe);
                        uncached.remove(recipe);
                    }
                }
            }
        }
        long endTime = System.nanoTime();
        BotanyPotsMod.LOG.info("Built {} {} cache in {}ms. {} / {} entries cached.", level.field_9236 ? "Client" : "Server", this.recipeType.get(), (endTime - startTime) / 1_000_000f, recipes.size() - uncached.size(), recipes.size());
    }

    /**
     * Creates a new cache for a given recipe type.
     *
     * @param type The type of recipe to build a cache for.
     * @param <T>  The type of the recipe.
     * @return The created recipe cache.
     */
    public static <T extends BotanyPotRecipe> SidedReloadableCache<RecipeCache<T>> of(Supplier<class_3956<T>> type) {
        return SidedReloadableCache.of(level -> {
            final RecipeCache<T> cache = new RecipeCache<>(type);
            cache.buildCache(level);
            return cache;
        });
    }
}