package net.darkhax.botanypots.common.impl.command;

import com.google.gson.JsonObject;
import com.mojang.brigadier.arguments.BoolArgumentType;
import com.mojang.brigadier.builder.LiteralArgumentBuilder;
import com.mojang.brigadier.context.CommandContext;
import net.darkhax.bookshelf.common.api.function.CachedSupplier;
import net.darkhax.bookshelf.common.api.util.CommandHelper;
import net.darkhax.botanypots.common.api.BotanyPotsPlugin;
import net.darkhax.botanypots.common.api.command.generator.DataHelper;
import net.darkhax.botanypots.common.api.command.generator.crop.CropGenerator;
import net.darkhax.botanypots.common.api.command.generator.soil.SoilGenerator;
import net.darkhax.botanypots.common.api.data.recipes.crop.Crop;
import net.darkhax.botanypots.common.api.data.recipes.fertilizer.Fertilizer;
import net.darkhax.botanypots.common.api.data.recipes.soil.Soil;
import net.darkhax.botanypots.common.impl.BotanyPotsMod;
import net.darkhax.botanypots.common.impl.command.generator.MissingCropGenerator;
import net.darkhax.botanypots.common.impl.command.generator.MissingSoilGenerator;
import net.minecraft.class_124;
import net.minecraft.class_1747;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_1937;
import net.minecraft.class_2168;
import net.minecraft.class_2170;
import net.minecraft.class_2248;
import net.minecraft.class_2397;
import net.minecraft.class_2473;
import net.minecraft.class_2558;
import net.minecraft.class_2561;
import net.minecraft.class_2960;
import net.minecraft.class_3218;
import net.minecraft.class_3481;
import net.minecraft.class_6862;
import net.minecraft.class_7923;
import net.minecraft.class_7924;
import org.apache.commons.io.FileUtils;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.Comparator;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Objects;
import java.util.StringJoiner;
import java.util.function.Consumer;

public class MissingCommand {

    private static final CachedSupplier<Map<class_2960, SoilGenerator>> SOIL_GENERATORS = CachedSupplier.cache(() -> {
        final Map<class_2960, SoilGenerator> generators = new LinkedHashMap<>();
        BotanyPotsPlugin.PLUGINS.get().forEach(plugin -> plugin.registerSoilGenerators(generators::put));
        generators.put(BotanyPotsMod.id("missing_fallback"), new MissingSoilGenerator());
        return generators;
    });

    private static final CachedSupplier<Map<class_2960, CropGenerator>> CROP_GENERATORS = CachedSupplier.cache(() -> {
        final Map<class_2960, CropGenerator> generators = new LinkedHashMap<>();
        BotanyPotsPlugin.PLUGINS.get().forEach(plugin -> plugin.registerCropGenerators(generators::put));
        generators.put(BotanyPotsMod.id("missing_fallback"), new MissingCropGenerator());
        return generators;
    });

    private static final class_6862<class_1792> SAPLING_TAG = class_6862.method_40092(class_7924.field_41197, class_2960.method_60656("saplings"));
    private static final Comparator<class_2960> ID_COMPARE = Comparator.comparing(class_2960::toString);

    public static void build(LiteralArgumentBuilder<class_2168> parent) {
        final LiteralArgumentBuilder<class_2168> cmd = class_2170.method_9247("missing");
        final LiteralArgumentBuilder<class_2168> seeds = class_2170.method_9247("seeds");
        seeds.executes(MissingCommand::dumpMissingCrops);
        seeds.then(class_2170.method_9244("include_saplings", BoolArgumentType.bool()).executes(MissingCommand::dumpMissingCrops).then(class_2170.method_9244("generate", BoolArgumentType.bool()).executes(MissingCommand::dumpMissingCrops)));
        cmd.then(seeds);

        final LiteralArgumentBuilder<class_2168> soils = class_2170.method_9247("soils");
        soils.executes(MissingCommand::dumpMissingSoils);
        soils.then(class_2170.method_9244("generate", BoolArgumentType.bool()).executes(MissingCommand::dumpMissingSoils));
        cmd.then(soils);

        parent.then(cmd);
    }

    private static Map<class_1799, SoilGenerator> collectMissingSoilItems(class_3218 level) {
        final Map<class_1799, SoilGenerator> missing = new HashMap<>();
        for (class_1792 item : class_7923.field_41178) {
            final class_1799 stack = item.method_7854();
            if (!isSoil(stack, level) && !isCrop(stack, level)) {
                for (SoilGenerator generator : SOIL_GENERATORS.get().values()) {
                    if (generator.canGenerateSoil(level, stack)) {
                        missing.put(stack, generator);
                        break;
                    }
                }
            }
        }
        return missing;
    }

    private static int dumpMissingSoils(CommandContext<class_2168> ctx) {
        final class_3218 level = ctx.getSource().method_9225();
        final boolean generate = CommandHelper.getBooleanArg("generate", ctx, () -> false);
        final Map<class_1799, SoilGenerator> missing = collectMissingSoilItems(level);
        if (missing.isEmpty()) {
            ctx.getSource().method_9226(() -> BotanyPotsCommands.modMessage(class_2561.method_43471("commands.botanypots.dump.no_results")), false);
            return 0;
        }
        else {
            if (generate) {
                final File outDir = setupDir("botanypots/generated/soils");
                for (Map.Entry<class_1799, SoilGenerator> entry : missing.entrySet()) {
                    final class_2960 itemId = class_7923.field_41178.method_10221(entry.getKey().method_7909());
                    final File soilFile = new File(outDir, itemId.method_12836() + "/soil/" + itemId.method_12832() + ".json");
                    final JsonObject obj = entry.getValue().generateData(level, entry.getKey());
                    writeFile(soilFile, DataHelper.GSON.toJson(obj));
                }
            }
            final StringJoiner entries = new StringJoiner(System.lineSeparator());
            entries.add("Potential missing soil IDs");
            missing.keySet().stream().map(stack -> class_7923.field_41178.method_10221(stack.method_7909())).sorted(Comparator.comparing(class_2960::toString)).forEach(entry -> entries.add(entry.toString()));
            ctx.getSource().method_9226(() -> BotanyPotsCommands.modMessage(class_2561.method_43469("commands.botanypots.dump.missing_soils", class_2561.method_43470(Integer.toString(missing.size())).method_27694(style -> style.method_10977(class_124.field_1061))).method_27694(style -> style.method_10958(new class_2558(class_2558.class_2559.field_21462, entries.toString())))), false);
            return missing.size();
        }
    }

    private static Map<class_1799, CropGenerator> collectMissingCropItems(boolean collectSaplings, class_3218 level) {
        final Map<class_1799, CropGenerator> missing = new HashMap<>();
        for (class_1792 item : class_7923.field_41178) {
            final class_1799 stack = item.method_7854();
            if (!isSoil(stack, level) && !isCrop(stack, level) && !isLeaves(item) && (collectSaplings || !isSapling(item))) {
                for (CropGenerator generator : CROP_GENERATORS.get().values()) {
                    if (generator.canGenerateCrop(level, stack)) {
                        missing.put(stack, generator);
                        break;
                    }
                }
            }
        }
        return missing;
    }

    private static int dumpMissingCrops(CommandContext<class_2168> ctx) {
        final class_3218 level = ctx.getSource().method_9225();
        final boolean includeSaplings = CommandHelper.getBooleanArg("include_saplings", ctx, () -> false);
        final boolean generate = CommandHelper.getBooleanArg("generate", ctx, () -> false);
        final Map<class_1799, CropGenerator> missingCrops = collectMissingCropItems(includeSaplings, level);
        if (missingCrops.isEmpty()) {
            ctx.getSource().method_9226(() -> BotanyPotsCommands.modMessage(class_2561.method_43471("commands.botanypots.dump.no_results")), false);
            return 0;
        }
        if (generate) {
            final File outdir = setupDir("botanypots/generated/crops");
            for (Map.Entry<class_1799, CropGenerator> entry : missingCrops.entrySet()) {
                final class_2960 itemId = class_7923.field_41178.method_10221(entry.getKey().method_7909());
                final File cropFile = new File(outdir, itemId.method_12836() + "/crop/" + itemId.method_12832() + ".json");
                writeFile(cropFile, DataHelper.GSON.toJson(entry.getValue().generateData(level, entry.getKey())));
            }
            ctx.getSource().method_9226(() -> class_2561.method_43471("commands.botanypots.dump.generated").method_27694(style -> style.method_10958(new class_2558(class_2558.class_2559.field_11746, outdir.getAbsolutePath()))), false);
        }
        else {
            final StringJoiner entries = new StringJoiner(System.lineSeparator());
            entries.add("Potential missing crop IDs");
            missingCrops.forEach((k, v) -> entries.add(class_7923.field_41178.method_10221(k.method_7909()).toString()));
            ctx.getSource().method_9226(() -> BotanyPotsCommands.modMessage(class_2561.method_43469("commands.botanypots.dump.missing_crops", class_2561.method_43470(Integer.toString(missingCrops.size())).method_27694(style -> style.method_10977(class_124.field_1061))).method_27694(style -> style.method_10958(new class_2558(class_2558.class_2559.field_21462, entries.toString())))), false);
        }
        return missingCrops.size();
    }

    private static void processTag(class_6862<class_1792> key, Consumer<class_1792> consumer) {
        class_7923.field_41178.method_40266(key).ifPresent(named -> named.forEach(entry -> consumer.accept(entry.comp_349())));
    }

    private static boolean isCrop(class_1799 stack, class_1937 level) {
        return Objects.requireNonNull(Crop.CACHE.apply(level)).isCached(stack);
    }

    private static boolean isSoil(class_1799 stack, class_1937 level) {
        return Objects.requireNonNull(Soil.CACHE.apply(level)).isCached(stack);
    }

    private static boolean isFertilizer(class_1799 stack, class_1937 level) {
        return Objects.requireNonNull(Fertilizer.CACHE.apply(level)).isCached(stack);
    }

    private static boolean isSapling(class_1792 item) {
        return item.method_7854().method_31573(SAPLING_TAG) || item instanceof class_1747 blockItem && (blockItem.method_7711() instanceof class_2473);
    }

    private static boolean isLeaves(class_1792 item) {
        if (item instanceof class_1747 blockItem) {
            final class_2248 block = blockItem.method_7711();
            return block instanceof class_2397 || block.method_9564().method_26164(class_3481.field_15503);
        }
        return false;
    }

    private static void writeFile(File file, String text) {
        if (!file.getParentFile().exists()) {
            file.getParentFile().mkdirs();
        }
        try (FileWriter writer = new FileWriter(file)) {
            writer.append(text);
        }
        catch (IOException e) {
            // No
        }
    }

    private static File setupDir(String dir) {
        final File file = new File(dir);
        if (file.exists()) {
            try {
                FileUtils.deleteDirectory(file);
            }
            catch (Exception e) {
                BotanyPotsMod.LOG.error("Failed to setup dir {}.", dir, e);
            }
        }
        file.mkdirs();
        return file;
    }
}
