package net.darkhax.bookshelf.common.api.loot;

import com.mojang.datafixers.util.Either;
import net.darkhax.bookshelf.common.api.function.CachedSupplier;
import net.darkhax.bookshelf.common.api.registry.register.RegisterLootDescription;
import net.darkhax.bookshelf.common.api.service.Services;
import net.darkhax.bookshelf.common.impl.data.loot.entries.LootItemStack;
import net.darkhax.bookshelf.common.mixin.access.loot.AccessorCompositeEntryBase;
import net.darkhax.bookshelf.common.mixin.access.loot.AccessorLootItem;
import net.darkhax.bookshelf.common.mixin.access.loot.AccessorLootPool;
import net.darkhax.bookshelf.common.mixin.access.loot.AccessorLootTable;
import net.darkhax.bookshelf.common.mixin.access.loot.AccessorNestedLootTable;
import net.darkhax.bookshelf.common.mixin.access.loot.AccessorTagEntry;
import net.minecraft.class_124;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_1802;
import net.minecraft.class_2561;
import net.minecraft.class_52;
import net.minecraft.class_5321;
import net.minecraft.class_5338;
import net.minecraft.class_5455;
import net.minecraft.class_55;
import net.minecraft.class_67;
import net.minecraft.class_6862;
import net.minecraft.class_6880;
import net.minecraft.class_73;
import net.minecraft.class_79;
import net.minecraft.class_7923;
import net.minecraft.class_7924;
import net.minecraft.class_9290;
import net.minecraft.class_9334;
import org.jetbrains.annotations.NotNull;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Consumer;
import java.util.function.Function;

/**
 * Provides a system for describing what items can be dropped by a loot table.
 */
public class LootPoolEntryDescriptions {

    private static final CachedSupplier<class_1799> UNKNOWN_ITEM_DISPLAY = CachedSupplier.cache(() -> {
        final class_1799 stack = new class_1799(class_1802.field_8615);
        stack.method_57379(class_9334.field_50239, class_2561.method_43471("tooltips.bookshelf.loot.unknown"));
        stack.method_57379(class_9334.field_49632, new class_9290(List.of(), List.of(class_2561.method_43471("tooltips.bookshelf.loot.unknown.desc").method_27692(class_124.field_1080))));
        return stack;
    });

    private static final CachedSupplier<class_1799> EMPTY_ITEM_DISPLAY = CachedSupplier.cache(() -> {
        final class_1799 stack = new class_1799(class_1802.field_8077);
        stack.method_57379(class_9334.field_50239, class_2561.method_43471("tooltips.bookshelf.loot.empty"));
        stack.method_57379(class_9334.field_49632, new class_9290(List.of(), List.of(class_2561.method_43471("tooltips.bookshelf.loot.empty.desc").method_27692(class_124.field_1080))));
        return stack;
    });

    private static final CachedSupplier<class_1799> DYNAMIC_DISPLAY = CachedSupplier.cache(() -> {
        final class_1799 stack = new class_1799(class_1802.field_16538);
        stack.method_57379(class_9334.field_50239, class_2561.method_43471("tooltips.bookshelf.loot.dynamic"));
        stack.method_57379(class_9334.field_49632, new class_9290(List.of(), List.of(class_2561.method_43471("tooltips.bookshelf.loot.dynamic.desc").method_27692(class_124.field_1080))));
        return stack;
    });

    private static final Map<class_5338, LootPoolEntryDescriber> DESCRIBERS = new HashMap<>();
    private static boolean hasInitialized = false;

    public static final LootPoolEntryDescriber EMPTY = (server, entry, collector) -> {
        if (entry instanceof class_73) {
            collector.accept(EMPTY_ITEM_DISPLAY.get());
        }
    };

    public static final LootPoolEntryDescriber ITEM = (server, entry, collector) -> {
        if (entry instanceof AccessorLootItem accessor) {
            collector.accept(new class_1799(accessor.bookshelf$item()));
        }
    };

    public static final LootPoolEntryDescriber LOOT_TABLE = (server, entry, collector) -> {
        if (entry instanceof AccessorNestedLootTable accessor) {
            getPotentialItems(server, accessor.bookshelf$contents(), collector);
        }
    };
    public static final LootPoolEntryDescriber DYNAMIC = (server, entry, collector) -> {
        if (entry instanceof class_67) {
            collector.accept(DYNAMIC_DISPLAY.get());
        }
    };
    public static final LootPoolEntryDescriber TAG = (server, entry, collector) -> {
        if (entry instanceof AccessorTagEntry tagEntry) {
            getTagItems(tagEntry.bookshelf$tag(), collector);
        }
    };
    public static final LootPoolEntryDescriber COMPOSITE = (server, entry, collector) -> {
        if (entry instanceof AccessorCompositeEntryBase accessor) {
            getPotentialItems(server, accessor.bookshelf$children(), collector);
        }
    };
    public static final LootPoolEntryDescriber ITEM_STACK = (server, entry, collector) -> {
        if (entry instanceof LootItemStack loot) {
            collector.accept(loot.getBaseStack());
        }
    };

    private static void bootstrap() {
        if (!hasInitialized) {
            final RegisterLootDescription register = new RegisterLootDescription(DESCRIBERS::put);
            Services.CONTENT_PROVIDERS.get().forEach(provider -> provider.registerLootDescriptions(register));
            hasInitialized = true;
        }
    }

    /**
     * Generates a list of unique items that can generate from a loot table.
     *
     * @param registries The current registries.
     * @param table      The loot table to examine.
     * @return A list containing the entries.
     */
    public static List<class_1799> getUniqueItems(@NotNull class_5455 registries, class_52 table) {
        final List<class_1799> items = new ArrayList<>();
        getPotentialItems(registries, table, stack -> addStacking(items, stack));
        return items;
    }

    /**
     * Gets potential drops for a loot table.
     *
     * @param registries The current registries.
     * @param table      The loot table to examine.
     * @param consumer   Collects entries in your desired format.
     */
    public static void getPotentialItems(@NotNull class_5455 registries, Either<class_5321<class_52>, class_52> table, Consumer<class_1799> consumer) {
        final class_52 resolved = table.map(rl -> registries.method_30530(class_7924.field_50079).method_29107(rl), Function.identity());
        if (resolved != null) {
            getPotentialItems(registries, resolved, consumer);
        }
    }

    /**
     * Gets potential drops for a loot table.
     *
     * @param registries The current registries.
     * @param table      The loot table to examine.
     * @param consumer   Collects entries in your desired format.
     */
    public static void getPotentialItems(@NotNull class_5455 registries, class_52 table, Consumer<class_1799> consumer) {
        if (table instanceof AccessorLootTable tableAccess) {
            for (class_55 pool : tableAccess.bookshelf$pools()) {
                if (pool instanceof AccessorLootPool poolAccess) {
                    getPotentialItems(registries, poolAccess.bookshelf$entries(), consumer);
                }
            }
        }
    }

    /**
     * Gets potential drops for a list of loot pool entries.
     *
     * @param registries The current registries.
     * @param entries    A list of entries to examine.
     * @param collector  Collects entries in your desired format.
     */
    public static void getPotentialItems(@NotNull class_5455 registries, List<class_79> entries, Consumer<class_1799> collector) {
        for (class_79 entry : entries) {
            getPotentialItems(registries, entry, collector);
        }
    }

    /**
     * Gets potential drops from a loot pool entry.
     *
     * @param registries The current registries.
     * @param entry      The pool entry to examine.
     * @param collector  Collects entries in your desired format.
     */
    public static void getPotentialItems(@NotNull class_5455 registries, class_79 entry, Consumer<class_1799> collector) {
        bootstrap();
        final LootPoolEntryDescriber describer = DESCRIBERS.get(entry.method_29318());
        if (describer != null) {
            describer.getPotentialDrops(registries, entry, collector);
        }
        else {
            collector.accept(UNKNOWN_ITEM_DISPLAY.get());
        }
    }

    /**
     * Adds an ItemStack to a list, only if the item does not stack with any of the items already in the list.
     *
     * @param items The list to add to.
     * @param toAdd The entry to add.
     */
    private static void addStacking(List<class_1799> items, class_1799 toAdd) {
        for (class_1799 existing : items) {
            if (Objects.equals(existing, toAdd) || class_1799.method_31577(existing, toAdd)) {
                return;
            }
        }
        items.add(toAdd);
    }

    private static void getTagItems(class_6862<class_1792> tag, Consumer<class_1799> collector) {
        for (class_6880<class_1792> item : class_7923.field_41178.method_40286(tag)) {
            collector.accept(new class_1799(item));
        }
    }
}