package net.darkhax.bookshelf.common.impl.command;

import com.mojang.brigadier.builder.LiteralArgumentBuilder;
import com.mojang.brigadier.context.CommandContext;
import com.mojang.brigadier.exceptions.CommandSyntaxException;
import net.darkhax.bookshelf.common.api.commands.IEnumCommand;
import net.darkhax.bookshelf.common.api.commands.PermissionLevel;
import net.darkhax.bookshelf.common.api.loot.LootPoolEntryDescriptions;
import net.darkhax.bookshelf.common.api.util.CommandHelper;
import net.darkhax.bookshelf.common.api.util.TextHelper;
import net.darkhax.bookshelf.common.impl.Constants;
import net.darkhax.bookshelf.common.impl.data.loot.modifiers.ILootPoolHooks;
import net.darkhax.bookshelf.common.mixin.access.loot.AccessorLootTable;
import net.minecraft.class_1074;
import net.minecraft.class_2168;
import net.minecraft.class_2248;
import net.minecraft.class_2561;
import net.minecraft.class_2960;
import net.minecraft.class_52;
import net.minecraft.class_5321;
import net.minecraft.class_5455;
import net.minecraft.class_6880;
import net.minecraft.class_7157;
import net.minecraft.class_7871;
import net.minecraft.class_7923;
import net.minecraft.class_7924;
import net.minecraft.server.MinecraftServer;
import java.util.Collection;
import java.util.Locale;
import java.util.Optional;
import java.util.StringJoiner;
import java.util.concurrent.atomic.AtomicBoolean;

public enum DebugCommands implements IEnumCommand {

    MISSING_TAG_NAMES(DebugCommands::findMissingTagNames),
    MISSING_BLOCK_DROPS(DebugCommands::findMissingBlockDrops),
    LOOT_POOL_HASH(DebugCommands::findLootTableHashes),
    SIMPLE_TABLES(DebugCommands::printTables);

    private static void printTables(MinecraftServer server, StringJoiner out) {
        final Collection<class_2960> tableKeys = server.method_58576().method_58290(class_7924.field_50079);
        final class_5455 registries = server.method_58576().method_58289();
        for (class_2960 tableKey : tableKeys) {
            final class_52 table = server.method_58576().method_58295(class_5321.method_29179(class_7924.field_50079, tableKey));
            out.add(tableKey + " = " + LootPoolEntryDescriptions.getUniqueItems(registries, table));
        }
    }

    private static void findMissingTagNames(MinecraftServer server, StringJoiner out) {
        server.method_30611().method_40311().forEach(entry -> {
            AtomicBoolean hasLogged = new AtomicBoolean(false);
            entry.comp_351().method_40273().forEach(tag -> {
                if (!tag.comp_327().method_12836().equals(class_2960.field_33381) && !class_1074.method_4663(TextHelper.getTagName(tag))) {
                    if (!hasLogged.get()) {
                        hasLogged.set(true);
                        out.add("## " + entry.comp_350().method_29177());
                    }
                    out.add("\"" + TextHelper.getTagName(tag) + "\": \"\",");
                }
            });
            if (hasLogged.get()) {
                out.add("");
            }
        });
    }

    private static void findMissingBlockDrops(MinecraftServer server, StringJoiner out) {
        final class_7871<class_52> lootTables = server.method_58576().method_58294().method_46750(class_7924.field_50079).orElseThrow();
        for (class_2248 block : class_7923.field_41175) {
            final Optional<class_6880.class_6883<class_52>> result = lootTables.method_46746(block.method_26162());
            if (result.isEmpty()) {
                final class_2960 id = block.method_26162().method_29177();
                out.add(class_7923.field_41175.method_10221(block) + " - " + id + " - data/" + id.method_12836() + "/loot_table/" + id.method_12832());
            }
        }
    }

    private static void findLootTableHashes(MinecraftServer server, StringJoiner out) {
        final class_7871<class_52> lootTables = server.method_58576().method_58294().method_46750(class_7924.field_50079).orElseThrow();
        for (class_2960 table : server.method_58576().method_58290(class_7924.field_50079)) {
            if (table.method_12832().startsWith("chests") || table.method_12832().startsWith("dispensers") || table.method_12832().startsWith("gameplay") || table.method_12832().startsWith("pots") || table.method_12832().startsWith("spawners")) {
                out.add("## " + table.toString());
                lootTables.method_46746(class_5321.method_29179(class_7924.field_50079, table)).ifPresent(val -> {
                    if (val.comp_349() instanceof AccessorLootTable accessor) {
                        for (int index = 0; index < accessor.bookshelf$pools().size(); index++) {
                            if (accessor.bookshelf$pools().get(index) instanceof ILootPoolHooks fingerprintable) {
                                out.add("- " + index + " | " + fingerprintable.bookshelf$getHash());
                            }
                        }
                    }
                });
            }
        }
    }

    public static LiteralArgumentBuilder<class_2168> build(class_7157 context) {
        return CommandHelper.buildFromEnum("debug", DebugCommands.class);
    }

    private final DebugTask debugTask;

    DebugCommands(DebugTask task) {
        this.debugTask = task;
    }

    @Override
    public String getCommandName() {
        return this.name().toLowerCase(Locale.ROOT);
    }

    @Override
    public int run(CommandContext<class_2168> context) throws CommandSyntaxException {
        final StringJoiner joiner = new StringJoiner(System.lineSeparator());
        this.debugTask.getDebugOutput(context.getSource().method_9211(), joiner);
        Constants.LOG.warn(joiner.toString());
        final String debugInfo = joiner.toString();
        if (debugInfo.isBlank()) {
            context.getSource().method_9213(class_2561.method_43471("commands.bookshelf.debug.no_info"));
            return 0;
        }
        else if (debugInfo.length() > 10000) {
            context.getSource().method_9213(class_2561.method_43471("commands.bookshelf.debug.too_long"));
            return 0;
        }
        else {
            context.getSource().method_9226(() -> TextHelper.setCopyText(class_2561.method_43471("commands.bookshelf.debug.yes_info"), debugInfo), false);
            return 1;
        }
    }

    @Override
    public PermissionLevel requiredPermissionLevel() {
        return PermissionLevel.OWNER;
    }

    @FunctionalInterface
    public interface DebugTask {
        void getDebugOutput(MinecraftServer server, StringJoiner output);
    }
}
