package net.darkhax.bookshelf.common.api.util;

import com.mojang.brigadier.arguments.BoolArgumentType;
import com.mojang.brigadier.builder.ArgumentBuilder;
import com.mojang.brigadier.builder.LiteralArgumentBuilder;
import com.mojang.brigadier.context.CommandContext;
import com.mojang.brigadier.exceptions.CommandSyntaxException;
import net.darkhax.bookshelf.common.api.commands.IEnumCommand;
import net.darkhax.bookshelf.common.api.commands.PermissionLevel;
import net.minecraft.class_1297;
import net.minecraft.class_2168;
import net.minecraft.class_2186;
import net.minecraft.class_2300;
import java.util.function.Supplier;

public class CommandHelper {

    /**
     * Creates a command with branching paths that represent the values of an enum.
     *
     * @param parent    The name of the root parent command node.
     * @param enumClass The enum class to use.
     * @param <T>       The type of the enum.
     * @return The newly created command node.
     */
    public static <T extends Enum<T> & IEnumCommand> LiteralArgumentBuilder<class_2168> buildFromEnum(String parent, Class<T> enumClass) {
        final LiteralArgumentBuilder<class_2168> parentNode = LiteralArgumentBuilder.literal(parent);
        parentNode.requires(getLowestLevel(enumClass));
        buildFromEnum(parentNode, enumClass);
        return parentNode;
    }

    /**
     * Creates branching command paths that represent the values of an enum.
     *
     * @param parent    The parent node to branch off from.
     * @param enumClass The enum class to use.
     * @param <T>       The type of the enum.
     */
    public static <T extends Enum<T> & IEnumCommand> void buildFromEnum(ArgumentBuilder<class_2168, ?> parent, Class<T> enumClass) {
        if (!enumClass.isEnum()) {
            throw new IllegalStateException("Class '" + enumClass.getCanonicalName() + "' is not an enum!");
        }
        for (T enumEntry : enumClass.getEnumConstants()) {
            final LiteralArgumentBuilder<class_2168> literal = LiteralArgumentBuilder.literal(enumEntry.getCommandName());
            literal.requires(enumEntry.requiredPermissionLevel()).executes(enumEntry);
            parent.then(literal);
        }
    }

    /**
     * Gets the lowest required permission level for an enum command.
     *
     * @param enumClass The enum class to use.
     * @param <T>       The type of the enum.
     * @return The lowest required permission level for an enum command.
     */
    public static <T extends Enum<T> & IEnumCommand> PermissionLevel getLowestLevel(Class<T> enumClass) {
        if (!enumClass.isEnum()) {
            throw new IllegalStateException("Class '" + enumClass.getCanonicalName() + "' is not an enum!");
        }
        PermissionLevel level = PermissionLevel.OWNER;
        for (T enumEntry : enumClass.getEnumConstants()) {
            if (enumEntry.requiredPermissionLevel().get() < level.get()) {
                level = enumEntry.requiredPermissionLevel();
            }
        }
        return level;
    }

    /**
     * @deprecated This only works on Fabric.
     */
    @Deprecated
    public static <T> boolean hasArgument(String argument, CommandContext<T> context) {
        return hasArgument(argument, context, Object.class);
    }

    public static <T, C> boolean hasArgument(String argument, CommandContext<C> context, Class<T> argType) {
        try {
            return context.getArgument(argument, argType) != null;
        }
        catch (Exception e) {
            return false;
        }
    }

    public static <T, C> T getArgument(String argument, CommandContext<C> context, Class<T> argType, Supplier<T> fallback) {
        return hasArgument(argument, context, argType) ? context.getArgument(argument, argType) : fallback.get();
    }

    public static class_1297 getEntity(String argName, CommandContext<class_2168> ctx, Supplier<class_1297> fallback) throws CommandSyntaxException {
        return CommandHelper.hasArgument(argName, ctx, class_2300.class) ? class_2186.method_9313(ctx, argName) : fallback.get();
    }

    public static class_1297 getEntityOrSender(String argName, CommandContext<class_2168> ctx) throws CommandSyntaxException {
        return CommandHelper.hasArgument(argName, ctx, class_2300.class) ? class_2186.method_9313(ctx, argName) : ctx.getSource().method_9228();
    }

    public static boolean getBooleanArg(String argName, CommandContext<class_2168> ctx, Supplier<Boolean> fallback) {
        return CommandHelper.hasArgument(argName, ctx, Boolean.class) ? BoolArgumentType.getBool(ctx, argName) : fallback.get();
    }

    public static boolean getBooleanArg(String argName, CommandContext<class_2168> ctx) {
        return getBooleanArg(argName, ctx, () -> false);
    }
}