package net.darkhax.bookshelf.common.api.loot;

import com.mojang.datafixers.util.Either;
import net.darkhax.bookshelf.common.api.function.CachedSupplier;
import net.darkhax.bookshelf.common.api.registry.register.RegisterLootDescription;
import net.darkhax.bookshelf.common.api.service.Services;
import net.darkhax.bookshelf.common.impl.data.loot.entries.LootItemStack;
import net.darkhax.bookshelf.common.mixin.access.loot.AccessorCompositeEntryBase;
import net.darkhax.bookshelf.common.mixin.access.loot.AccessorLootItem;
import net.darkhax.bookshelf.common.mixin.access.loot.AccessorLootPool;
import net.darkhax.bookshelf.common.mixin.access.loot.AccessorLootTable;
import net.darkhax.bookshelf.common.mixin.access.loot.AccessorNestedLootTable;
import net.darkhax.bookshelf.common.mixin.access.loot.AccessorTagEntry;
import net.minecraft.class_124;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_1802;
import net.minecraft.class_2371;
import net.minecraft.class_2561;
import net.minecraft.class_52;
import net.minecraft.class_5321;
import net.minecraft.class_5338;
import net.minecraft.class_55;
import net.minecraft.class_67;
import net.minecraft.class_6862;
import net.minecraft.class_6880;
import net.minecraft.class_73;
import net.minecraft.class_79;
import net.minecraft.class_7923;
import net.minecraft.class_9290;
import net.minecraft.class_9334;
import net.minecraft.server.MinecraftServer;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Function;

/**
 * Provides a system for describing what items can be dropped by a loot table.
 */
public class LootPoolEntryDescriptions {

    private static final CachedSupplier<List<class_1799>> UNKNOWN_ITEM_DISPLAY = CachedSupplier.cache(() -> {
        final class_1799 stack = new class_1799(class_1802.field_8615);
        stack.method_57379(class_9334.field_50239, class_2561.method_43471("tooltips.bookshelf.loot.unknown"));
        stack.method_57379(class_9334.field_49632, new class_9290(List.of(), List.of(class_2561.method_43471("tooltips.bookshelf.loot.unknown.desc").method_27692(class_124.field_1080))));
        return List.of(stack);
    });

    private static final CachedSupplier<List<class_1799>> EMPTY_ITEM_DISPLAY = CachedSupplier.cache(() -> {
        final class_1799 stack = new class_1799(class_1802.field_8077);
        stack.method_57379(class_9334.field_50239, class_2561.method_43471("tooltips.bookshelf.loot.empty"));
        stack.method_57379(class_9334.field_49632, new class_9290(List.of(), List.of(class_2561.method_43471("tooltips.bookshelf.loot.empty.desc").method_27692(class_124.field_1080))));
        return List.of(stack);
    });

    private static final CachedSupplier<List<class_1799>> DYNAMIC_DISPLAY = CachedSupplier.cache(() -> {
        final class_1799 stack = new class_1799(class_1802.field_16538);
        stack.method_57379(class_9334.field_50239, class_2561.method_43471("tooltips.bookshelf.loot.dynamic"));
        stack.method_57379(class_9334.field_49632, new class_9290(List.of(), List.of(class_2561.method_43471("tooltips.bookshelf.loot.dynamic.desc").method_27692(class_124.field_1080))));
        return List.of(stack);
    });

    private static final Map<class_5338, LootPoolEntryDescriber> DESCRIBERS = new HashMap<>();
    private static boolean hasInitialized = false;

    public static final LootPoolEntryDescriber EMPTY = (server, entry) -> Optional.ofNullable(entry instanceof class_73 ? EMPTY_ITEM_DISPLAY.get() : null);
    public static final LootPoolEntryDescriber ITEM = (server, entry) -> Optional.ofNullable(entry instanceof AccessorLootItem accessor ? List.of(new class_1799(accessor.bookshelf$item())) : null);
    public static final LootPoolEntryDescriber LOOT_TABLE = (server, entry) -> Optional.ofNullable(entry instanceof AccessorNestedLootTable accessor ? getPotentialItems(server, accessor.bookshelf$contents()) : null);
    public static final LootPoolEntryDescriber DYNAMIC = (server, entry) -> Optional.ofNullable(entry instanceof class_67 ? DYNAMIC_DISPLAY.get() : null);
    public static final LootPoolEntryDescriber TAG = (server, entry) -> Optional.ofNullable(entry instanceof AccessorTagEntry tagEntry ? getTagItems(tagEntry.bookshelf$tag()) : null);
    public static final LootPoolEntryDescriber COMPOSITE = (server, entry) -> Optional.ofNullable(entry instanceof AccessorCompositeEntryBase access ? getPotentialItems(server, access.bookshelf$children()) : null);
    public static final LootPoolEntryDescriber ITEM_STACK = (server, entry) -> Optional.ofNullable(entry instanceof LootItemStack loot ? List.of(loot.getBaseStack()) : null);

    private static void bootstrap() {
        if (!hasInitialized) {
            final RegisterLootDescription register = new RegisterLootDescription(DESCRIBERS::put);
            Services.CONTENT_PROVIDERS.get().forEach(provider -> provider.registerLootDescriptions(register));
            hasInitialized = true;
        }
    }

    /**
     * Gets a list of items that can be produced by a loot table.
     *
     * @param server The current Minecraft server instance.
     * @param table  The loot table to analyze.
     * @return A list of items that can be produced by the entry.
     */
    public static List<class_1799> getPotentialItems(MinecraftServer server, Either<class_5321<class_52>, class_52> table) {
        final class_52 resolved = table.map(key -> server.method_58576().method_58295(key), Function.identity());
        return resolved == null ? List.of() : getPotentialItems(server, resolved);
    }

    /**
     * Gets a list of items that can be produced by a loot table.
     *
     * @param server The current Minecraft server instance.
     * @param table  The loot table to analyze.
     * @return A list of items that can be produced by the entry.
     */
    public static List<class_1799> getPotentialItems(MinecraftServer server, class_52 table) {
        final List<class_1799> items = class_2371.method_10211();
        if (table instanceof AccessorLootTable tableAccess) {
            for (class_55 pool : tableAccess.bookshelf$pools()) {
                if (pool instanceof AccessorLootPool poolAccess) {
                    getPotentialItems(server, poolAccess.bookshelf$entries()).forEach(stack -> addStacking(items, stack));
                }
            }
        }
        return items;
    }

    /**
     * Gets a list of items that can be produced by a list of loot pool entries.
     *
     * @param server  The current Minecraft server instance.
     * @param entries A list of loot pool entries to analyze.
     * @return A list of items that can be produced by the entry.
     */
    public static List<class_1799> getPotentialItems(MinecraftServer server, List<class_79> entries) {
        final List<class_1799> items = class_2371.method_10211();
        for (class_79 entry : entries) {
            items.addAll(getPotentialItems(server, entry));
        }
        return items;
    }

    /**
     * Gets a list of items that can be produced by a loot pool entry.
     *
     * @param server The current Minecraft server instance.
     * @param entry  The loot pool entry to analyze.
     * @return A list of items that can be produced by the entry.
     */
    public static List<class_1799> getPotentialItems(MinecraftServer server, class_79 entry) {
        bootstrap();
        final LootPoolEntryDescriber describer = DESCRIBERS.get(entry.method_29318());
        return describer != null ? describer.getPotentialDrops(server, entry).orElse(UNKNOWN_ITEM_DISPLAY.get()) : UNKNOWN_ITEM_DISPLAY.get();
    }

    /**
     * Adds an ItemStack to a list, only if the item does not stack with any of the items already in the list.
     *
     * @param items The list to add to.
     * @param toAdd The entry to add.
     */
    private static void addStacking(List<class_1799> items, class_1799 toAdd) {
        for (class_1799 existing : items) {
            if (Objects.equals(existing, toAdd) || class_1799.method_31577(existing, toAdd)) {
                return;
            }
        }
        items.add(toAdd);
    }

    private static List<class_1799> getTagItems(class_6862<class_1792> tag) {
        final List<class_1799> items = new ArrayList<>();
        for (class_6880<class_1792> item : class_7923.field_41178.method_40286(tag)) {
            items.add(new class_1799(item));
        }
        return items;
    }
}