package net.darkhax.bookshelf.common.api.util;

import net.darkhax.bookshelf.common.api.PhysicalSide;
import net.darkhax.bookshelf.common.api.annotation.OnlyFor;
import net.darkhax.bookshelf.common.api.service.Services;
import net.darkhax.bookshelf.common.api.text.unit.Units;
import net.darkhax.bookshelf.common.mixin.access.client.AccessorFontManager;
import net.darkhax.bookshelf.common.mixin.access.client.AccessorMinecraft;
import net.darkhax.bookshelf.common.mixin.access.entity.AccessorEntity;
import net.minecraft.class_1074;
import net.minecraft.class_1297;
import net.minecraft.class_1799;
import net.minecraft.class_1937;
import net.minecraft.class_2558;
import net.minecraft.class_2561;
import net.minecraft.class_2568;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_3544;
import net.minecraft.class_5244;
import net.minecraft.class_5250;
import net.minecraft.class_6862;
import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.Nullable;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.stream.Collectors;

public class TextHelper {

    /**
     * Creates translated text from a resource location.
     *
     * @param prefix   The prefix to add at the start of the key.
     * @param suffix   The suffix to add at the end of the key.
     * @param location The resource location to use in the key.
     * @param args     An optional array of arguments to format into the translated text.
     * @return A translated component based on the resource location.
     */
    public static class_5250 fromResourceLocation(@Nullable String prefix, @Nullable String suffix, class_2960 location, Object... args) {
        final StringBuilder builder = new StringBuilder();
        if (prefix != null) {
            builder.append(prefix).append(".");
        }
        builder.append(location.method_12836()).append(".").append(location.method_12832());
        if (suffix != null) {
            builder.append(".").append(suffix);
        }
        return class_2561.method_43469(builder.toString(), args);
    }

    /**
     * Formats a duration of time in ticks into its real world time counterpart. This method should ONLY be used if a
     * world context is not available or if the duration of time is not affected by custom world tick rates.
     *
     * @param ticks The duration of ticks.
     * @return The formatted time duration.
     */
    public static class_5250 formatDuration(int ticks) {

        return formatDuration(ticks, true, 1f);
    }

    /**
     * Formats a duration of time in ticks into its real world time counterpart.
     *
     * @param ticks The duration of ticks.
     * @param level The world level that the duration is taking place. This is used to account for custom tick rates set
     *              using the game rule.
     * @return The formatted time duration.
     */
    public static class_5250 formatDuration(int ticks, class_1937 level) {
        return formatDuration(ticks, true, level);
    }

    /**
     * Formats a duration of time in ticks into its real world time counterpart.
     *
     * @param ticks        The duration of ticks.
     * @param includeHover Should the raw tick amount be shown when hovering the text?
     * @param level        The world level that the duration is taking place. This is used to account for custom tick
     *                     rates set using the game rule.
     * @return The formatted time duration.
     */
    public static class_5250 formatDuration(int ticks, boolean includeHover, class_1937 level) {
        return formatDuration(ticks, includeHover, level.method_54719().method_54748());
    }

    /**
     * Formats a duration of time in ticks into its real world time counterpart.
     *
     * @param ticks            The duration of ticks.
     * @param showTicksOnHover Should the raw tick amount be shown when hovering the text?
     * @param tickRate         The tick rate of the current world.
     * @return The formatted time duration.
     */
    public static class_5250 formatDuration(int ticks, boolean showTicksOnHover, float tickRate) {
        class_5250 timeText = class_2561.method_43470(class_3544.method_15439(ticks, tickRate));
        if (showTicksOnHover) {
            timeText = Units.TICK.format(ticks);
        }
        return timeText;
    }

    /**
     * Applies hover text to a text component.
     *
     * @param base  The base text component to append.
     * @param hover The text to display while hovering.
     * @return A component instance with the hover event applied.
     */
    public static class_5250 withHover(class_2561 base, class_2561 hover) {
        return withHover(base, new class_2568(class_2568.class_5247.field_24342, hover));
    }

    /**
     * Applies hover text based on an entity to a text component.
     *
     * @param base  The base text component to append.
     * @param hover The Entity to display in the hover text.
     * @return A component instance with the hover event applied.
     */
    public static class_5250 withHover(class_2561 base, class_1297 hover) {
        return withHover(base, hoverEvent(hover));
    }

    /**
     * Applies hover text based on an item to a text component.
     *
     * @param base  The base text component to append.
     * @param hover The ItemStack to display in the hover text.
     * @return A component instance with the hover event applied.
     */
    public static class_5250 withHover(class_2561 base, class_1799 hover) {
        return withHover(base, new class_2568(class_2568.class_5247.field_24343, new class_2568.class_5249(hover)));
    }

    /**
     * Applies a hover event to a text component.
     *
     * @param base  The base text component to append.
     * @param hover The hover event to apply.
     * @return A component instance with the hover event applied.
     */
    public static class_5250 withHover(class_2561 base, class_2568 hover) {
        return mutable(base).method_27694(style -> style.method_10949(hover));
    }

    /**
     * Creates a new hover event for an entity.
     *
     * @param entity The entity to create a HoverEvent for.
     * @return When Mixins are available the entity will create its own HoverEvent, otherwise a fallback based on the
     * default implementation will be used.
     */
    public static class_2568 hoverEvent(class_1297 entity) {
        if (entity instanceof AccessorEntity access) {
            return access.bookshelf$createHoverEvent();
        }
        return new class_2568(class_2568.class_5247.field_24344, new class_2568.class_5248(entity.method_5864(), entity.method_5667(), entity.method_5477()));
    }

    /**
     * Provides mutable access to a component.
     *
     * @param component The component to access.
     * @return If the component is already mutable the same component instance will be returned. Otherwise, a mutable
     * copy of the component will be created.
     */
    public static class_5250 mutable(class_2561 component) {
        return component instanceof class_5250 mutable ? mutable : component.method_27661();
    }

    /**
     * Recursively applies a font to text and all of its subcomponents.
     *
     * @param text The text to apply the font to.
     * @param font The ID of the font to apply.
     * @return The input text with the font applied to its style and the style of its subcomponents.
     */
    public static class_2561 applyFont(class_2561 text, class_2960 font) {
        if (text == class_5244.field_39003) {
            return text;
        }
        final class_5250 modified = mutable(text);
        modified.method_27694(style -> style.method_27704(font));
        modified.method_10855().forEach(sib -> applyFont(sib, font));
        return modified;
    }

    /**
     * Attempts to localize several different translation keys and will return the first one that is available on the
     * client. If no keys are mapped the result will be null.
     *
     * @param id   An ID the format within each key using basic string formatting. The first parameter is the namespace
     *             and the second is the path. For example if a key was "tooltip.{0}.{1}.info" the ID "minecraft:stick"
     *             will produce a final key of "tooltip.minecraft.stick.info".
     * @param keys An array of translation keys to attempt localizing.
     * @return A component for the first translation key that is mapped, or null if none of the keys are mapped.
     */
    @Nullable
    @OnlyFor(PhysicalSide.CLIENT)
    public static class_5250 lookupTranslationWithAlias(class_2960 id, String... keys) {
        for (String key : keys) {
            final class_5250 lookupResult = lookupTranslation(key.formatted(id.method_12836(), id.method_12832()));
            if (lookupResult != null) {
                return lookupResult;
            }
        }
        return null;
    }

    /**
     * Attempts to localize several different translation keys and will return the first one that is available on the
     * client. If no keys are mapped the result will be null.
     *
     * @param keys   An array of translation keys to attempt localizing.
     * @param params Arguments that are passed into the translated text.
     * @return A component for the first translation key that is mapped, or null if none of the keys are mapped.
     */
    @Nullable
    @OnlyFor(PhysicalSide.CLIENT)
    public static class_5250 lookupTranslationWithAlias(String[] keys, Object... params) {
        for (String key : keys) {
            final class_5250 lookupResult = lookupTranslation(key, params);
            if (lookupResult != null) {
                return lookupResult;
            }
        }
        return null;
    }

    /**
     * Attempts to localize text. If the translation key is not mapped on the client the component will be null.
     *
     * @param key  The translation key to localize.
     * @param args Arguments that are passed into the translated text.
     * @return If the key can be translated a component will be returned, otherwise null.
     */
    @Nullable
    @OnlyFor(PhysicalSide.CLIENT)
    public static class_5250 lookupTranslation(String key, Object... args) {
        return lookupTranslation(key, (s, o) -> null, args);
    }

    /**
     * Attempts to localize text. If the translation key is not mapped on the client the fallback will be used.
     *
     * @param key      The translation key to localize.
     * @param fallback The fallback text to use when the key is unavailable.
     * @param args     Arguments that are passed into the translated text.
     * @return If the key can be translated a component will be returned, otherwise the fallback will be used.
     */
    @Nullable
    @OnlyFor(PhysicalSide.CLIENT)
    public static class_5250 lookupTranslation(String key, class_5250 fallback, Object... args) {
        return lookupTranslation(key, (s, o) -> fallback, args);
    }

    /**
     * Attempts to localize text. If the translation key is not mapped on the client it will try to use the fallback.
     *
     * @param key      The translation key to localize.
     * @param fallback A function that provides fallback text based on the original translation key and arguments. Both
     *                 the function and the result of this function may be null.
     * @param args     Arguments that are passed into the translated text.
     * @return If the key can be translated a component will be returned, otherwise the fallback will be used.
     */
    @Nullable
    @OnlyFor(PhysicalSide.CLIENT)
    public static class_5250 lookupTranslation(String key, @Nullable BiFunction<String, Object[], class_5250> fallback, Object... args) {
        if (!Services.PLATFORM.isPhysicalClient()) {
            throw new IllegalStateException("Text can not be translated on the server.");
        }
        return class_1074.method_4663(key) ? class_2561.method_43469(key, args) : fallback != null ? fallback.apply(key, args) : null;
    }

    /**
     * Creates a text component that will copy the value to the players clipboard when they click it.
     *
     * @param text The text to display and copy to the clipboard.
     * @return A component that displays text and copies that text to the clipboard when the player clicks on it.
     */
    public static class_5250 copyText(String text) {
        return setCopyText(class_2561.method_43470(text), text);
    }

    /**
     * Adds a click event to a text component that will copy text to the players clipboard when they click on it.
     *
     * @param component The component to attack the click event to.
     * @param copy      The text to be copied to the clipboard.
     * @return A text component that will copy the text when the player clicks on it.
     */
    public static class_5250 setCopyText(class_5250 component, String copy) {
        return component.method_27694(style -> style.method_10958(new class_2558(class_2558.class_2559.field_21462, copy)));
    }

    /**
     * Joins several components together using a separator.
     *
     * @param separator The separator to insert between other components.
     * @param toJoin    The components to join together.
     * @return A component containing the joint components.
     */
    public static class_5250 join(class_2561 separator, class_2561... toJoin) {
        return join(separator, Arrays.stream(toJoin).iterator());
    }

    /**
     * Joins several components together using a separator.
     *
     * @param separator The separator to insert between other components.
     * @param toJoin    The components to join together.
     * @return A component containing the joint components.
     */
    public static class_5250 join(class_2561 separator, Collection<class_2561> toJoin) {
        return join(separator, toJoin.iterator());
    }

    /**
     * Joins several components together using a separator.
     *
     * @param separator The separator to insert between other components.
     * @param toJoin    The components to join together.
     * @return A component containing the joint components.
     */
    public static class_5250 join(class_2561 separator, Iterator<class_2561> toJoin) {
        final class_5250 joined = class_2561.method_43470("");
        while (toJoin.hasNext()) {
            joined.method_10852(toJoin.next());
            if (toJoin.hasNext()) {
                joined.method_10852(separator);
            }
        }
        return joined;
    }

    /**
     * Finds a set of possible matches within an iterable group of strings. This can be used to take invalid user input
     * and attempt to find a plausible match using known good values.
     * <p>
     * Possible matches are determined using the Levenshtein distance between the input value and the potential
     * candidates. The Levenshtein distance represents the number of characters that need to be changed in order for the
     * strings to match. For example "abc" to "def" has a difference of three, while "123" to "1234" has a distance of
     * 1.
     *
     * @param input      The input string.
     * @param candidates An iterable group of possible candidates.
     * @return A set of possible matches for the input. This set will include all candidates that have the lowest
     * possible distance. For example if there were 100 candidates and five had a distance of one all five of the lowest
     * distance values will be returned.
     */
    public static Set<String> getPossibleMatches(String input, Iterable<String> candidates) {
        return getPossibleMatches(input, candidates, Integer.MAX_VALUE);
    }

    /**
     * Finds a set of possible matches within an iterable group of strings. This can be used to take invalid user input
     * and attempt to find a plausible match using known good values.
     * <p>
     * Possible matches are determined using the Levenshtein distance between the input value and the potential
     * candidates. The Levenshtein distance represents the number of characters that need to be changed in order for the
     * strings to match. For example "abc" to "def" has a difference of three, while "123" to "1234" has a distance of
     * 1.
     *
     * @param input      The input string.
     * @param candidates An iterable group of possible candidates.
     * @param threshold  The maximum distance allowed for a value to be considered. For example if the threshold is two,
     *                   only entries with a distance of two or less will be considered.
     * @return A set of possible matches for the input. This set will include all candidates that have the lowest
     * possible distance. For example if there were 100 candidates and five had a distance of one all five of the lowest
     * distance values will be returned.
     */
    public static Set<String> getPossibleMatches(String input, Iterable<String> candidates, int threshold) {
        final HashSet<String> bestMatches = new HashSet();
        int distance = threshold;
        for (String candidate : candidates) {
            final int currentDistance = StringUtils.getLevenshteinDistance(input, candidate);
            if (currentDistance < distance) {
                distance = currentDistance;
                bestMatches.clear();
                bestMatches.add(candidate);
            }
            else if (currentDistance == distance) {
                bestMatches.add(candidate);
            }
        }
        return bestMatches;
    }

    /**
     * Formats a collection of values to a string using {@link Object#toString()}. If the collection has more than one
     * value each entry will be separated by commas. Each value will also be quoted.
     *
     * @param collection The collection of values to format.
     * @param <T>        The type of value being formatted.
     * @return The formatted string.
     */
    public static <T> String formatCollection(Collection<T> collection) {
        return formatCollection(collection, entry -> "\"" + entry.toString() + "\"", ", ");
    }

    /**
     * Formats a collection of values to a string. If the collection has more than one value each entry will be
     * separated using the delimiter.
     *
     * @param collection The collection of values to format.
     * @param formatter  A function used to format the value to a string.
     * @param delimiter  A delimiter used to separate values in a list.
     * @param <T>        The type of value being formatted.
     * @return The formatted string.
     */
    public static <T> String formatCollection(Collection<T> collection, Function<T, String> formatter, String delimiter) {
        return collection.size() == 1 ? formatter.apply(collection.stream().findFirst().get()) : collection.stream().map(formatter).collect(Collectors.joining(delimiter));
    }

    @OnlyFor(PhysicalSide.CLIENT)
    public static Set<class_2960> getRegisteredFonts() {
        if (!Services.PLATFORM.isPhysicalClient()) {
            return Collections.emptySet();
        }
        return ((AccessorFontManager) (((AccessorMinecraft) class_310.method_1551()).bookshelf$getFontManager())).bookshelf$getFonts().keySet();
    }

    /**
     * Creates a translation key that should map to a display name for the tag.
     * <p>
     * Tags for vanilla registries use the format tag.reg_path.namespace.path and tags for modded registries use the
     * format tag.reg_namespace.reg_path.namespace.path.
     * <p>
     * This is a new standard being pushed by the Fabric API and recipe viewers. While it has not been universally
     * adopted yet, it should be considered best practice to do so moving forward.
     *
     * @param tag The tag to provide a name key for.
     * @return A translation key that should map to a display name.
     */
    public static String getTagName(class_6862<?> tag) {
        final StringBuilder builder = new StringBuilder();
        builder.append("tag.");
        final class_2960 regId = tag.comp_326().method_29177();
        final class_2960 tagId = tag.comp_327();
        if (!regId.method_12836().equals(class_2960.field_33381)) {
            builder.append(regId.method_12836()).append(".");
        }
        builder.append(regId.method_12832().replace("/", ".")).append(".").append(tagId.method_12836()).append(".").append(tagId.method_12832().replace("/", ".").replace(":", "."));
        return builder.toString();
    }
}