package net.darkhax.bookshelf.common.api.network;

import net.darkhax.bookshelf.common.api.PhysicalSide;
import net.darkhax.bookshelf.common.api.annotation.OnlyFor;
import net.darkhax.bookshelf.common.api.service.Services;
import net.darkhax.bookshelf.common.impl.Constants;
import net.minecraft.class_310;
import net.minecraft.class_3218;
import net.minecraft.class_3222;
import net.minecraft.class_3324;
import net.minecraft.class_8710;
import net.minecraft.class_9129;
import net.minecraft.class_9139;
import net.minecraft.server.MinecraftServer;
import org.jetbrains.annotations.Nullable;

/**
 * Defines a custom payload packet. These packets must be registered using an
 * {@link net.darkhax.bookshelf.common.api.registry.IContentProvider}.
 *
 * @param <T> The type of the payload.
 */
public interface IPacket<T extends class_8710> {

    /**
     * Gets the payload type.
     *
     * @return The payload type.
     */
    class_8710.class_9154<T> type();

    /**
     * Gets a stream codec that can serialize the payload across the network.
     *
     * @return The stream coded used to serialize the payload.
     */
    class_9139<class_9129, T> streamCodec();

    /**
     * Defines how the packet is meant to be sent and where it should be handled.
     *
     * @return The intended destination of the packet.
     */
    Destination destination();

    /**
     * This method will be called when the custom payload is received. This method can be called on both the client and
     * server, depending on the destination type defined by {@link #destination()}.
     *
     * @param sender   The sender of the packet. This will always be null for packets handled on the client.
     * @param isServer True when the packet is being handled on the server.
     * @param payload  The payload that was received.
     */
    void handle(@Nullable class_3222 sender, boolean isServer, T payload);

    /**
     * Sends the packet from the server to a specific player.
     *
     * @param recipient The intended recipient of the payload.
     * @param payload   The payload to send.
     */
    default void toPlayer(class_3222 recipient, T payload) {
        if (!this.destination().handledByClient()) {
            Constants.LOG.error("Attempted to send invalid packet {} to client! Class: {} Destination: {} Payload: {}", this.type().comp_2242(), this.getClass(), this.destination(), payload.toString());
            throw new IllegalStateException("Attempted to send invalid packet " + this.type().comp_2242() + " to client!");
        }
        Services.NETWORK.sendToPlayer(recipient, payload);
    }

    /**
     * Sends the packet from the server to all connected players.
     *
     * @param level   A serverside level, used to access the player list.
     * @param payload The payload to send.
     */
    default void toAllPlayers(class_3218 level, T payload) {
        toAllPlayers(level.method_8503(), payload);
    }

    /**
     * Sends the packet from the server to all connected players.
     *
     * @param server  The server instance, used to access the player list.
     * @param payload The payload to send.
     */
    default void toAllPlayers(MinecraftServer server, T payload) {
        toAllPlayers(server.method_3760(), payload);
    }

    /**
     * Sends the packet from the server to all connected players.
     *
     * @param playerList The player list.
     * @param payload    The payload to send.
     */
    default void toAllPlayers(class_3324 playerList, T payload) {
        for (class_3222 player : playerList.method_14571()) {
            toPlayer(player, payload);
        }
    }

    /**
     * Sends a packet from a client to the server.
     *
     * @param payload The payload to send.
     */
    @OnlyFor(PhysicalSide.CLIENT)
    default void toServer(T payload) {
        if (!this.destination().handledByServer()) {
            Constants.LOG.error("Attempted to send invalid packet {} to server! Class: {} Destination: {} Payload: {}", this.type().comp_2242(), this.getClass(), this.destination(), payload.toString());
            throw new IllegalStateException("Attempted to send invalid packet " + this.type().comp_2242() + " to server!");
        }
        if (class_310.method_1551().method_1562() == null) {
            Constants.LOG.error("Attempted to send packet {} before a connection to a server has been established!", this.type().comp_2242());
            throw new IllegalStateException("Attempted to send packet " + this.type().comp_2242() + " before being connected to a server!");
        }
        Services.NETWORK.sendToServer(payload);
    }
}