package net.darkhax.bookshelf.api.util;

import net.minecraft.class_1657;

public final class ExperienceHelper {

    /**
     * Attempts to charge the player an experience point cost. If the player can not afford the full amount they will
     * not be charged and false will be returned.
     *
     * @param player The player to charge.
     * @param cost   The amount to charge the player in experience points.
     * @return True if the amount was paid.
     */
    public static boolean chargeExperiencePoints(class_1657 player, int cost) {

        final int playerExperience = getExperiencePoints(player);

        if (playerExperience >= cost) {

            player.method_7255(-cost);

            // The underlying EXP system uses a float which is prone to rounding errors. This will sometimes leave
            // players with a small fraction of exp progress that is worth less than 1 exp point. These rounding errors
            // are so small that they do not introduce functionality issues however they can trigger a vanilla bug
            // where the EXP bar will still render a few pixels of progress even when the player has no exp points. To
            // prevent this issue here we simply reset all progress when the player spends all of their points.
            if (getExperiencePoints(player) <= 0) {

                player.field_7510 = 0f;
            }

            return true;
        }

        return false;
    }

    /**
     * Calculates the amount of experience points the player currently has. This should be used in favour of
     * {@link class_1657#field_7495} which deceptively does not track the amount of experience the player currently
     * has.
     * <p>
     * Contrary to popular belief the {@link class_1657#field_7495} value actually loosely represents how much
     * experience points the player has earned during their current life. This value is akin to the old player score
     * metric and appears to be predominantly legacy code. Relying on this value is often incorrect as negative changes
     * to the player level such as enchanting, the anvil, and the level command will not reduce this value.
     *
     * @param player The player to calculate the total experience points of.
     * @return The amount of experience points held by the player.
     */
    public static int getExperiencePoints(class_1657 player) {

        // Start by calculating how many EXP points the player's current level is worth.
        int exp = getTotalPointsForLevel(player.field_7520);

        // Add the amount of experience points the player has earned towards their next level.
        exp += player.field_7510 * getTotalPointsForLevel(player.field_7520 + 1);

        return exp;
    }

    /**
     * Calculates the amount of additional experience points required to reach the given level when starting from the
     * previous level. This will also be the amount of experience points that an individual level is worth.
     *
     * @param level The level to calculate the point step for.
     * @return The amount of points required to reach the given level when starting from the previous level.
     */
    public static int getPointForLevel(int level) {

        if (level == 0) {

            return 0;
        }

        else if (level > 30) {

            return 112 + (level - 31) * 9;
        }

        else if (level > 15) {

            return 37 + (level - 16) * 5;
        }

        else {

            return 7 + (level - 1) * 2;
        }
    }

    /**
     * Calculates the amount of additional experience points required to reach the target level when starting from the
     * starting level.
     *
     * @param startingLevel The level to start the calculation at.
     * @param targetLevel   The level to reach.
     * @return The amount of additional experience points required to go from the starting level to the target level.
     */
    public static int getPointsForLevel(int startingLevel, int targetLevel) {

        if (targetLevel < startingLevel) {

            throw new IllegalArgumentException("Starting level must be lower than the target level!");
        }

        else if (startingLevel < 0) {

            throw new IllegalArgumentException("Level bounds must be positive!");
        }

        // If the levels are the same there is no point difference.
        else if (targetLevel == startingLevel) {

            return 0;
        }

        int requiredPoints = 0;

        for (int lvl = startingLevel + 1; lvl <= targetLevel; lvl++) {

            requiredPoints += getPointForLevel(lvl);
        }

        return requiredPoints;
    }

    /**
     * Calculates the total amount of experience points required to reach a given level when starting at level 0.
     *
     * @param level The target level to reach.
     * @return The amount of experience points required to reach the target level.
     */
    public static int getTotalPointsForLevel(int level) {

        return getPointsForLevel(0, level);
    }
}
