package net.darkhax.bookshelf.api.block;

import net.minecraft.class_1538;
import net.minecraft.class_1937;
import net.minecraft.class_2246;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2680;

/**
 * This interface allows implementing blocks to respond to lighting strikes and redirect lightning strikes to other
 * adjacent blocks.
 */
public interface ILightningConductive {

    /**
     * A default array of faces for conductive blocks to redirect lightning to.
     */
    class_2350[] LIGHTNING_REDIRECTION_FACES = new class_2350[]{class_2350.field_11043, class_2350.field_11034, class_2350.field_11035, class_2350.field_11039, class_2350.field_11033};

    /**
     * An empty array of lightning redirection faces. Can be used when the block should not redirect lightning.
     */
    class_2350[] NO_REDIRECTION_FACES = new class_2350[]{class_2350.field_11043, class_2350.field_11034, class_2350.field_11035, class_2350.field_11039, class_2350.field_11033};

    /**
     * This method is called when a lightning bolt directly strikes the blocks.
     *
     * @param world     The world the block is in.
     * @param pos       The position the lightning bolt struck.
     * @param state     The state of the block struck by the lighting bolt.
     * @param lightning The lightning bolt that struck the block.
     */
    default void onDirectLightningStrike(class_1937 world, class_2338 pos, class_2680 state, class_1538 lightning) {

        // No-op
    }

    /**
     * This method is called when a lightning bolt strikes an adjacent conductive block such as a lightning rod.
     *
     * @param world         The world the block is in.
     * @param strikePos     The position the lightning bolt struck.
     * @param strikeState   The state of the block struck by the lightning bolt.
     * @param indirectPos   The position of this block in the world.
     * @param indirectState The state of this block.
     * @param lightning     The lighting bolt that indirectly struck our block.
     */
    default void onIndirectLightingStrike(class_1937 world, class_2338 strikePos, class_2680 strikeState, class_2338 indirectPos, class_2680 indirectState, class_1538 lightning) {

        // No-Op
    }

    /**
     * Checks if the block should trigger
     * {@link ILightningConductive#onIndirectLightingStrike(class_1937, class_2338, class_2680, class_2338, class_2680,
     * class_1538)} of adjacent blocks when directly struck by lightning.
     *
     * @param world       The world the block is in.
     * @param strikePos   The position the lightning bolt struck.
     * @param strikeState The state of the block struck by the lightning bolt.
     * @return Will the lightning indirectly strike adjacent blocks.
     */
    default boolean canRedirectLightning(class_1937 world, class_2338 strikePos, class_2680 strikeState) {

        return strikeState.method_27852(class_2246.field_27171);
    }

    /**
     * Gets an array of block faces that should conduct lightning when struck. If
     * {@link #canRedirectLightning(class_1937, class_2338, class_2680)} is true,
     * {@link #onIndirectLightingStrike(class_1937, class_2338, class_2680, class_2338, class_2680, class_1538)} will be
     * invoked on blocks adjacent to the faces returned by this method.
     *
     * @param world       The world the block is in.
     * @param strikePos   The position the lightning bolt struck.
     * @param strikeState The state of the block struck by the lightning bolt.
     * @return An array of adjacent faces that will be indirectly struck by lightning.
     */
    default class_2350[] getLightningRedirectionFaces(class_1937 world, class_2338 strikePos, class_2680 strikeState) {

        return canRedirectLightning(world, strikePos, strikeState) ? LIGHTNING_REDIRECTION_FACES : NO_REDIRECTION_FACES;
    }
}
