package net.darkhax.bookshelf.impl.gametest;

import com.google.common.collect.Iterables;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import io.netty.buffer.Unpooled;
import net.darkhax.bookshelf.api.serialization.ISerializer;
import net.darkhax.bookshelf.mixin.accessors.util.random.AccessorWeightedRandomList;
import net.minecraft.class_2487;
import net.minecraft.class_2499;
import net.minecraft.class_2520;
import net.minecraft.class_2540;
import net.minecraft.class_4516;
import net.minecraft.class_6005;
import net.minecraft.class_6008;
import net.minecraft.class_6302;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.BiPredicate;

public class TestSerialization<T> implements ITestable {

    /**
     * A short string that denotes the type of data serialized by the serializer being tested. This is purely to help
     * display information about the tests.
     */
    private final String type;

    /**
     * The serializer to test.
     */
    private final ISerializer<T> serializer;

    /**
     * A singleton instance to test serialization with. This is considered immutable.
     */
    private final T singleton;

    /**
     * A collection of instances to test collection based serialization with. This is considered immutable.
     */
    private final T[] collection;

    private final BiPredicate<T, T> equality;

    public TestSerialization(String type, ISerializer<T> serializer, T... collection) {

        this(type, serializer, Objects::equals, collection[0], collection);
    }

    public TestSerialization(String type, ISerializer<T> serializer, BiPredicate<T, T> equality, T... collection) {

        this(type, serializer, equality, collection[0], collection);
    }

    public TestSerialization(String type, ISerializer<T> serializer, BiPredicate<T, T> equality, T singleton, T... collection) {

        this.type = type;
        this.serializer = serializer;
        this.equality = equality;
        this.singleton = singleton;
        this.collection = collection;
    }

    @class_6302
    public void testJsonSingleton(class_4516 helper) {

        final JsonElement toJson = this.serializer.toJSON(this.singleton);
        final T fromJson = this.serializer.fromJSON(toJson);

        if (!this.equality.test(this.singleton, fromJson)) {

            helper.method_35995("Singleton from" + this.type + " JSON does not match! a= " + this.singleton + " b= " + fromJson);
        }

        else {

            helper.method_36036();
        }
    }

    @class_6302
    public void testBytebufSingleton(class_4516 helper) {

        final class_2540 buffer = new class_2540(Unpooled.buffer());
        this.serializer.toByteBuf(buffer, this.singleton);
        final T fromBuffer = this.serializer.fromByteBuf(buffer);

        if (!this.equality.test(this.singleton, fromBuffer)) {

            helper.method_35995("Singleton from" + this.type + " ByteBuf does not match! a= " + this.singleton + " b= " + fromBuffer);
        }

        else {

            helper.method_36036();
        }
    }

    @class_6302
    public void testNbtSingleton(class_4516 helper) {

        final class_2520 toNBT = this.serializer.toNBT(this.singleton);
        final T fromNBT = this.serializer.fromNBT(toNBT);

        if (!this.equality.test(this.singleton, fromNBT)) {

            helper.method_35995("Singleton from" + this.type + " NBT does not match! a= " + this.singleton + " b= " + fromNBT);
        }

        else {

            helper.method_36036();
        }
    }

    @class_6302
    public void testJsonWeightedSingleton(class_4516 helper) {

        final class_6008.class_6010<T> weightedSingleton = class_6008.method_34980(this.singleton, 14);
        final JsonElement toJson = this.serializer.toJSONWeighted(weightedSingleton);
        final class_6008.class_6010<T> fromJson = this.serializer.fromJSONWeighted(toJson);

        if (!weightedSingleton.method_34979().equals(fromJson.method_34979())) {

            helper.method_35995("Weight for JSON does not match! a= " + weightedSingleton.method_34979() + " b= " + fromJson.method_34979());
        }

        if (!this.equality.test(this.singleton, fromJson.method_34983())) {

            helper.method_35995("Weighted Singleton from" + this.type + " JSON does not match! a= " + this.singleton + " b= " + fromJson.method_34983());
        }

        else {

            helper.method_36036();
        }
    }

    @class_6302
    public void testBytebufWeightedSingleton(class_4516 helper) {

        final class_6008.class_6010<T> weightedSingleton = class_6008.method_34980(this.singleton, 14);
        final class_2540 buffer = new class_2540(Unpooled.buffer());
        this.serializer.toByteBufWeighted(buffer, weightedSingleton);
        final class_6008.class_6010<T> fromBuffer = this.serializer.fromByteBufWeighted(buffer);

        if (!weightedSingleton.method_34979().equals(fromBuffer.method_34979())) {

            helper.method_35995("Weight for ByteBuf does not match! a= " + weightedSingleton.method_34979() + " b= " + fromBuffer.method_34979());
        }

        if (!this.equality.test(this.singleton, fromBuffer.method_34983())) {

            helper.method_35995("Weighted Singleton from" + this.type + " ByteBuf does not match! a= " + this.singleton + " b= " + fromBuffer.method_34983());
        }

        else {

            helper.method_36036();
        }
    }

    @class_6302
    public void testJsonOptionalSingleton(class_4516 helper) {

        final Optional<T> original = Optional.ofNullable(this.singleton);
        final JsonElement toJson = this.serializer.toJSONOptional(original);
        final Optional<T> fromJson = this.serializer.fromJSONOptional(toJson);

        if (original.isPresent() && fromJson.isPresent() && !this.equality.test(original.get(), fromJson.get())) {

            helper.method_35995("Optional Singleton from" + this.type + " JSON does not match! a= " + original.get() + " b= " + fromJson.get());
        }

        if (original.isPresent() != fromJson.isPresent()) {

            helper.method_35995("Optional Singleton from " + this.type + " JSON availability mismatch. a= " + original.isPresent() + " b= " + fromJson.isPresent());
        }

        else {

            helper.method_36036();
        }
    }

    @class_6302
    public void testBytebufOptionalSingleton(class_4516 helper) {

        final Optional<T> original = Optional.ofNullable(this.singleton);
        final class_2540 buffer = new class_2540(Unpooled.buffer());
        this.serializer.toByteBufOptional(buffer, original);
        final Optional<T> fromByteBuf = this.serializer.fromByteBufOptional(buffer);

        if (original.isPresent() && fromByteBuf.isPresent() && !this.equality.test(original.get(), fromByteBuf.get())) {

            helper.method_35995("Optional Singleton from" + this.type + " ByteBuf does not match! a= " + original.get() + " b= " + fromByteBuf.get());
        }

        if (original.isPresent() != fromByteBuf.isPresent()) {

            helper.method_35995("Optional Singleton from " + this.type + " ByteBuf availability mismatch. a= " + original.isPresent() + " b= " + fromByteBuf.isPresent());
        }

        else {

            helper.method_36036();
        }
    }

    @class_6302
    public void testJsonEmptyOptionalSingleton(class_4516 helper) {

        final Optional<T> original = Optional.empty();
        final JsonElement toJson = this.serializer.toJSONOptional(original);
        final Optional<T> fromJson = this.serializer.fromJSONOptional(toJson);

        if (original.isPresent() || fromJson.isPresent()) {

            helper.method_35995("Empty Optional Singleton from" + this.type + " JSON was not empty! a= " + original.isPresent() + " b= " + fromJson.isPresent());
        }

        else {

            helper.method_36036();
        }
    }

    @class_6302
    public void testBytebufEmptyOptionalSingleton(class_4516 helper) {

        final Optional<T> original = Optional.empty();
        final class_2540 buffer = new class_2540(Unpooled.buffer());
        this.serializer.toByteBufOptional(buffer, original);
        final Optional<T> fromByteBuf = this.serializer.fromByteBufOptional(buffer);

        if (original.isPresent() || fromByteBuf.isPresent()) {

            helper.method_35995("Empty Optional Singleton from" + this.type + " JSON was not empty! a= " + original.isPresent() + " b= " + fromByteBuf.isPresent());
        }

        else {

            helper.method_36036();
        }
    }

    @class_6302
    public void testJsonList(class_4516 helper) {

        final List<T> list = List.of(this.collection);
        final JsonElement written = this.serializer.toJSONList(list);
        final List<T> read = this.serializer.fromJSONList(written);

        assertIterableEqual(helper, list, read);
    }

    @class_6302
    public void testBytebufList(class_4516 helper) {

        final class_2540 buffer = new class_2540(Unpooled.buffer());
        final List<T> list = List.of(this.collection);
        this.serializer.toByteBufList(buffer, list);
        final List<T> read = this.serializer.fromByteBufList(buffer);

        assertIterableEqual(helper, list, read);
    }

    @class_6302
    public void testNbtList(class_4516 helper) {

        final List<T> list = List.of(this.collection);
        final class_2487 tag = new class_2487();
        this.serializer.toNBTList(tag, "test_list", list);

        if (!tag.method_10545("test_list")) {

            helper.method_35995("List was not written to tag.");
        }

        if (tag.method_10580("test_list") instanceof class_2499 listTag && listTag.size() != list.size()) {

            helper.method_35995("List size does not match on write. Has " + listTag.size() + " expected " + list.size());
        }

        final List<T> read = this.serializer.fromNBTList(tag, "test_list");

        assertIterableEqual(helper, list, read);
    }

    @class_6302
    public void testJsonSet(class_4516 helper) {

        final Set<T> set = this.createSet();
        final JsonElement written = this.serializer.toJSONSet(set);
        final Set<T> read = this.serializer.fromJSONSet(written);

        assertIterableEqual(helper, set, read);
    }

    @class_6302
    public void testBytebufSet(class_4516 helper) {

        final class_2540 buffer = new class_2540(Unpooled.buffer());
        final Set<T> set = this.createSet();
        this.serializer.writeByteBufSet(buffer, set);
        final Set<T> read = this.serializer.readByteBufSet(buffer);

        assertIterableEqual(helper, set, read);
    }

    @class_6302
    public void testJsonWeightedList(class_4516 helper) {

        final class_6005<T> original = this.createWeightedList();
        final JsonElement written = this.serializer.toJSONWeightedList(original);
        final class_6005<T> read = this.serializer.fromJSONWeightedList(written);

        assertWeightedListEqual(helper, original, read);
    }

    @class_6302
    public void testBytebufWeightedList(class_4516 helper) {

        final class_6005<T> original = this.createWeightedList();
        final class_2540 buffer = new class_2540(Unpooled.buffer());
        this.serializer.toByteBufWeightedList(buffer, original);
        final class_6005<T> read = this.serializer.fromByteBufWeightedList(buffer);

        assertWeightedListEqual(helper, original, read);
    }

    @class_6302
    public void testJsonListEmpty(class_4516 helper) {

        final List<T> list = new ArrayList<>();
        final JsonObject testObj = new JsonObject();

        this.serializer.toJSONList(testObj, "test", list);

        if (testObj.has("test")) {

            helper.method_35995("Empty list should not be serialized.");
        }

        helper.method_36036();
    }

    @class_6302
    public void testJsonListNull(class_4516 helper) {

        final JsonObject testObj = new JsonObject();

        this.serializer.toJSONList(testObj, "test", null);

        if (testObj.has("test")) {

            helper.method_35995("Empty list should not be serialized.");
        }

        helper.method_36036();
    }

    @class_6302
    public void testJsonNullable(class_4516 helper) {

        final JsonElement nullJson = this.serializer.toJSONNullable(null);

        if (nullJson != null) {

            helper.method_35995("Expected JSON to be null.");
        }

        final T nullValue = this.serializer.fromJSONNullable(nullJson);

        if (nullValue != null) {

            helper.method_35995("Expected value to be null.");
        }

        final JsonElement json = this.serializer.toJSONNullable(this.singleton);

        if (json == null) {

            helper.method_35995("Expected JSON to not be null.");
        }

        final T value = this.serializer.fromJSONNullable(json);

        if (value == null) {

            helper.method_35995("Expected value to not be null.");
        }

        if (!this.equality.test(this.singleton, value)) {

            helper.method_35995("Value written and read do not match.");
        }

        else {

            helper.method_36036();
        }
    }

    @class_6302
    public void testBytebufNullable(class_4516 helper) {

        final class_2540 nullBuffer = new class_2540(Unpooled.buffer());
        this.serializer.toByteBufNullable(nullBuffer, null);
        final T nullValue = this.serializer.fromByteBufNullable(nullBuffer);

        if (nullValue != null) {

            helper.method_35995("Expected value to be null.");
        }

        final class_2540 buffer = new class_2540(Unpooled.buffer());
        this.serializer.toByteBufNullable(buffer, this.singleton);
        final T value = this.serializer.fromByteBufNullable(buffer);

        if (value == null) {

            helper.method_35995("Expected value to not be null.");
        }

        if (!this.equality.test(this.singleton, value)) {

            helper.method_35995("Value written and read do not match.");
        }

        else {

            helper.method_36036();
        }
    }


    private Set<T> createSet() {

        final Set<T> values = new LinkedHashSet<>();
        Collections.addAll(values, this.collection);
        return values;
    }

    private class_6005<T> createWeightedList() {

        final class_6005.class_6006<T> weightedList = class_6005.method_34971();

        for (T value : this.collection) {

            weightedList.method_34975(value, 25);
        }

        return weightedList.method_34974();
    }

    private void assertWeightedListEqual(class_4516 helper, class_6005<T> original, class_6005<T> result) {

        final AccessorWeightedRandomList<class_6008.class_6010<T>> accessOriginal = ((AccessorWeightedRandomList<class_6008.class_6010<T>>) original);
        final AccessorWeightedRandomList<class_6008.class_6010<T>> accessResult = ((AccessorWeightedRandomList<class_6008.class_6010<T>>) result);

        if (accessOriginal.bookshelf$getEntries().size() != accessResult.bookshelf$getEntries().size()) {

            helper.method_35995("List of type " + this.type + " has incorrect size. Original=" + accessOriginal.bookshelf$getEntries().size() + " Result=" + accessResult.bookshelf$getEntries().size());
        }

        else {

            for (int index = 0; index < accessOriginal.bookshelf$getEntries().size(); index++) {

                final class_6008.class_6010<T> a = accessOriginal.bookshelf$getEntries().get(index);
                final class_6008.class_6010<T> b = accessResult.bookshelf$getEntries().get(index);

                if (a.method_34979().method_34976() != b.method_34979().method_34976()) {

                    helper.method_35995("Weighted list of " + this.type + " do not match! index=" + index + " a=" + a.method_34979().method_34976() + " b=" + b.method_34979().method_34976());
                    return;
                }

                if (!this.equality.test(a.method_34983(), b.method_34983())) {

                    helper.method_35995("Weighted list of " + this.type + " do not match! index=" + index + " a=" + a.method_34983() + " b=" + b.method_34983());
                    return;
                }
            }

            helper.method_36036();
        }
    }

    private void assertIterableEqual(class_4516 helper, Iterable<T> original, Iterable<T> result) {

        final int originalSize = Iterables.size(original);
        final int resultSize = Iterables.size(result);

        if (originalSize != resultSize) {

            helper.method_35995("List of type " + this.type + " has incorrect size. Original=" + originalSize + " Result=" + resultSize);
        }

        else {

            for (int index = 0; index < originalSize; index++) {

                final T a = Iterables.get(original, index);
                final T b = Iterables.get(result, index);

                if (!this.equality.test(a, b)) {

                    helper.method_35995("List of " + this.type + " do not match! index=" + index + " a=" + a + " b=" + b);
                    return;
                }
            }

            helper.method_36036();
        }
    }

    @Override
    public String getDefaultBatch() {

        return "bookshelf_serialization_" + this.type;
    }
}