package net.darkhax.bookshelf.api.util;

import net.darkhax.bookshelf.Constants;
import net.darkhax.bookshelf.api.inventory.ContainerInventoryAccess;
import net.darkhax.bookshelf.api.inventory.IInventoryAccess;
import net.darkhax.bookshelf.api.inventory.WorldlyContainerInventoryAccess;
import net.darkhax.bookshelf.mixin.accessors.inventory.AccessorCraftingMenu;
import net.darkhax.bookshelf.mixin.accessors.inventory.AccessorInventoryMenu;
import net.darkhax.bookshelf.mixin.accessors.inventory.AccessorTransientCraftingContainer;
import net.minecraft.class_1263;
import net.minecraft.class_1278;
import net.minecraft.class_1304;
import net.minecraft.class_1309;
import net.minecraft.class_1657;
import net.minecraft.class_1661;
import net.minecraft.class_1703;
import net.minecraft.class_1799;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2371;
import net.minecraft.class_2487;
import net.minecraft.class_2540;
import net.minecraft.class_2586;
import net.minecraft.class_2680;
import net.minecraft.class_3222;
import net.minecraft.class_3908;
import net.minecraft.class_3954;
import net.minecraft.class_5819;
import net.minecraft.class_8566;
import javax.annotation.Nullable;
import java.util.function.BiConsumer;
import java.util.function.Consumer;

public interface IInventoryHelper {

    /**
     * Gets the internal menu context that is backing a crafting container.
     *
     * @param container The crafting container to retrieve internal menu context from.
     * @return The internal menu, or null if the menu could not be found.
     */
    @Nullable
    default class_1703 getCraftingContainer(class_8566 container) {

        return (container instanceof AccessorTransientCraftingContainer accessor) ? accessor.bookshelf$getMenu() : null;
    }

    @Nullable
    default class_1657 getCraftingPlayer(class_1263 container) {

        if (container instanceof class_1661 playerInv) {

            return playerInv.field_7546;
        }

        else if (container instanceof class_8566 crafting) {

            final class_1703 menu = getCraftingContainer(crafting);

            if (menu instanceof AccessorCraftingMenu accessor) {

                return accessor.bookshelf$getPlayer();
            }

            else if (menu instanceof AccessorInventoryMenu accessor) {

                return accessor.bookshelf$getOwner();
            }
        }

        // TODO add some way for dynamic container resolution?
        return null;
    }

    /**
     * Damages an ItemStack by a set amount.
     * <p>
     * ItemStacks with the Unbreakable tag are treated as immune to stack damage as per Minecraft's base spec.
     * <p>
     * ItemStacks with the unbreaking enchantment will have a chance of ignoring the damage.
     * <p>
     * ItemStacks that do not have durability will not be modified.
     * <p>
     * Players in Creative Mode can not damage ItemStacks.
     *
     * @param stack  The ItemStack to damage.
     * @param amount The amount of damage to apply to the item.
     * @param owner  The entity that owns the ItemStack. This is optional but will be used for certain events.
     * @param slot   The slot the ItemStack is in. This is optional and used for the item break animation.
     */
    default class_1799 damageStack(class_1799 stack, int amount, @Nullable class_1309 owner, @Nullable class_1304 slot) {

        // Only items with durability can be damaged. Items with an Unbreakable tag override damage.
        if (stack.method_7963()) {

            final class_5819 random = owner != null ? owner.method_6051() : Constants.RANDOM_SOURCE;

            // If an owner is present, use the entity aware version.
            if (owner != null) {

                stack.method_7956(amount, owner, e -> {

                    if (slot != null) {
                        e.method_20235(slot);
                    }
                });
            }

            // Try to damage the stack directly.
            else if (stack.method_7970(amount, random, null)) {

                // Destroy the ItemStack when it has no more durability.
                stack.method_7934(1);
                stack.method_7974(0);
            }
        }

        return stack;
    }

    default boolean isUnbreakableItem(class_1799 stack) {
        final class_2487 tag = stack.method_7969();
        return tag != null && (tag.method_10577("Unbreaking") || tag.method_10577("Unbreakable"));
    }

    default class_2371<class_1799> keepDamageableItems(class_8566 inv, class_2371<class_1799> keptItems, int damageAmount) {

        @Nullable
        final class_1657 player = this.getCraftingPlayer(inv);

        for (int i = 0; i < keptItems.size(); i++) {

            final class_1799 input = inv.method_5438(i).method_7972();

            if (input.method_7909().method_7846() || isUnbreakableItem(input)) {

                final class_1799 stack = this.damageStack(input, damageAmount, player, null);

                if (!stack.method_7960()) {

                    keptItems.set(i, stack);
                }
            }
        }

        return keptItems;
    }

    /**
     * Creates a list view of a Container's inventory contents.
     *
     * @param inventory The container to view.
     * @return A list view of the provided containers inventory contents.
     */
    default class_2371<class_1799> toList(class_1263 inventory) {

        final class_2371<class_1799> items = class_2371.method_10213(inventory.method_5439(), class_1799.field_8037);
        applyForEach(inventory, items::set);
        return items;
    }

    /**
     * Fills an inventory with a list of ItemStack. The inventory size and container size must be identical.
     *
     * @param inventory The inventory to fill.
     * @param fillWith  The items to fill the inventory with.
     */
    default void fill(class_1263 inventory, class_2371<class_1799> fillWith) {

        if (inventory.method_5439() != fillWith.size()) {

            throw new IllegalStateException("Inventory size did not match! inv_size=" + inventory.method_5439() + " fill_size=" + fillWith.size());
        }

        for (int slotId = 0; slotId < fillWith.size(); slotId++) {

            inventory.method_5447(slotId, fillWith.get(slotId));
        }
    }

    /**
     * Applies an action to every item within the inventory.
     *
     * @param inventory The inventory container.
     * @param action    The action to apply.
     */
    default void applyForEach(class_1263 inventory, Consumer<class_1799> action) {

        for (int slotId = 0; slotId < inventory.method_5439(); slotId++) {

            action.accept(inventory.method_5438(slotId));
        }
    }

    /**
     * Applies an action to every item within the inventory.
     *
     * @param inventory The inventory container.
     * @param action    The action to apply.
     */
    default void applyForEach(class_1263 inventory, BiConsumer<Integer, class_1799> action) {

        for (int slotId = 0; slotId < inventory.method_5439(); slotId++) {

            action.accept(slotId, inventory.method_5438(slotId));
        }
    }

    /**
     * Checks if two ItemStack may be stacked together.
     *
     * @param original The original stack.
     * @param toStack  The stack attempting to be stacked into the original.
     * @return Can the two stacks be stacked together.
     */
    default boolean canItemsStack(class_1799 original, class_1799 toStack) {

        return !toStack.method_7960() && class_1799.method_7984(original, toStack) && original.method_7985() == toStack.method_7985() && (!original.method_7985() || original.method_7969().equals(toStack.method_7969()));
    }

    /**
     * Attempts to provide access to an inventory stored at the given position.
     *
     * @param level     The world to query.
     * @param pos       The position to look at.
     * @param direction The side of the inventory being accessed.
     * @return Access to the inventory at the given position.
     */
    @Nullable
    default IInventoryAccess getInventory(class_1937 level, class_2338 pos, @Nullable class_2350 direction) {

        final class_2680 state = level.method_8320(pos);

        // Handle blocks like the composter that think they're inventories.
        if (state.method_26204() instanceof class_3954 holder) {

            final class_1278 container = holder.method_17680(state, level, pos);

            if (container != null) {

                return new ContainerInventoryAccess<>(container);
            }
        }

        // Handle vanilla tile entity containers
        else {

            final class_2586 be = level.method_8321(pos);

            if (be instanceof class_1263 container) {

                if (container instanceof class_1278 worldly) {

                    return new WorldlyContainerInventoryAccess<>(worldly);
                }

                return new ContainerInventoryAccess<>(container);
            }
        }

        return null;
    }

    default void openMenu(class_3222 player, class_3908 provider) {

        this.openMenu(player, provider, buf -> {});
    }

    default void openMenu(class_3222 player, class_3908 provider, Consumer<class_2540> buf) {

        this.openMenu(player, provider, buf, false);
    }

    void openMenu(class_3222 player, class_3908 provider, Consumer<class_2540> buf, boolean allowFakes);

    default class_1799 getCraftingRemainder(class_1799 stack) {

        return this.hasCraftingRemainder(stack) ? new class_1799(stack.method_7909().method_7858()) : class_1799.field_8037;
    }

    default boolean hasCraftingRemainder(class_1799 stack) {

        return stack.method_7909().method_7857();
    }

    boolean isFakePlayer(class_1657 player);
}