package net.darkhax.bookshelf.api.util;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import net.minecraft.class_1291;
import net.minecraft.class_1293;
import net.minecraft.class_1294;
import net.minecraft.class_1297;
import net.minecraft.class_1299;
import net.minecraft.class_1304;
import net.minecraft.class_1308;
import net.minecraft.class_1309;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_1826;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_238;
import net.minecraft.class_239;
import net.minecraft.class_243;
import net.minecraft.class_3545;
import net.minecraft.class_3959;
import net.minecraft.class_3959.class_242;
import net.minecraft.class_3959.class_3960;

public final class EntityHelper {

    /**
     * A cache of spawn egg colors mapped to the entity type. Populated by {@link #getEggColors(class_1299)}.
     */
    private static final Map<class_1299<?>, class_3545<Integer, Integer>> eggColorCache = new HashMap<>();

    /**
     * Calculates the distance between two entities.
     *
     * @param firstEntity  The first entity to use.
     * @param secondEntity The second entity to use.
     * @return double The distance between the two entities passed.
     */
    public static double getDistanceFromEntity(class_1297 firstEntity, class_1297 secondEntity) {

        return MathsHelper.getDistanceBetweenPoints(firstEntity.method_19538(), secondEntity.method_19538());
    }

    /**
     * Calculates the distance between an entity and a BlockPos.
     *
     * @param entity The Entity to use for the first position.
     * @param pos    The BlockPos to use for the second position.
     * @return double The distance between the Entity and the BlockPos.
     */
    public static double getDistaceFromPos(class_1297 entity, class_2338 pos) {

        return MathsHelper.getDistanceBetweenPoints(entity.method_19538(), class_243.method_24953(pos));
    }

    /**
     * Pushes an entity towards a specific direction.
     *
     * @param entityToMove The entity that you want to push.
     * @param direction    The direction to push the entity.
     * @param force        The amount of force to push the entity with.
     */
    public static void pushTowards(class_1297 entityToMove, class_2350 direction, double force) {

        pushTowards(entityToMove, entityToMove.method_24515().method_10079(direction.method_10153(), 1), force);
    }

    /**
     * Pushes an Entity towards a BlockPos.
     *
     * @param entityToMove The entity that you want to push.
     * @param pos          The BlockPos to push the entity towards.
     * @param force        The amount of force to push the entity with.
     */
    public static void pushTowards(class_1297 entityToMove, class_2338 pos, double force) {

        final class_2338 entityPos = entityToMove.method_24515();
        final double distanceX = (double) pos.method_10263() - entityPos.method_10263();
        final double distanceY = (double) pos.method_10264() - entityPos.method_10264();
        final double distanceZ = (double) pos.method_10260() - entityPos.method_10260();
        final double distance = Math.sqrt(distanceX * distanceX + distanceY * distanceY + distanceZ * distanceZ);

        if (distance > 0) {
            entityToMove.method_18799(new class_243(distanceX / distance * force, distanceY / distance * force, distanceZ / distance * force));
        }
    }

    /**
     * Pushes an entity towards another one.
     *
     * @param entityToMove The entity that should be pushed towards the other entity.
     * @param destination  The destination entity, that the entity to move should be pushed towards.
     * @param force        The amount of force to push the entityToMove with.
     */
    public static void pushTowards(class_1297 entityToMove, class_1297 destination, double force) {

        final double distanceX = destination.method_23317() - entityToMove.method_23317();
        final double distanceY = destination.method_23318() - entityToMove.method_23318();
        final double distanceZ = destination.method_23321() - entityToMove.method_23321();
        final double distance = Math.sqrt(distanceX * distanceX + distanceY * distanceY + distanceZ * distanceZ);

        if (distance > 0) {
            entityToMove.method_18799(new class_243(distanceX / distance * force, distanceY / distance * force, distanceZ / distance * force));
        }
    }

    /**
     * Creates a Vector3d that represents the additional motion that would be needed to push an entity towards a
     * destination.
     *
     * @param entityToMove The entity to push.
     * @param direction    The direction to push the entity.
     * @param force        The amount of force to use.
     */
    public static void pushTowardsDirection(class_1297 entityToMove, class_2350 direction, double force) {

        final class_2338 entityPos = entityToMove.method_24515();
        final class_2338 destination = entityToMove.method_24515().method_10079(direction.method_10153(), 1);

        final double distanceX = (double) destination.method_10263() - entityPos.method_10263();
        final double distanceY = (double) destination.method_10264() - entityPos.method_10264();
        final double distanceZ = (double) destination.method_10260() - entityPos.method_10260();
        final double distance = Math.sqrt(distanceX * distanceX + distanceY * distanceY + distanceZ * distanceZ);

        if (distance > 0) {
            entityToMove.method_18799(new class_243(distanceX / distance * force, distanceY / distance * force, distanceZ / distance * force));
        }
    }

    /**
     * Checks if two entities are close enough together.
     *
     * @param firstEntity  The first entity to check.
     * @param secondEntity The second entity to check.
     * @param maxDistance  The maximum distance that the entities can be apart.
     * @return boolean True if the distance between the entities are within range of the maxDistance.
     */
    public static boolean areEntitiesCloseEnough(class_1297 firstEntity, class_1297 secondEntity, double maxDistance) {

        return getDistanceFromEntity(firstEntity, secondEntity) < maxDistance * maxDistance;
    }

    /**
     * Gets a List of entities that are within the provided area.
     *
     * @param <T>         The type of entities to look for.
     * @param entityClass The type of entity you are looking for.
     * @param world       The world to search in.
     * @param pos         The position to start the search around.
     * @param range       The range of the search.
     * @return A List containing all entities of the specified type that are within the range.
     */
    public static <T extends class_1297> List<T> getEntitiesInArea(Class<T> entityClass, class_1937 world, class_2338 pos, int range) {

        return world.method_18467(entityClass, new class_238(pos.method_10069(-range, -range, -range), pos.method_10069(range + 1, range + 1, range + 1)));
    }

    /**
     * A check to see if an entity is wearing a full suit of the armor. This check is based on the class names of
     * armor.
     *
     * @param living:     The living entity to check the armor of.
     * @param armorClass: The class of the armor to check against.
     * @return boolean: True if every piece of armor the entity is wearing are the same class as the provied armor
     * class.
     */
    public static boolean isWearingFullSet(class_1308 living, Class<class_1792> armorClass) {

        for (final class_1304 slot : class_1304.values()) {
            if (slot.method_5925().equals(class_1304.class_1305.field_6178)) {

                final class_1799 armor = living.method_6118(slot);

                if (armor.method_7960() || !armor.method_7909().getClass().equals(armorClass)) {
                    return false;
                }
            }
        }

        return true;
    }

    /**
     * Performs a ray trace for the look vector of an entity.
     *
     * @param entity    The entity to perform a ray trace on.
     * @param length    The distance to cast the rays.
     * @param blockMode The mode used when detecting blocks.
     * @param fluidMode The mode used when detecting fluids.
     * @return An object containing the results of the ray trace.
     */
    public static class_239 rayTrace(class_1309 entity, double length, class_3960 blockMode, class_242 fluidMode) {

        final class_243 startingPosition = new class_243(entity.method_23317(), entity.method_23318() + entity.method_5751(), entity.method_23321());
        final class_243 lookVector = entity.method_5720();
        final class_243 endingPosition = startingPosition.method_1031(lookVector.field_1352 * length, lookVector.field_1351 * length, lookVector.field_1350 * length);
        return entity.method_37908().method_17742(new class_3959(startingPosition, endingPosition, blockMode, fluidMode, entity));
    }

    /**
     * Checks if an entity can be affected by fire. While fire immune entities can already negate the effects of fire
     * doing prechecks using this method can be used to avoid rendering flickers or filter out these types of entities.
     *
     * @param toCheck The entity to check.
     * @return Whether or not this entity can be affected by fire.
     */
    public static boolean isAffectedByFire(class_1309 toCheck) {

        return !toCheck.method_5753() && !toCheck.method_6059(class_1294.field_5918);
    }

    /**
     * Clears potion effect from an entity based on whether or not the effects are positive or negative.
     *
     * @param entity         The entity to remove effects from.
     * @param removePositive Should positive effects be cleared?
     * @param removeNegative Should negative effects be cleared?
     */
    public static void clearEffects(class_1309 entity, boolean removePositive, boolean removeNegative) {

        final Set<class_1291> toClear = new HashSet<>();

        for (final class_1293 effect : entity.method_6026()) {

            final boolean isGood = effect.method_5579().method_5573();

            if (isGood && removePositive || !isGood && removeNegative) {

                toClear.add(effect.method_5579());
            }
        }

        for (final class_1291 effect : toClear) {

            entity.method_6016(effect);
        }
    }

    /**
     * Get the egg color associated with an entity type. If the entity does not have an egg type this will be 0 for both
     * values.
     *
     * @param type The entity type to get a color for.
     * @return A Tuple containing the primary and secondary egg colors.
     */
    public static class_3545<Integer, Integer> getEggColors(class_1299<?> type) {

        return eggColorCache.computeIfAbsent(type, key -> {

            class_1826 item = class_1826.method_8019(key);

            if (item != null) {

                return new class_3545<>(item.method_8016(0), item.method_8016(1));
            }

            return new class_3545<>(0, 0);
        });
    }
}