package net.darkhax.bookshelf.api.function;

import net.darkhax.bookshelf.mixin.accessors.world.AccessorRecipeManager;
import net.minecraft.class_1860;
import net.minecraft.class_1863;
import net.minecraft.class_1937;
import net.minecraft.class_2378;
import net.minecraft.class_2960;
import net.minecraft.class_3956;
import net.minecraft.class_5321;
import javax.annotation.Nullable;
import java.lang.ref.WeakReference;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Supplier;

/**
 * A cached value that is lazily loaded and will be invalidated automatically after the game has been reloaded.
 *
 * @param <T> The type of the cached value.
 */
public class ReloadableCache<T> implements Function<class_1937, T> {

    /**
     * An internal function that is responsible for producing the value to cache.
     */
    private final Function<class_1937, T> delegate;

    /**
     * A weak reference to the recipe manager that was active when the value was last cached. This is an internal
     * implementation detail that is used to detect when the game has been reloaded.
     */
    private WeakReference<class_1863> recipeManager = new WeakReference<>(null);

    /**
     * A flag that tracks if a value has been cached.
     */
    private boolean cached = false;

    /**
     * The value that is currently cached.
     */
    @Nullable
    private T cachedValue;

    protected ReloadableCache(Function<class_1937, T> delegate) {

        this.delegate = delegate;
    }

    @Nullable
    @Override
    public T apply(class_1937 level) {

        if (!this.isCached() || hasGameReloaded(level)) {

            this.recipeManager = new WeakReference<>(level.method_8433());
            this.cachedValue = this.delegate.apply(level);
            this.cached = true;
        }

        return this.cachedValue;
    }

    /**
     * Manually invalidates the cache. This will result in a new value being cached the next time {@link #apply(class_1937)}
     * is invoked.
     */
    public void invalidate() {

        this.cached = false;
        this.cachedValue = null;
        this.recipeManager = new WeakReference<>(null);
    }

    /**
     * Checks if the cache has cached a value. This is not a substitute for null checking.
     *
     * @return Has the supplier cached a value.
     */
    public boolean isCached() {

        return this.cached;
    }

    /**
     * Checks if the game has reloaded since the last time the cache was updated.
     *
     * @param level The current game level. This is used to provide context about the current state of the game.
     * @return If the game has reloaded since the last time the cache was updated.
     */
    public boolean hasGameReloaded(class_1937 level) {

        return this.recipeManager.get() != level.method_8433();
    }

    /**
     * Invokes the consumer with the cached value. This will cause a value to be cached if one has not been cached
     * alread.
     *
     * @param level    The current game level. This is used to provide context about the current state of the game.
     * @param consumer The consumer to invoke.
     */
    public void apply(class_1937 level, Consumer<T> consumer) {

        consumer.accept(this.apply(level));
    }

    /**
     * Creates a cache of a value that will be recalculated when the game reloads.
     *
     * @param supplier The supplier used to produce the cached value.
     * @param <T>      The type of value held by the cache.
     * @return A cache of a value that will be recalculated when the game reloads.
     */
    public static <T> ReloadableCache<T> of(Supplier<T> supplier) {

        return new ReloadableCache<>(level -> supplier.get());
    }

    /**
     * Creates a cache of a value that will be recalculated when the game reloads.
     *
     * @param delegate A function used to produce the value to cache.
     * @param <T>      The type of value held by the cache.
     * @return A cache of a value that will be recalculated when the game reloads.
     */
    public static <T> ReloadableCache<T> of(Function<class_1937, T> delegate) {

        return new ReloadableCache<>(delegate);
    }

    /**
     * Creates a cache of a registry value that will be reaquired when the game reloads.
     *
     * @param registry The registry to lookup the value in.
     * @param id       The ID of the value to lookup.
     * @param <T>      The type of value held by the cache.
     * @return A cache of a registry value that will be reaquired when the game reloads.
     */
    public static <T> ReloadableCache<T> of(class_5321<? extends class_2378<T>> registry, class_2960 id) {

        return ReloadableCache.of(level -> level.method_30349().method_30530(registry).method_10223(id));
    }

    /**
     * Creates a cache of a recipe value that will be reaquired when the game reloads.
     *
     * @param type The type of recipe to lookup.
     * @param id   The ID of the value to lookup.
     * @param <T>  The type of value held by the cache.
     * @return A cache of a registry value that will be reaquired when the game reloads.
     */
    public static <T> ReloadableCache<T> of(class_3956<? extends class_1860<?>> type, class_2960 id) {
        
        return ReloadableCache.of(level -> {

            if (level.method_8433() instanceof AccessorRecipeManager accessor) {

                return (T) accessor.bookshelf$getTypeMap(type).get(id);
            }

            return null;
        });
    }
}