package net.darkhax.bookshelf.api.block.entity;

import net.darkhax.bookshelf.api.Services;
import net.minecraft.class_1262;
import net.minecraft.class_1263;
import net.minecraft.class_1264;
import net.minecraft.class_1657;
import net.minecraft.class_1661;
import net.minecraft.class_1703;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_2371;
import net.minecraft.class_2487;
import net.minecraft.class_2520;
import net.minecraft.class_2591;
import net.minecraft.class_2624;
import net.minecraft.class_2680;
import java.util.Set;

/**
 * An implementation of BlockEntity that holds an inventory. The inventory is persisted and synced to the client.
 *
 * @param <T> The type of inventory held by the BlockEntity.
 */
public abstract class InventoryBlockEntity<T extends class_1263> extends class_2624 {

    /**
     * The inventory held by the block entity.
     */
    private final T inventory = this.createInventory();

    public InventoryBlockEntity(class_2591<?> type, class_2338 pos, class_2680 state) {

        super(type, pos, state);
    }

    /**
     * Gets the inventory currently held by the block entity.
     *
     * @return The inventory currently held by the block entity.
     */
    public final T getInventory() {

        return this.inventory;
    }

    /**
     * Creates a new instance of the inventory held by the block entity. The resulting inventory should be effectively a
     * clean slate that represents the default state for the inventory. Persisting the state of the inventory is managed
     * by {@link #method_11014(net.minecraft.class_2487)} and {@link #method_11007(net.minecraft.class_2487)}.
     * <p>
     * This method should only be invoked once per tile entity instance. The resulting value is stored with
     * {@link #inventory}.
     *
     * @return A new inventory to be held by the block entity.
     */
    public abstract T createInventory();

    /**
     * Drops the contents of the held inventory into the world. This is used in situations where the block has been
     * removed from the world and allows the inventory to drop the contents before they are destroyed.
     * <p>
     * The default behaviour will drop the entire contents of the inventory onto the ground.
     *
     * @param state The state of the block.
     * @param world The level the block is in.
     * @param pos   The position of the block.
     */
    public void dropContents(class_2680 state, class_1937 world, class_2338 pos) {

        class_1264.method_5451(world, pos, this.getInventory());
    }

    /**
     * Update the state of the held inventory using state data read from NBT.
     * <p>
     * Implementations of this method must be capable of reading from the result of {@link #saveInventory()}.
     * <p>
     * The inventory instance should always be accessed using {@link #getInventory()}. While each instance of the block
     * entity will have their own unique instance of the inventory, that instance will be reused across multiple read
     * and write cycles. You must ensure that the entire state of the inventory is properly reinitialized on read.
     *
     * @param tag A tag containing the inventory state data.
     */
    public void readInventory(class_2487 tag) {

        final class_2371<class_1799> tempInvStacks = class_2371.method_10213(this.getInventory().method_5439(), class_1799.field_8037);
        class_1262.method_5429(tag, tempInvStacks);
        Services.INVENTORY_HELPER.fill(this.getInventory(), tempInvStacks);
    }

    /**
     * Stores the state of the currently held inventory in an NBT tag.
     * <p>
     * Implementations of this method must be capable of writing a tag that can be read by
     * {@link #readInventory(class_2487)}.
     * <p>
     * The inventory instance should always be accessed using {@link #getInventory()}.
     *
     * @return A tag holding the state of the current inventory.
     */
    public class_2487 saveInventory() {

        return class_1262.method_5426(new class_2487(), Services.INVENTORY_HELPER.toList(this.getInventory()));
    }

    /**
     * Notifies the world that the tile has been updated and requires re-saving. This has the same behaviour as
     * {@link net.minecraft.class_2586#method_5431()} but avoids invoking
     * {@link class_1263#method_5431()} unintentionally.
     */
    public void markDirty() {

        super.method_5431();
    }

    @Override
    public void method_11014(class_2487 tag) {

        super.method_11014(tag);

        // Reads the state of the inventory from the tile tag.
        if (tag.method_10573("Inventory", class_2520.field_33260)) {

            this.readInventory(tag.method_10562("Inventory"));
        }
    }

    @Override
    public void method_11007(class_2487 tag) {

        super.method_11007(tag);

        // Writes the state of the inventory to the tile tag.
        tag.method_10566("Inventory", this.saveInventory());
    }


    @Override
    protected class_1703 method_5465(int i, class_1661 inventory) {

        // Default implementation for MenuConstructor.
        return null;
    }

    @Override
    public final int method_5439() {

        // Delegate to the held inventory.
        return this.getInventory().method_5439();
    }

    @Override
    public final boolean method_5442() {

        // Delegate to the held inventory.
        return this.getInventory().method_5442();
    }

    @Override
    public final class_1799 method_5438(int slot) {

        // Delegate tot he held inventory.
        return this.getInventory().method_5438(slot);
    }

    @Override
    public final class_1799 method_5434(int slot, int amount) {

        // Delegate to the held inventory.
        return this.getInventory().method_5434(slot, amount);
    }

    @Override
    public final class_1799 method_5441(int slot) {

        // Delegate to the held inventory.
        return this.getInventory().method_5441(slot);
    }

    @Override
    public final void method_5447(int slot, class_1799 stack) {

        // Delegate to the held inventory.
        this.getInventory().method_5447(slot, stack);
    }

    @Override
    public final int method_5444() {

        // Delegate to the held inventory.
        return this.getInventory().method_5444();
    }

    @Override
    public final void method_5431() {

        // BlockEntity has a method with the same name. It seems vanilla intends for these to be unified for block entities with containers.
        super.method_5431();

        // Delegate to the held inventory.
        this.getInventory().method_5431();
    }

    @Override
    public final boolean method_5443(class_1657 player) {

        // Delegate to the held inventory.
        return this.getInventory().method_5443(player);
    }

    @Override
    public final void method_5435(class_1657 player) {

        // Delegate to the held inventory.
        this.getInventory().method_5435(player);
    }

    @Override
    public final void method_5432(class_1657 player) {

        // Delegate to the held inventory.
        this.getInventory().method_5432(player);
    }

    @Override
    public final boolean method_5437(int slot, class_1799 stack) {

        // Delegate to the held inventory.
        return this.getInventory().method_5437(slot, stack);
    }

    @Override
    public final int method_18861(class_1792 toCount) {

        // Delegate to the held inventory.
        return this.getInventory().method_18861(toCount);
    }

    @Override
    public final boolean method_18862(Set<class_1792> toFind) {

        // Delegate to the held inventory.
        return this.getInventory().method_18862(toFind);
    }

    @Override
    public final void method_5448() {

        // Delegate to the held inventory.
        this.getInventory().method_5448();
    }
}