package commonnetwork.api;

import net.minecraft.class_1923;
import net.minecraft.class_2338;
import net.minecraft.class_2535;
import net.minecraft.class_2818;
import net.minecraft.class_3215;
import net.minecraft.class_3218;
import net.minecraft.class_3222;
import net.minecraft.server.MinecraftServer;
import java.util.List;

public interface NetworkHandler
{
    /**
     * Sends the packet to the server, if the server has the packet registered.
     *
     * @param packet - the packet
     * @param <T>    - The packet
     */
    default <T> void sendToServer(T packet)
    {
        sendToServer(packet, false);
    }

    /**
     * Sends the packet to the server. Can ignore the check if the server has the packet registered.
     * Likely use case for this is talking to bukkit/spigot/paper servers.
     *
     * @param packet      - the packet
     * @param ignoreCheck - ignore the check if the server has the packet registered.
     * @param <T>         - The packet
     */
    <T> void sendToServer(T packet, boolean ignoreCheck);

    /**
     * Sends the packet to the client player, only if the player has the packet registered.
     *
     * @param packet - the packet
     * @param player - the player
     * @param <T>    - The packet
     */
    default <T> void sendToClient(T packet, class_3222 player)
    {
        sendToClient(packet, player, false);
    }

    /**
     * Sends the packet to the client player..
     *
     * @param packet      - the packet
     * @param player      - the player
     * @param ignoreCheck - ignore the check if the client has the packet registered.
     * @param <T>         - The packet
     */
    <T> void sendToClient(T packet, class_3222 player, boolean ignoreCheck);

    /**
     * Sends the packet to the connection.
     *
     * @param packet     - the packet
     * @param connection - the connection
     * @param <T>        - The packet
     */
    <T> void send(T packet, class_2535 connection);

    /**
     * Sends the packet to the client players, only if the players has the packet registered.
     *
     * @param packet  - the packet
     * @param players - the players
     * @param <T>     - The packet
     */
    default <T> void sendToClients(T packet, List<class_3222> players)
    {
        sendToClients(packet, players, false);
    }

    /**
     * Sends the packet to the client players.
     *
     * @param packet      - the packet
     * @param players     - the players
     * @param ignoreCheck - ignore the check if the client has the packet registered.
     * @param <T>         - The packet
     */
    default <T> void sendToClients(T packet, List<class_3222> players, boolean ignoreCheck)
    {
        for (class_3222 player : players)
        {
            sendToClient(packet, player, ignoreCheck);
        }
    }

    /**
     * Sends the packet to all the client players in the server, only if the players has the packet registered.
     *
     * @param packet - the packet
     * @param server - the server
     * @param <T>    - The packet
     */
    default <T> void sendToAllClients(T packet, MinecraftServer server)
    {
        sendToAllClients(packet, server, false);
    }

    /**
     * Sends the packet to all the client players in the server
     *
     * @param packet      - the packet
     * @param server      - the server
     * @param ignoreCheck - ignore the check if the client has the packet registered.
     * @param <T>         - The packet
     */
    default <T> void sendToAllClients(T packet, MinecraftServer server, boolean ignoreCheck)
    {
        sendToClients(packet, server.method_70562().method_68990().method_14571(), ignoreCheck);
    }

    /**
     * Sends the packet to all the client players in the level, only if the players has the packet registered.
     *
     * @param packet - the packet
     * @param level  - the level
     * @param <T>    - The packet
     */
    default <T> void sendToClientsInLevel(T packet, class_3218 level)
    {
        sendToClientsInLevel(packet, level, false);
    }

    /**
     * Sends the packet to all the client players in the level.
     *
     * @param packet      - the packet
     * @param level       - the level
     * @param ignoreCheck - ignore the check if the client has the packet registered.
     * @param <T>         - The packet
     */
    default <T> void sendToClientsInLevel(T packet, class_3218 level, boolean ignoreCheck)

    {
        sendToClients(packet, level.method_18456(), ignoreCheck);
    }

    /**
     * Sends the packet to all the client players loading a chunk, only if the players has the packet registered.
     *
     * @param packet - the packet
     * @param chunk  - the chunk
     * @param <T>    - The packet
     */
    default <T> void sendToClientsLoadingChunk(T packet, class_2818 chunk)
    {
        sendToClientsLoadingChunk(packet, chunk, false);
    }

    /**
     * Sends the packet to all the client players loading a chunk.
     *
     * @param packet      - the packet
     * @param chunk       - the chunk
     * @param ignoreCheck - ignore the check if the client has the packet registered.
     * @param <T>         - The packet
     */
    default <T> void sendToClientsLoadingChunk(T packet, class_2818 chunk, boolean ignoreCheck)
    {
        class_3215 chunkCache = (class_3215) chunk.method_12200().method_8398();
        sendToClients(packet, chunkCache.field_17254.method_17210(chunk.method_12004(), false), ignoreCheck);
    }

    /**
     * Sends the packet to all the client players loading a position, only if the players has the packet registered.
     *
     * @param packet - the packet
     * @param level  - the level
     * @param pos    - the chunkpos
     * @param <T>    - The packet
     */
    default <T> void sendToClientsLoadingPos(T packet, class_3218 level, class_1923 pos)
    {
        sendToClientsLoadingPos(packet, level, pos, false);
    }

    /**
     * Sends the packet to all the client players loading a position.
     *
     * @param packet      - the packet
     * @param level       - the level
     * @param pos         - the chunkpos
     * @param ignoreCheck - ignore the check if the client has the packet registered.
     * @param <T>         - The packet
     */
    default <T> void sendToClientsLoadingPos(T packet, class_3218 level, class_1923 pos, boolean ignoreCheck)
    {
        sendToClientsLoadingChunk(packet, level.method_8497(pos.field_9181, pos.field_9180), ignoreCheck);
    }

    /**
     * Sends the packet to all the client players loading a position, only if the players has the packet registered.
     *
     * @param packet - the packet
     * @param level  - the level
     * @param pos    - the blockpos
     * @param <T>    - The packet
     */
    default <T> void sendToClientsLoadingPos(T packet, class_3218 level, class_2338 pos)
    {
        sendToClientsLoadingPos(packet, level, pos, false);
    }

    /**
     * Sends the packet to all the client players loading a position
     *
     * @param packet      - the packet
     * @param level       - the level
     * @param pos         - the blockpos
     * @param ignoreCheck - ignore the check if the client has the packet registered.
     * @param <T>         - The packet
     */
    default <T> void sendToClientsLoadingPos(T packet, class_3218 level, class_2338 pos, boolean ignoreCheck)
    {
        sendToClientsLoadingPos(packet, level, new class_1923(pos), ignoreCheck);
    }

    /**
     * Sends the packet to all the client players in range of a position, only if the players has the packet registered.
     *
     * @param packet - the packet
     * @param level  - the level
     * @param pos    - the blockpos
     * @param range  - the range
     * @param <T>    - The packet
     */
    default <T> void sendToClientsInRange(T packet, class_3218 level, class_2338 pos, double range)
    {
        sendToClientsInRange(packet, level, pos, range, false);
    }

    /**
     * Sends the packet to all the client players in range of a position.
     *
     * @param packet      - the packet
     * @param level       - the level
     * @param pos         - the blockpos
     * @param range       - the range
     * @param ignoreCheck - ignore the check if the client has the packet registered.
     * @param <T>         - The packet
     */
    default <T> void sendToClientsInRange(T packet, class_3218 level, class_2338 pos, double range, boolean ignoreCheck)
    {
        for (class_3222 player : level.method_18456())
        {
            if (player.method_5649(pos.method_10263(), pos.method_10264(), pos.method_10260()) <= range * range)
            {
                sendToClient(packet, player, ignoreCheck);
            }
        }
    }
}
