package mezz.jei.library.render;

import com.google.common.base.Preconditions;
import mezz.jei.api.ingredients.IIngredientRenderer;
import mezz.jei.api.ingredients.IIngredientTypeWithSubtypes;
import mezz.jei.common.platform.IPlatformFluidHelperInternal;
import mezz.jei.common.platform.IPlatformRenderHelper;
import mezz.jei.common.platform.Services;
import net.minecraft.class_1058;
import net.minecraft.class_124;
import net.minecraft.class_1836;
import net.minecraft.class_1921;
import net.minecraft.class_2561;
import net.minecraft.class_332;
import net.minecraft.class_3611;
import net.minecraft.class_3612;
import net.minecraft.class_5250;
import net.minecraft.class_7764;
import net.minecraft.class_8690;
import java.text.NumberFormat;
import java.util.ArrayList;
import java.util.List;

public class FluidTankRenderer<T> implements IIngredientRenderer<T> {
	private static final NumberFormat nf = NumberFormat.getIntegerInstance();
	private static final int MIN_FLUID_HEIGHT = 1; // ensure tiny amounts of fluid are still visible

	private final IPlatformFluidHelperInternal<T> fluidHelper;
	private final long capacity;
	private final TooltipMode tooltipMode;
	private final int width;
	private final int height;

	enum TooltipMode {
		SHOW_AMOUNT,
		SHOW_AMOUNT_AND_CAPACITY,
		ITEM_LIST
	}

	public FluidTankRenderer(IPlatformFluidHelperInternal<T> fluidHelper) {
		this(fluidHelper, fluidHelper.bucketVolume(), TooltipMode.ITEM_LIST, 16, 16);
	}

	public FluidTankRenderer(IPlatformFluidHelperInternal<T> fluidHelper, long capacity, boolean showCapacity, int width, int height) {
		this(fluidHelper, capacity, showCapacity ? TooltipMode.SHOW_AMOUNT_AND_CAPACITY : TooltipMode.SHOW_AMOUNT, width, height);
	}

	private FluidTankRenderer(IPlatformFluidHelperInternal<T> fluidHelper, long capacity, TooltipMode tooltipMode, int width, int height) {
		Preconditions.checkArgument(capacity > 0, "capacity must be > 0");
		Preconditions.checkArgument(width > 0, "width must be > 0");
		Preconditions.checkArgument(height > 0, "height must be > 0");
		this.fluidHelper = fluidHelper;
		this.capacity = capacity;
		this.tooltipMode = tooltipMode;
		this.width = width;
		this.height = height;
	}

	@Override
	public void render(class_332 guiGraphics, T fluidStack) {
		render(guiGraphics, fluidStack, 0, 0);
	}

	@Override
	public void render(class_332 guiGraphics, T ingredient, int posX, int posY) {
		IIngredientTypeWithSubtypes<class_3611, T> type = fluidHelper.getFluidIngredientType();
		class_3611 fluid = type.getBase(ingredient);
		if (fluid.method_15780(class_3612.field_15906)) {
			return;
		}

		fluidHelper.getStillFluidSprite(ingredient)
			.ifPresent(fluidStillSprite -> {
				int fluidColor = fluidHelper.getColorTint(ingredient);

				long amount = fluidHelper.getAmount(ingredient);
				if (amount > 0) {
					long longScaledAmount = (amount * height) / capacity;
					int scaledAmount = Math.clamp(longScaledAmount, MIN_FLUID_HEIGHT, height);
					drawTiledSprite(guiGraphics, width, height, fluidColor, scaledAmount, fluidStillSprite, posX, posY);
				}
			});
	}

	private static void drawTiledSprite(class_332 guiGraphics, final int tiledWidth, final int tiledHeight, int color, int scaledAmount, class_1058 sprite, int posX, int posY) {
		IPlatformRenderHelper renderHelper = Services.PLATFORM.getRenderHelper();
		class_7764 spriteContents = sprite.method_45851();
		class_8690.class_8694 tileScaling = new class_8690.class_8694(spriteContents.method_45807(), spriteContents.method_45815());

		posY = posY + tiledHeight - scaledAmount;

		guiGraphics.method_44379(posX, posY, posX + tiledWidth, posY + scaledAmount);
		{
			renderHelper.blitTiledSprite(
				guiGraphics,
				class_1921::method_62277,
				sprite,
				tileScaling,
				posX,
				posY,
				tiledWidth,
				scaledAmount,
				color
			);
		}
		guiGraphics.method_44380();
	}

	@Override
	public List<class_2561> getTooltip(T fluidStack, class_1836 tooltipFlag) {
		List<class_2561> tooltip = new ArrayList<>();

		IIngredientTypeWithSubtypes<class_3611, T> type = fluidHelper.getFluidIngredientType();
		class_3611 fluidType = type.getBase(fluidStack);
		if (fluidType.method_15780(class_3612.field_15906)) {
			return tooltip;
		}

		fluidHelper.getTooltip(tooltip, fluidStack, tooltipFlag);

		long amount = fluidHelper.getAmount(fluidStack);
		long milliBuckets = (amount * 1000) / fluidHelper.bucketVolume();

		if (tooltipMode == TooltipMode.SHOW_AMOUNT_AND_CAPACITY) {
			class_5250 amountString = class_2561.method_43469("jei.tooltip.liquid.amount.with.capacity", nf.format(milliBuckets), nf.format(capacity));
			tooltip.add(amountString.method_27692(class_124.field_1080));
		} else if (tooltipMode == TooltipMode.SHOW_AMOUNT) {
			class_5250 amountString = class_2561.method_43469("jei.tooltip.liquid.amount", nf.format(milliBuckets));
			tooltip.add(amountString.method_27692(class_124.field_1080));
		}

		return tooltip;
	}

	@Override
	public int getWidth() {
		return width;
	}

	@Override
	public int getHeight() {
		return height;
	}
}
