package mezz.jei.library.render;

import com.google.common.base.Preconditions;
import mezz.jei.api.ingredients.IIngredientRenderer;
import mezz.jei.api.ingredients.IIngredientTypeWithSubtypes;
import mezz.jei.common.platform.IPlatformFluidHelperInternal;
import net.minecraft.class_1058;
import net.minecraft.class_1059;
import net.minecraft.class_124;
import net.minecraft.class_1836;
import net.minecraft.class_1921;
import net.minecraft.class_2561;
import net.minecraft.class_310;
import net.minecraft.class_332;
import net.minecraft.class_3611;
import net.minecraft.class_3612;
import net.minecraft.class_4588;
import net.minecraft.class_5250;
import org.joml.Matrix4f;

import java.text.NumberFormat;
import java.util.ArrayList;
import java.util.List;

public class FluidTankRenderer<T> implements IIngredientRenderer<T> {
	private static final NumberFormat nf = NumberFormat.getIntegerInstance();
	private static final int TEXTURE_SIZE = 16;
	private static final int MIN_FLUID_HEIGHT = 1; // ensure tiny amounts of fluid are still visible

	private final IPlatformFluidHelperInternal<T> fluidHelper;
	private final long capacity;
	private final TooltipMode tooltipMode;
	private final int width;
	private final int height;

	enum TooltipMode {
		SHOW_AMOUNT,
		SHOW_AMOUNT_AND_CAPACITY,
		ITEM_LIST
	}

	public FluidTankRenderer(IPlatformFluidHelperInternal<T> fluidHelper) {
		this(fluidHelper, fluidHelper.bucketVolume(), TooltipMode.ITEM_LIST, 16, 16);
	}

	public FluidTankRenderer(IPlatformFluidHelperInternal<T> fluidHelper, long capacity, boolean showCapacity, int width, int height) {
		this(fluidHelper, capacity, showCapacity ? TooltipMode.SHOW_AMOUNT_AND_CAPACITY : TooltipMode.SHOW_AMOUNT, width, height);
	}

	private FluidTankRenderer(IPlatformFluidHelperInternal<T> fluidHelper, long capacity, TooltipMode tooltipMode, int width, int height) {
		Preconditions.checkArgument(capacity > 0, "capacity must be > 0");
		Preconditions.checkArgument(width > 0, "width must be > 0");
		Preconditions.checkArgument(height > 0, "height must be > 0");
		this.fluidHelper = fluidHelper;
		this.capacity = capacity;
		this.tooltipMode = tooltipMode;
		this.width = width;
		this.height = height;
	}

	@Override
	public void render(class_332 guiGraphics, T fluidStack) {
		render(guiGraphics, fluidStack, 0, 0);
	}

	@Override
	public void render(class_332 guiGraphics, T ingredient, int posX, int posY) {
		IIngredientTypeWithSubtypes<class_3611, T> type = fluidHelper.getFluidIngredientType();
		class_3611 fluid = type.getBase(ingredient);
		if (fluid.method_15780(class_3612.field_15906)) {
			return;
		}

		fluidHelper.getStillFluidSprite(ingredient)
			.ifPresent(fluidStillSprite -> {
				int fluidColor = fluidHelper.getColorTint(ingredient);

				long amount = fluidHelper.getAmount(ingredient);
				long scaledAmount = (amount * height) / capacity;
				if (amount > 0 && scaledAmount < MIN_FLUID_HEIGHT) {
					scaledAmount = MIN_FLUID_HEIGHT;
				}
				if (scaledAmount > height) {
					scaledAmount = height;
				}

				drawTiledSprite(guiGraphics, width, height, fluidColor, scaledAmount, fluidStillSprite, posX, posY);
			});
	}

	private static void drawTiledSprite(class_332 guiGraphics, final int tiledWidth, final int tiledHeight, int color, long scaledAmount, class_1058 sprite, int posX, int posY) {
		@SuppressWarnings("deprecation")
		class_1921 rendertype = class_1921.method_62277(class_1059.field_5275);
		class_4588 bufferBuilder = class_310.method_1551().method_22940().method_23000().getBuffer(rendertype);

		Matrix4f matrix = guiGraphics.method_51448().method_23760().method_23761();

		final int xTileCount = tiledWidth / TEXTURE_SIZE;
		final int xRemainder = tiledWidth - (xTileCount * TEXTURE_SIZE);
		final long yTileCount = scaledAmount / TEXTURE_SIZE;
		final long yRemainder = scaledAmount - (yTileCount * TEXTURE_SIZE);

		final int yStart = tiledHeight + posY;

		for (int xTile = 0; xTile <= xTileCount; xTile++) {
			for (int yTile = 0; yTile <= yTileCount; yTile++) {
				int width = (xTile == xTileCount) ? xRemainder : TEXTURE_SIZE;
				long height = (yTile == yTileCount) ? yRemainder : TEXTURE_SIZE;
				int x = posX + (xTile * TEXTURE_SIZE);
				int y = yStart - ((yTile + 1) * TEXTURE_SIZE);
				if (width > 0 && height > 0) {
					long maskTop = TEXTURE_SIZE - height;
					int maskRight = TEXTURE_SIZE - width;

					drawTextureWithMasking(bufferBuilder, matrix, x, y, sprite, color, maskTop, maskRight, 100);
				}
			}
		}
	}

	private static void drawTextureWithMasking(class_4588 bufferBuilder, Matrix4f matrix, float xCoord, float yCoord, class_1058 textureSprite, int color, long maskTop, long maskRight, float zLevel) {
		float uMin = textureSprite.method_4594();
		float uMax = textureSprite.method_4577();
		float vMin = textureSprite.method_4593();
		float vMax = textureSprite.method_4575();
		uMax = uMax - (maskRight / 16F * (uMax - uMin));
		vMax = vMax - (maskTop / 16F * (vMax - vMin));

		bufferBuilder.method_22918(matrix, xCoord, yCoord + 16, zLevel)
			.method_39415(color)
			.method_22913(uMin, vMax);
		bufferBuilder.method_22918(matrix, xCoord + 16 - maskRight, yCoord + 16, zLevel)
			.method_39415(color)
			.method_22913(uMax, vMax);
		bufferBuilder.method_22918(matrix, xCoord + 16 - maskRight, yCoord + maskTop, zLevel)
			.method_39415(color)
			.method_22913(uMax, vMin);
		bufferBuilder.method_22918(matrix, xCoord, yCoord + maskTop, zLevel)
			.method_39415(color)
			.method_22913(uMin, vMin);
	}

	@Override
	public List<class_2561> getTooltip(T fluidStack, class_1836 tooltipFlag) {
		List<class_2561> tooltip = new ArrayList<>();

		IIngredientTypeWithSubtypes<class_3611, T> type = fluidHelper.getFluidIngredientType();
		class_3611 fluidType = type.getBase(fluidStack);
		if (fluidType.method_15780(class_3612.field_15906)) {
			return tooltip;
		}

		fluidHelper.getTooltip(tooltip, fluidStack, tooltipFlag);

		long amount = fluidHelper.getAmount(fluidStack);
		long milliBuckets = (amount * 1000) / fluidHelper.bucketVolume();

		if (tooltipMode == TooltipMode.SHOW_AMOUNT_AND_CAPACITY) {
			class_5250 amountString = class_2561.method_43469("jei.tooltip.liquid.amount.with.capacity", nf.format(milliBuckets), nf.format(capacity));
			tooltip.add(amountString.method_27692(class_124.field_1080));
		} else if (tooltipMode == TooltipMode.SHOW_AMOUNT) {
			class_5250 amountString = class_2561.method_43469("jei.tooltip.liquid.amount", nf.format(milliBuckets));
			tooltip.add(amountString.method_27692(class_124.field_1080));
		}

		return tooltip;
	}

	@Override
	public int getWidth() {
		return width;
	}

	@Override
	public int getHeight() {
		return height;
	}
}
